# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from textwrap import dedent

from django.conf import settings
from django.utils.translation import get_language
from django.utils.http import urlencode

from common.utils.locations import langify
from common.xgettext.i18n import gettext

# Шаблоны
from travel.rasp.morda.morda.templates.base import Base


class BaseTemplate(Base):
    page = 'informer-new'
    show_search_form = False
    show_awaps = False

    def get_code_button(self):
        return {
            'elem': 'line',
            'content': {
                'block': 'b-form-button',
                'js': True,
                'mods': {'theme': 'action-l', 'size': 'l', 'gen': 'code'},
                'content': gettext('Код для вставки')
            }
        }

    def menu_items(self):
        raw_menu_items = [
            ('', gettext('Все виджеты')),
            ('informer/', gettext('Информер')),
            ('search-string/', gettext('Поисковая строка'))
        ]

        if self.context.request.NATIONAL_VERSION == 'ru':
            raw_menu_items.append(('ticket/', gettext('Поиск авиабилетов')))

        menu_items = []

        for href, title in raw_menu_items:
            item = {
                'elem': 'item'
            }
            if self.context.current == href:
                item['elemMods'] = {'state': 'current'}
                item['content'] = title
            else:
                item['content'] = {
                    'block': 'b-link',
                    'url': langify('%s/informers/%s' % (self.context.host, href)),
                    'content': title
                }
            menu_items.append(item)

        return menu_items

    def content(self):
        return {
            'block': 'l-page',
            'mods': {'layout': '16-80'},
            'content': {
                'elem': 'row',
                'content': [
                    {'elem': 'gap'},
                    {
                        'elem': 'left',
                        'content': {
                            'elem': 'left-i',
                            'content': {
                                'block': 'b-big-menu',
                                'mods': {'theme': 'informer'},
                                'content': self.menu_items()
                            }
                        }
                    },
                    self.right_content(),
                    {'elem': 'gap'}
                ]
            }
        }

    def right_content(self):
        raise NotImplementedError


class InformersTemplate(BaseTemplate):

    def right_content(self):
        national_version = self.context.request.NATIONAL_VERSION
        lang_with_region = "%s-%s" % (get_language(), self.context.request.NATIONAL_VERSION.upper())

        return {
            'elem': 'right',
            'content': [
                {
                    'block': 'b-page-title',
                    'content': {
                        'elem': 'title',
                        'content': gettext('Виджеты')
                    }
                },
                {
                    'block': 'b-static-text',
                    'content': {
                        'tag': 'p',
                        'content': gettext('Виджеты можно настроить и разместить на страницах вашего сайта.'),
                    }
                },
                {
                    'block': 'b-informer-preview',
                    'content': [
                        {
                            'elem': 'item',
                            'content': [
                                {
                                    'block': 'b-link',
                                    'content': [
                                        {
                                            'block': 'b-icon',
                                            'url': self.static_url('blocks-desktop/b-informer-preview/b-informer-preview__informer_%s.png' % lang_with_region),
                                            'alt': gettext('Информер')
                                        },
                                        '<br />',
                                        gettext('Информер')
                                    ],
                                    'url': self.context.host + '/informers/informer/'
                                },
                                {
                                    'block': 'b-static-text',
                                    'content': {
                                        'tag': 'p',
                                        'content': gettext('Расписания рейсов для станций, аэропортов и по направлениям.'),
                                    }
                                }
                            ]
                        },
                        {
                            'elem': 'item',
                            'content': [
                                {
                                    'block': 'b-link',
                                    'content': [
                                        {
                                            'block': 'b-icon',
                                            'url': self.static_url('blocks-desktop/b-informer-preview/b-informer-preview__search_%s.png' % lang_with_region),
                                            'alt': gettext('Поисковая строка')
                                        },
                                        '<br />',
                                        gettext('Поисковая строка')
                                    ],
                                    'url': self.context.host + '/informers/search-string/'
                                },
                                {
                                    'block': 'b-static-text',
                                    'content': {
                                        'tag': 'p',
                                        'content': gettext('Поиск рейсов различными видами транспорта.')
                                    }
                                }
                            ]
                        },
                        {
                            'elem': 'item',
                            'content': [
                                {
                                    'block': 'b-link',
                                    'content': [
                                        {
                                            'block': 'b-icon',
                                            'url': self.static_url('blocks-desktop/b-informer-preview/b-informer-preview__ticket_%s.png' % lang_with_region),
                                            'alt': gettext('Поиск авиабилетов')
                                        },
                                        '<br />',
                                        gettext('Поиск авиабилетов'),
                                    ],
                                    'url': self.context.host + '/informers/ticket/'
                                },
                                {
                                    'block': 'b-static-text',
                                    'content': {
                                        'tag': 'p',
                                        'content': gettext('Поиск, выбор и покупка авиабилетов.'),
                                    }
                                }
                            ]
                        } if national_version in settings.TICKET_NATIONAL_VERSIONS else None
                    ]
                }
            ]
        }


class SearchStringTemplate(BaseTemplate):

    def right_content(self):
        return {
            'elem': 'right',
            'content': [
                {
                    'block': 'b-page-title',
                    'content': { 'elem': 'title', 'content': gettext('Поисковая строка') }
                },
                {
                    'block': 'b-tabbed-pane',
                    'mods': {'preset': 'big'},
                    'content': [
                        {
                            'elem': 'tabs',
                            'content': [
                                {
                                    'elem': 'tab',
                                    'elemMods': {'state': 'current'},
                                    'content': gettext('Вертикальная')
                                },
                                {
                                    'elem': 'tab',
                                    'content': gettext('Горизонтальная')
                                }
                            ]
                        },
                        {
                            'elem': 'panels',
                            'content': [
                                {
                                    'elem': 'panel',
                                    'elemMods': {'state': 'current'},
                                    'content': {
                                        'block': 'b-informer-new',
                                        'content': [
                                            {
                                                'elem': 'line',
                                                'content': {
                                                    'block': 'b-iframe',
                                                    'mods': {'type': 'search-vert'},
                                                    'content': '''<iframe frameborder="0" src="%s" id="ext-preview2" style="width: 200px; height: 220px; overflow: hidden; border: 0; *width: 201px;"></iframe>''' % self.context.url_vert
#                                                    'content': '''<iframe frameborder="0" src="%s" width="200px" height="220px" id="ext-preview2" style="
#                                                                    overflow: hidden; border: 0; -webkit-box-shadow: 0 2px 3px rgba(0, 0, 0, 0.5);
#                                                                    -moz-box-shadow: 0 2px 3px rgba(0, 0, 0, 0.5); box-shadow: 0 2px 3px rgba(0, 0, 0, 0.5);
#                                                                    *width: 203px;"></iframe>''' % self.context.url_vert
                                                }
                                            },
                                            self.get_code_button(),
                                            {
                                                'block': 'b-form-input',
                                                'mods': {'type': 'textarea', 'readonly': 'yes'},
                                                'attrs': {'rows': 10, 'cols': 10},
                                                'content': ''
                                            }
                                        ]
                                    }
                                },
                                {
                                    'elem': 'panel',
                                    'content': {
                                        'block': 'b-informer-new',
                                        'content': [
                                            {
                                                'elem': 'line',
                                                'content': {
                                                    'block': 'b-iframe',
                                                    'mods': {'type': 'search-horiz'},
                                                    'content': '''<iframe frameborder="0" src="%s" id="ext-preview3" style="width: 700px; height: 110px; overflow: hidden; border: 0;"></iframe>''' % self.context.url_horiz
#                                                    'content': '''<iframe frameborder="0" src="%s" id="ext-preview3" width="700px" height="110px"
#                                                                    style="overflow: hidden;
#                                                                    border: 0; -webkit-box-shadow: 0 2px 3px rgba(0, 0, 0, 0.5);
#                                                                    -moz-box-shadow: 0 2px 3px rgba(0, 0, 0, 0.5);
#                                                                    box-shadow: 0 2px 3px rgba(0, 0, 0, 0.5);"></iframe>''' % self.context.url_horiz
                                                }
                                            },
                                            self.get_code_button(),
                                            {
                                                'block': 'b-form-input',
                                                'mods': {'type': 'textarea', 'readonly': 'yes'},
                                                'attrs': {'rows': 10, 'cols': 10},
                                                'content': ''
                                            }
                                        ]
                                    }
                                }
                            ]
                        }
                    ]
                }
            ]
        }


class TicketTemplate(BaseTemplate):

    def right_content(self):
        return {
            'elem': 'right',
            'content': [
                {
                    'block': 'b-page-title',
                    'content': { 'elem': 'title', 'content': gettext('Поиск авиабилетов') }
                },
                {
                    'block': 'b-informer-new',
                    'content': [
                        {
                            'elem': 'line',
                            'content': {
                                'block': 'b-iframe',
                                'mods': {'type': 'search-ticket'},
                                'content': '''<iframe frameborder="0" src="%s" id="ext-preview4" style="width: 230px; height: 290px; overflow: hidden; border: 0;"></iframe>''' % self.context.widget_url
#                                'content': '''<iframe frameborder="0" src="%s" id="ext-preview4" width="238px" height="220px" style="overflow: hidden; border: 0; -webkit-box-shadow: 0 2px 3px rgba(0, 0, 0, 0.5); -moz-box-shadow: 0 2px 3px rgba(0, 0, 0, 0.5); box-shadow: 0 2px 3px rgba(0, 0, 0, 0.5);"></iframe>''' % self.context.widget_url
                            }
                        },
                        self.get_code_button(),
                        {
                            'block': 'b-form-input',
                            'mods': {'type': 'textarea', 'readonly': 'yes'},
                            'attrs': {'rows': 10, 'cols': 10},
                            'content': ''
                        }
                    ]
                }
            ]
        }


class InformerTemplate(BaseTemplate):

    def page_mods(self):
        return {'type': 'informer'}

    def radio(self, value, html, klass=None, active=None):
        btn = {
            'elem': 'button',
            'inputAttrs': { 'value': value},
            'content': html
        }

        if active == value:
            btn['elemMods'] = { 'checked': 'yes' }

        if klass:
            btn['cls'] = klass

        return btn

    def station_form(self, type, hint, data_informer, data_type, hidden=True):
        type0, type1 = self.context.default.types
        request = self.context.request

        suggest_query = {
            'format': 'old',
            'lang': get_language(),
            'client_city': request.client_city and request.client_city._geo_id,
            'national_version': request.NATIONAL_VERSION,
        }

        if type != 'any':
            suggest_query['t_type_code'] = type

        input = {
            'block': 'b-form-input',
            'attrs': {'style': 'margin-top: 20px;'},
            'mods': {'theme': 'grey', 'pick': 'yes', 'autocomplete': 'yes', 'has-clear': 'yes'},
            'js': {
                'dataprovider': {'url': settings.SUGGEST_URL + 'station?' + urlencode(suggest_query) },
                'debounceDelay': 100,
            },
            'content': [
                {'elem': 'label', 'content': '<b>%s</b>' % hint},

                {
                    'elem': 'input',
                    'clear': {'elem': 'clear', 'mods': {'visibility': 'visible'}},
                    'attrs': {'value': self.context.default.station.L_title() if type0 == 'schedule' else ''},
                },
                {
                    'elem': 'key',
                    'attrs': {
                        'name': 'station',
                        'type': 'hidden',
                        'value': self.context.default.station.point_key if type0 == 'schedule' else ''
                    }
                },
            ]
        }

        form = {
            'block': 'b-form',
            'cls': 'js-schedule-' + type,
            'attrs': {'data-informer': data_informer, 'data-type': data_type},
            'content': input
        }

        if hidden:
            form['mix'] = [{'block': 'i-hidden'}]

        return form

    def trip_stations_form(self, action, type, data_informer, data_type, hidden=True):
        suggest_type = type

        if suggest_type == 'any':
            suggest_type = 'all'

        form = {
            'block': 'b-informers-search',
            'js': {
                'name': 'b-informers-search',
                'searchType': suggest_type
            },
            'attrs': {'action': action, 'data-informer': data_informer, 'data-type': data_type},
            'cls': 'js-trip-' + type,
            'content': [
                {
                    'block': 'b-layout-table',
                    'mods': {'layout': 'ext-trip'},
                    'content': [
                        {
                            'elem': 'row',
                            'content': [
                                {
                                    'elem': 'cell',
                                    'mods': { 'type': 'label' },
                                    # http://rasp.yandex.ru/informers/informer/
                                    # Настройка - расписание по направлению - метка перед полем От
                                    'content': gettext('От', key="informers-label-from"),
                                },
                                {
                                    'elem': 'cell',
                                    'mods': {'type': 'input'},
                                    'content': [
                                        {
                                            'block': 'b-form-input',
                                            'mods': {'theme': 'grey', 'pick': 'yes', 'autocomplete': 'yes', 'has-clear': 'yes'},
                                            'js': {
                                                'dataprovider': {
                                                    'name': 'b-rasp-search__dataprovider',
                                                    'field': 'from',
                                                    'form': 'b-informers-search',
                                                    'suggest-url': settings.SUGGEST_URL,
                                                    'user-params': {
                                                        'national_version': self.context.request.NATIONAL_VERSION,
                                                        'client_city': self.context.request.client_city._geo_id,
                                                        'lang': get_language(),
                                                    }
                                                },
                                                'debounceDelay': 100,
                                            },
                                            'content': [
                                                {'elem': 'input', 'clear': {'elem': 'clear', 'mods': {'visibility': 'visible'}}, 'attrs': {'name': 'fromName', 'value': self.context.default['from'].L_title() if self.context.default.get('from') else ''}},
                                                {'elem': 'key', 'tag': 'input', 'attrs': {'name': 'fromId', 'value': self.context.default['from'].point_key if self.context.default['from'] else ''}, 'mix': [{'block': 'i-hidden'}]}
                                            ]
                                        }
                                    ]
                                },
                                {
                                    'elem': 'cell',
                                    'mods': {'type': 'changer'},
                                    'content': {
                                        'block': 'b-form-button',
                                        'mods': {'theme': 'grey-m', 'size': 'm', 'mod': 'revert', 'valign': 'middle', 'action': 'click'},
                                        'type': 'button',
                                        'content': [
                                            {
                                                'block': 'b-icon',
                                                'alt': gettext('поменять местами'),
                                                'title': gettext('поменять местами'),
                                            },
                                            '&#0160;'
                                        ]
                                    }
                                },
                                {
                                    'elem': 'cell',
                                    'mods': { 'type': 'label' },
                                    # http://rasp.yandex.ru/informers/informer/
                                    # Настройка - расписание по направлению - метка перед полем До
                                    'content': [gettext('До', key='informers-label-to'), '&nbsp;']
                                },
                                {
                                    'elem': 'cell',
                                    'mods': {'type': 'input'},
                                    'content': [
                                        {
                                            'block': 'b-form-input',
                                            'mods': {'theme': 'grey', 'pick': 'yes', 'autocomplete': 'yes', 'has-clear': 'yes'},
                                            'js': {
                                                'dataprovider': {
                                                    'name': 'b-rasp-search__dataprovider',
                                                    'field': 'to',
                                                    'form': 'b-informers-search',
                                                    'suggest-url': settings.SUGGEST_URL,
                                                    'user-params': {
                                                        'national_version': self.context.request.NATIONAL_VERSION,
                                                        'client_city': self.context.request.client_city._geo_id,
                                                        'lang': get_language(),
                                                    }
                                                },
                                                'debounceDelay': 100,
                                            },
                                            'content': [
                                                {'elem': 'input', 'clear': {'elem': 'clear', 'mods': {'visibility': 'visible'}}, 'attrs': {'name': 'toName', 'value': self.context.default['to'].L_title() if self.context.default.get('to') else ''}},
                                                {'elem': 'key', 'tag': 'input', 'attrs': {'type': 'hidden', 'name': 'toId', 'value': self.context.default['to'].point_key if self.context.default.get('to') else ''}}
                                            ]
                                        }
                                    ]
                                },
                            ]
                        }
                    ]
                }
            ]
        }
        if hidden:
            form['mix'] = [{'block': 'i-hidden'}, {'block': 'g-hidden'}]

        return form

    def js_schedule_block(self):
        type0, type1 = self.context.default.types

        types = self.js_schedule_block_types()

        content = [
            {
                'block': 'b-form-radio',
                'js': {'name': 'b-form-radio'},
                'mods': {'size': 'm', 'theme': 'grey'},
                'inputAttrs': {'name': 'schedule-type'},
                'content': [
                    self.radio(type.value, type.html, 'js-schedule-type', type1 if type0 == 'schedule' else 'any')
                    for type in types
                ]
            }
        ]

        for type in types:
            hidden = not (type.value == 'any' and type0 == 'schedule')
            data_type = 'schedule' if type.value == 'any' else type.value

            content.append(self.station_form(type.value, type.title, 'station', data_type, hidden=hidden))

        return {
            'elem': 'line',
            'cls': 'js-schedule',
            'mix': [{'block': 'i-hidden'}] if type0 != 'schedule' else None,
            'content': content
        }

    def js_schedule_block_types(self):

        class Type(object):

            def __init__(self, value, html, title):
                self.value = value
                self.html = html
                self.title = title

        lang = get_language()

        result = [
            Type('any', gettext('любой транспорт'), gettext('Станция, вокзал, автовокзал')),
            Type('train', gettext('поезда'), gettext('Станция, вокзал'))
        ]

        if lang != 'tr':
            result.append(Type('suburban', gettext('электрички'), gettext('Станция, вокзал')))

        result.append(Type('bus', gettext('автобусы'), gettext('Автовокзал')))

        result.append(Type('water', gettext('теплоходы'), gettext('Порт, речной вокзал, пристань')))

        return result

    def js_tablo_block(self):
        type0, type1 = self.context.default.types
        station = self.context.default.station

        request = self.context.request

        suggest_query = {
            'format': 'old',
            't_type_code': 'plane',
            'lang': get_language(),
            'client_city': request.client_city and request.client_city._geo_id,
            'national_version': request.NATIONAL_VERSION,
        }

        return {
            'elem': 'line',
            'cls': 'js-tablo',
            'content': {
                'block': 'b-form',
                'js': {'name': 'b-form'},
                'attrs': {'data-informer': 'station', 'data-type': 'tablo'},
                'mix': [{'block': 'i-hidden'}] if type0 != 'tablo' else None,
                'content': {
                    'block': 'b-form-input',
                    'mods': {'theme': 'grey', 'pick': 'yes', 'autocomplete': 'yes', 'has-clear': 'yes'},
                    'js': {
                        'dataprovider': { 'url': settings.SUGGEST_URL + 'station?' + urlencode(suggest_query) },
                        'debounceDelay': 100,
                    },
                    'content': [
                        {'elem': 'label', 'content': { 'tag': 'b', 'content': gettext('Аэропорт') }},
                        {'elem': 'input', 'clear': {'elem': 'clear', 'mods': {'visibility': 'visible'}}, 'attrs': {'value': station.L_title() if type0 == 'tablo' and station else ''}},
                        {'elem': 'key', 'tag': 'input', 'attrs': {'type': 'hidden', 'name': 'station', 'value': station.point_key if type0 == 'tablo' and station else ''}}
                    ]
                }
            }
        }

    def js_trip_block(self):
        type0, type1 = self.context.default.types

        types = self.js_trip_block_types()

        content = [
            {
                'block': 'b-form-radio',
                'mods': {'size': 'm', 'theme': 'grey'},
                'inputAttrs': {'name': 'trip-type'},
                'content': [
                    self.radio(type.value, type.html, 'js-trip-type', type1 if type0 == 'trip' else 'any')
                    for type in types
                ]
            }
        ]

        for type in types:
            content.append(self.trip_stations_form(type.action, type.value, 'search', type.data_type, type.hidden))

        return {
            'elem': 'line',
            'cls': 'js-trip',
            'mix': [{'block': 'i-hidden'}] if type0 != 'trip' else None,
            'content': content
        }

    def js_trip_block_types(self):
        type0, type1 = self.context.default.types

        class Type(object):

            def __init__(self, value, html, action, data_type=None):
                self.value = value
                self.html = html
                self.action = action
                self.data_type = value if data_type is None else data_type
                self.hidden = not (type0 == 'trip' and type1 == value)

        lang = get_language()

        result = [
            Type('any', gettext('любой транспорт'), '/search/', ''),
            Type('plane', gettext('самолёты'), '/search/plane/'),
            Type('train', gettext('поезда'), '/search/train/')
        ]

        if lang != 'tr':
            result.append(Type('suburban', gettext('электрички'), '/search/suburban/'))

        result.append(Type('bus', gettext('автобусы'), '/search/bus/'))

        result.append(Type('water', gettext('теплоходы'), '/search/water/'))

        return result

    def color_input_cells(self, i, values):
        return [{
            'elem': 'cell',
            'content': {
                'block': 'b-form-input',
                'mods': { 'size': 'm', 'theme': 'grey', 'mod': 'short' },
                'content': [
                    { 'elem': 'input', 'attrs': {'value': values[i]} }
                ]
            }
        } for i in xrange(i)]

    def right_content(self):
        type0, type1 = self.context.default.types

        return {
            'elem': 'right',
            'content': [
                {
                    'block': 'b-page-title',
                    'content': [
                        { 'elem': 'title', 'content': gettext('Настройка информера') },
                        {
                            'elem': 'disclaimer',
                            'tag': 'p',
                            'attrs': {'style': 'width: 70%'},
                            'content': dedent(gettext("""\
                                Всего за пару минут вы можете установить на свой сайт или в блог
                                информер с данными о расписании транспорта. Выберите, какие данные вам интересны,
                                укажите тип транспорта (если потребуется), а также количество строк и
                                цвет информера. После этого вам остается получить код и разместить его у себя.
                            """))
                        }
                    ]
                },
                {
                    'block': 'b-tabbed-pane',
                    'mods': {'preset': 'big'},
                    'content': [
                        {
                            'elem': 'tabs',
                            'content': [
                                {
                                    'elem': 'tab',
                                    'elemMods': {'state': 'current'},
                                    'content': gettext('Настройка')
                                },
                                {
                                    'elem': 'tab',
                                    'content': gettext('Оформление')
                                }
                            ]
                        },
                        {
                            'elem': 'panels',
                            'content': [
                                {
                                    'elem': 'panel',
                                    'elemMods': {'state': 'current'},
                                    'content': {
                                        'block': 'b-informer-new',
                                        'content': [
                                            {
                                                'elem': 'line',
                                                'content': {
                                                    'block': 'b-form-radio',
                                                    'mods': {'size': 'm', 'theme': 'grey'},
                                                    'inputAttrs': {'name': 'informer-type'},
                                                    'content': [
                                                        self.radio(value, html, 'js-informer-type', type0)
                                                        for value, html in [
                                                            ('schedule', gettext('расписание по станции')),
                                                            ('tablo', gettext('табло аэропортов')),
                                                            ('trip', gettext('расписание по направлению')),
                                                        ]
                                                    ]
                                                }
                                            },
                                            self.js_schedule_block(),
                                            self.js_tablo_block(),
                                            self.js_trip_block(),
                                        ]
                                    }
                                },
                                {
                                    'elem': 'panel',
                                    'content': {
                                        'block': 'b-informer-new',
                                        'content': [
                                            {
                                                'elem': 'line',
                                                'content': [
                                                    {
                                                        'block': 'b-ext-i-rows',
                                                        'content': [
                                                            { 'elem': 'title', 'content': gettext('Строки') },
                                                            {
                                                                'block': 'b-form-radio',
                                                                'mods': { 'size': 'm', 'theme': 'grey', 'type': 'size' },
                                                                'attrs': { 'name': 'trip-rows' },
                                                                'inputAttrs': {'name': 'trip-rows'},
                                                                'content': [
                                                                    { 'elem': 'button', 'elemMods': { 'checked': 'yes' }, 'content': 5, 'inputAttrs': {'value': 5}},
                                                                    { 'elem': 'button', 'content': 15, 'inputAttrs': {'value': 15} },
                                                                    { 'elem': 'button', 'content': 25, 'inputAttrs': {'value': 25} }
                                                                ]
                                                            }
                                                        ]
                                                    },
                                                    {
                                                        'block': 'b-layout-table',
                                                        'mods': {'layout': '50-50'},
                                                        'content': [
                                                            {
                                                                'elem': 'cell',
                                                                'elemMods': { 'type': 'left' },
                                                                'content': {
                                                                    'block': 'b-informer-color',
                                                                    'mods': {'is_bem': 'yes'},
                                                                    'js': True,
                                                                    'content': [
                                                                        { 'elem': 'title', 'content': gettext('Цвет информера') },
                                                                        { 'elem': 'item', 'elemMods': { 'color': '1', 'state': 'current' } },
                                                                        { 'elem': 'item', 'elemMods': { 'color': '2' } },
                                                                        { 'elem': 'item', 'elemMods': { 'color': '3' } },
                                                                        { 'elem': 'item', 'elemMods': { 'color': '4' } },
                                                                        { 'elem': 'item', 'elemMods': { 'color': '5' } },
                                                                        { 'elem': 'item', 'elemMods': { 'color': '6' } },
                                                                        { 'elem': 'item', 'elemMods': { 'color': '7' } },
                                                                        { 'elem': 'item', 'elemMods': { 'color': '8' } }
                                                                    ]
                                                                }
                                                            },
                                                            {'elem': 'gap'},
                                                            {
                                                                'elem': 'cell',
                                                                'elemMods': { 'type': 'right' },
                                                                'content': {
                                                                    'block': 'b-color-selector',
                                                                    'js': True,
                                                                    'content': [
                                                                        { 'elem': 'title', 'content': gettext('Цвет фона вашего сайта') },
                                                                        { 'elem': 'picker', 'content': [
                                                                            { 'elem': 'ring', 'content': { 'elem': 'dot', 'position': 'left: 72px; top: 13px;' } },
                                                                            { 'elem': 'square', 'color': 'rgb(255, 211, 0)', 'content': { 'elem': 'dot', 'position': 'left: 45px; top: -7px;' } }
                                                                        ]},
                                                                        { 'elem': 'data', 'content': {
                                                                            'elem': 'layout',
                                                                            'content': [
                                                                                { 'elem': 'row', 'content': [
                                                                                    { 'elem': 'cell', 'attrs': { 'rowspan': 2 } },
                                                                                    { 'elem': 'cell', 'elemMods': { 'type': 'label' }, 'content': 'R' },
                                                                                    { 'elem': 'cell', 'elemMods': { 'type': 'label' }, 'content': 'G' },
                                                                                    { 'elem': 'cell', 'elemMods': { 'type': 'label' }, 'content': 'B' },
                                                                                    { 'elem': 'gap' },
                                                                                    { 'elem': 'cell', 'elemMods': { 'type': 'label' }, 'content': 'H°' },
                                                                                    { 'elem': 'cell', 'elemMods': { 'type': 'label' }, 'content': 'S' },
                                                                                    { 'elem': 'cell', 'elemMods': { 'type': 'label' }, 'content': 'V' }
                                                                                ] },
                                                                                {
                                                                                    'elem': 'row',
                                                                                    'content': self.color_input_cells(3, ('255', '204', '0')) + [{'elem': 'gap'}] + self.color_input_cells(3, ('48', '100', '100'))
                                                                                },
                                                                                {
                                                                                    'elem': 'row',
                                                                                    'content': [
                                                                                        { 'elem': 'cell', 'elemMods': { 'type': 'label' }, 'content': '#' },
                                                                                        { 'elem': 'cell', 'attrs': { 'colspan': 3 }, 'content': {
                                                                                            'block': 'b-form-input',
                                                                                            'mods': { 'size': 'm', 'theme': 'grey' },
                                                                                            'content': [
                                                                                                { 'elem': 'input', 'attrs': {'value': 'ffffff'} }
                                                                                            ]
                                                                                        } },
                                                                                        { 'elem': 'cell', 'attrs': { 'colspan': 3 }, 'content': ' ' }
                                                                                    ]
                                                                                }
                                                                            ]
                                                                        } }
                                                                    ]
                                                                }

                                                            }
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                }
                            ]
                        }
                    ]
                },
                {
                    'block': 'disclaimer',
                    'tag': 'p',
                    'content': { 'tag': 'b', 'content': gettext('Так информер будет выглядеть на вашем сайте') },
                },
                {
                    'block': 'b-informer-new',
                    'mods': {'change': 'background'},
                    'content': [
                        {
                            'elem': 'line',
                            'mods': {'under': 'yes'},
                            'content': [
                                {
                                    'block': 'b-iframe',
                                    'mods': {'type': 'informer'},
                                    'content': '''<iframe  frameborder="0" id="ext-preview"
                                                onload="this.height = (this.contentDocument || this.contentWindow.document).body.offsetHeight + 3"
                                                style="overflow: hidden; border: 0; width: 700px;"
                                                src="%s"></iframe>''' % self.context.default.url
#                                    'content': '''<iframe  frameborder="0" width="700px" id="ext-preview"
#                                                onload="this.height = (this.contentDocument || this.contentWindow.document).body.offsetHeight"
#                                                style="overflow: hidden; border: 0; -moz-box-shadow: 0 2px 3px rgba(0, 0, 0, 0.5);
#                                                -webkit-box-shadow: 0 2px 3px rgba(0, 0, 0, 0.5); box-shadow: 0 2px 3px rgba(0, 0, 0, 0.5);"
#                                                src="%s"></iframe>''' % self.context.default.url
                                },
                                {
                                    'elem': 'under-informer'
                                }
                            ]
                        },
                        self.get_code_button()
                    ]
                }
            ]
        }
