# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import json
import re
from datetime import date, timedelta
import math

from django.conf import settings
from django.utils.http import urlencode
from django.utils.translation import get_language
from pytils.numeral import choose_plural

from common.models.transport import TransportType
from common.models.disclaimers import StaticText
from common.models_utils import DisplayTCodeMixin
from common.utils.bemhtml import punctuate_content, ru_human_list_ili
from common.utils.date import DateTimeFormatter, human_date, human_duration, timedelta2minutes, astimezone
from common.utils.text import normalize, mdash_wrappable
from common.utils.locations import composeurl
from common.xgettext.i18n import gettext, xgettext, xngettext, xformat, dynamic_gettext, tngettext, tgettext
from common.utils.blablacar_utils import BLABLACAR_ALL_DAYS_DATE, is_valid_blablacar_direction
import timetable
import sort
from base import Base
from filters import BlockFilter
from small_description import SmallDescription
from utils import headnonzerononfail, segment_strong_number, search_url, buy_link


TEL_RE = re.compile(r'^\+?[\d\s]+$', re.U)

BLABLACAR_POSITION = 2
BLABLACAR_AWAPS_SHOW_URL = None
BLABLACAR_AWAPS_CLICK_URL = None


class BlockTimetable(timetable.Sortable):
    def __init__(self, base):
        self.base = base

        context = self.base.context

        self.context = context
        self.search_type = context.search_type
        self.point_from = context['from']
        self.point_to = context['to']

        segments = context.segments

        # Страница поиска редиректит реальных пользователей на новую морду, поэтому
        # убираем БАРиС и сегменты с пересадками, так как сарый поиск начинает из-за них пятисотить
        self.segments = [s for s in segments if hasattr(s, 'L_title') and not hasattr(s, 'segments')]

        interval_mask = [headnonzerononfail(lambda: s.thread.is_interval) for s in self.segments]

        self.max_icons = None

        self.interval_segments = [s for i, s in enumerate(self.segments) if interval_mask[i]]

        self.plain_segments = [s for i, s in enumerate(self.segments) if not interval_mask[i]]

        self.visible_notes = set()

        self.mode = 'date' if context['date'] else 'schedule'

        self.show_tariffs = self.show_days = self.show_stops = False

        self.show_t_ico = self.show_number = self.show_desc = True

        self.preset = 'trans'

        if self.mode == 'schedule':
            columns = {
                'departure': sort.TimeColumn('departure', self.base),
                'arrival': sort.TimeColumn('arrival', self.base),
                'duration': sort.RawColumn('duration'),
            }
        else:
            columns = {
                'departure': sort.DateTimeColumn('departure'),
                'arrival': sort.DateTimeColumn('arrival'),
                'duration': sort.RawColumn('duration'),
            }

            self.show_tariffs = True

        if self.search_type == 'suburban':
            self.show_t_ico = True
            self.show_tariffs = False
            self.show_stops = True
            self.show_number = False
            self.show_desc = False
            self.preset = 'suburban'

        if self.show_tariffs:
            columns['tariff'] = sort.TariffColumn()

        if self.mode == 'schedule':
            self.show_days = True

        self.sort = sort.Sort(self.base, 'departure', columns)

        self.init_column_count()

        self.init_gone_filter()

        self.transport_types = set()
        for s in self.segments:
            if hasattr(s, 'transport_types'):
                self.transport_types.update(s.transport_types)
            elif hasattr(s, 't_type'):
                self.transport_types.add(s.t_type)

    def init_column_count(self):
        column_count = self.context.request.GET.get('column_count')
        if column_count:
            try:
                column_count = int(column_count)
            except ValueError:
                pass

        if not column_count:
            column_count = 4 + self.show_t_ico + self.show_stops + self.show_days + self.show_tariffs

        self.column_count = column_count

    def init_gone_filter(self):
        filters = self.context.filters

        visible = [s for s in self.segments if not filters.is_filtered(s)]

        show_gone = self.context.request.GET.get('showGone')

        # RASP-5318, раскрывать ушедшие при добавлении пересадок, если все пересадки в ушедших
        if self.context.add_transfers and all(t.gone for t in self.context.transfers_variants):
            show_gone = True

        show_filter = any(s.gone for s in visible) and any(not s.gone for s in visible)

        self.hide_gone = show_filter and not show_gone

        content = []

        for code, title in [
            ('show', gettext('показать ушедшие')),
            ('hide', gettext('скрыть ушедшие')),
        ]:
            mods = { 'pseudo': 'yes' }

            if code == 'show' and show_gone or code == 'hide' and not show_gone:
                mods['visibility'] = 'hidden'

            content.append({ 'block': 'b-link', 'mods': mods, 'content': title })

        self.gone_filter = {
            'elem': 'show-hidden',
            'mix': [not show_filter and { 'block': 'i-hidden' } ],
            'content': [
                '(', content, ')'
            ]
        }

    def hrows(self):
        station_from = station_to = None

        if self.search_type == 'suburban' and self.segments:
            segment = self.segments[0]
            station_from = segment.station_from
            station_to = segment.station_to

        def station_elem(station):
            if not station:
                return

            return {
                'elem': 'header-station',
                'content': station.L_short_title_with_prefix()
            }

        columns = [
            {
                'elem': 'cell',
                'elemMods': { 'type': 'trip' },
                'attrs': { self.show_t_ico and 'colspan': 2 },
                'content': [
                    gettext('рейсы'),
                    self.gone_filter
                ]
            },
            [
                self.sort_hrow_cell('departure', gettext('отправление'), [station_elem(station_from) ]),
                self.sort_hrow_cell('arrival', gettext('прибытие'), [station_elem(station_to) ]),
                self.sort_hrow_cell('duration', gettext('в пути')),
            ]
        ]

        self.show_stops and columns.append({
            'elem': 'cell',
            'elemMods': { 'type': 'station' },
            'content': gettext('остановки'),
        })

        if self.show_days:
            columns.append({
                'elem': 'cell',
                'elemMods': { 'type': 'days' },
                'content': gettext('дни курсирования')
            })

        if self.show_tariffs:
            columns.append(self.sort_hrow_cell('tariff', gettext('цена')))

        currency_switch = self.base.currency_switch('grey')

        return [
            {
                'elem': 'hrow',
                'content': [
                    self.show_t_ico and { 'elem': 'cell' },
                    { 'elem': 'cell' },
                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'timeselector' },
                        'content': {
                            'block': 'b-timeselector',
                            'content': [
                                {
                                    'elem': 'selector',
                                    'content': [
                                        {
                                            'block': 'b-icon'
                                        },
                                        self.base.timeselector(),
                                    ]
                                },
                                { 'elem': 'line' }
                            ]
                        }
                    },
                    { 'elem': 'cell' },
                    { 'elem': 'cell' },
                    self.show_stops and { 'elem': 'cell' },
                    self.show_days and { 'elem': 'cell' },
                    self.show_tariffs and {
                        'elem': 'cell',
                        'elemMods': { 'type': 'priceselector' },
                        'content': currency_switch and {
                            'block': 'b-priceselector',
                            'content': [
                                { 'block': 'b-icon' }, ' ', currency_switch,
                            ]
                        }
                    },
                ]
            },
            {
                'elem': 'hrow',
                'elemMods': { 'position': 'last' },
                'content': columns
            }
        ]

    def segment_link(self, segment):
        thread = getattr(segment, 'thread', None)

        if not thread:
            return None

        params = {
            'departure': segment.start_date,
            'station_from': segment.station_from.id,
            'station_to': segment.station_to.id,
        }

        if self.point_from and self.point_to:
            params.update({
                'point_from': self.point_from.point_key,
                'point_to': self.point_to.point_key,
            })

        return composeurl('thread', args=[thread.uid], params=params)

    def tripname(self, segment):
        content = []

        if self.show_number:
            content.append(segment_strong_number(segment))

        title = segment.L_title() if hasattr(segment, 'L_title') else ''

        if title:
            content.append(mdash_wrappable(title))

        content = punctuate_content(content, ' ')

        link = self.segment_link(segment)

        if link:
            content = {'block': 'b-link', 'url': link, 'content': content}

        return {
            'elem': 'tripname',
            'content': content
        }

    def description(self, segment, hidden=False):
        content = timetable.segment_description(segment)

        add_eticket = show_eticket = False

        tariffs = segment.display_info.tariffs_info

        if tariffs:
            add_eticket = tariffs.et_marker or tariffs.dynamic
            show_eticket = tariffs.et_marker

            if show_eticket and not hidden:
                self.visible_notes.add('e-ticket')

        if content or add_eticket:
            return {
                'elem': 'description',
                'content': [
                    punctuate_content(content),
                    add_eticket and {
                        'elem': 'eticket',
                        'tag': 'span',
                        'mix': [not show_eticket and { 'block': 'i-hidden' } ],
                        'content': [' ', self.base.b_icon_type_eticket() ]
                    }
                ]
            }

    def point(self, station, rtstation, force_station=False):
        if station:
            if self.search_type == 'suburban' and not force_station:
                if rtstation and rtstation.L_platform():
                    return {
                        'elem': 'holder-name',
                        'content': {
                            'elem': 'platform',
                            'content': self.base.station_schema_or_platform(rtstation)
                        }
                    }

            else:
                content = [
                    {
                        'block': 'b-link',
                        'mods': { 'theme': 'gray' },
                        'url': composeurl('station', args=[station.pk]),
                        'content': station.L_popular_title_extra(rtstation=rtstation)
                    }
                ]

                if rtstation and rtstation.L_platform():
                    content.extend([
                        ', ',
                        self.base.station_schema(station, rtstation.L_platform()) if station.schema_image else {
                            'elem': 'platform',
                            'elemMods': { 'inline': 'yes' },
                            'content': rtstation.L_platform()
                        }
                    ])

                return { 'elem': 'holder-name', 'content': content }

    def pathtime(self, duration):
        return {
            'elem': 'pathtime',
            'content': human_duration(duration)
        }

    def express(self, thread):
        attrs = {}
        if thread.express_lite:
            content = thread.express_lite.title
            attrs['style'] = 'color: %s' % thread.express_lite.color
        elif thread.is_express:
            content = gettext('экспресс')
        elif thread.is_aeroexpress:
            content = gettext('аэроэкспресс')
        else:
            return

        return {
            'elem': 'express',
            'attrs': attrs,
            'content': content
        }

    def duration_with_express(self, segment):
        return [
            self.pathtime(segment.duration),
            headnonzerononfail(lambda: segment.thread) and self.express(segment.thread),
        ]

    def tariff(self, segment, place, is_min_tariff=False):
        content = self.base.b_currency(place.tariff, show_cents=False, from_=is_min_tariff or getattr(place, 'min', False))

        url = None

        if segment and segment.display_info and segment.display_info.tariffs_info and hasattr(segment.display_info.tariffs_info, 'url'):
            url = segment.display_info.tariffs_info.url
        elif getattr(place, 'link', False):
            url = place.link(self.context.request, segment, buy_link, self.point_from, self.point_to)

        if url:
            content = {
                'block': 'b-form-button',
                'mods': { 'theme': 'action' },
                'target': '_blank',
                'url': url,
                'content': content
            }

            metrika = getattr(place, 'metrika', None)

            if metrika:
                content['counter'] = 'yaCounter.hit(%s)' % json.dumps(metrika)

        return {
            'elem': 'money',
            'content': content
        }

    def aux_tariffs(self, tariffs):
        base = self.base.b_currency(tariffs.base.tariff, show_cents=False)

        def tariff_item(t):
            content = [t.type.title_with_link, ' ', self.base.b_currency(t.tariff, show_cents=False) ]

            if t.type == tariffs.base.type:
                content = { 'tag': 'strong', 'content': content }

            return {
                'elem': 'item',
                'content': {
                    'block': 'b-price',
                    'content': content,
                }
            }

        items = [tariff_item(t) for t in tariffs['usual']]

        if tariffs.get('season_ticket'):
            items.extend([
                { 'elem': 'separator' },
                {
                    'elem': 'item',
                    # http://rasp.yandex.ua/search/?fromName=%D0%9C%D0%BE%D1%81%D0%BA%D0%B2%D0%B0&fromId=&toName=%D0%9F%D0%B5%D1%82%D1%83%D1%88%D0%BA%D0%B8&toId=c10669&when=17+%D0%B0%D0%BF%D1%80%D0%B5%D0%BB%D1%8F
                    # При клике на цену проезда электрички, раздел "проездные"
                    'content': gettext('проездные'),
                },
            ])

            items.extend(tariff_item(t) for t in tariffs['season_ticket'])

        items[0]['position'] = 'first'
        items[-1]['position'] = 'last'

        return {
            'block': 'b-dropdowna',
            'mods': { 'color': 'black' },
            'content': [
                {
                    'elem': 'switcher',
                    'content': {
                        'block': 'b-link',
                        'mods': { 'pseudo': 'yes' },
                        'content': base
                    }
                },
                {
                    'block': 'b-popupa',
                    'mods': { 'theme': 'ffffff', 'direction': 'right' },
                    'content': [
                        { 'elem': 'tail' },
                        {
                            'elem': 'content',
                            'content': {
                                'block': 'b-menu',
                                'mods': { 'layout': 'vert'},
                                'mix': [{ 'block': 'b-dropdowna', 'elem': 'menu' } ],
                                'content': items
                            }
                        }
                    ]
                }
            ]
        }

    def tariffs(self, sort_row=None, segment=None):
        if segment is None:
            segment = sort_row.row

        aux_tariffs = segment.display_info.aux_tariffs
        tariffs = segment.display_info.tariffs_info

        if aux_tariffs:
            return {
                'block': 'b-price',
                'content': {
                    'elem': 'money',
                    'content': self.aux_tariffs(aux_tariffs)
                }
            }

        elif tariffs:
            content = self.tariffs_places(sort_row and sort_row.sub_sort_places or tariffs.places, segment)

            if content or tariffs.wait:
                return [
                    {
                        'elem': 'sub-places',
                        'js': tariffs.dynamic and tariffs.link(self.context.request, segment, buy_link,
                                                               self.point_from, self.point_to),
                        'content': content
                    },
                    tariffs.wait and {
                        'block': 'b-spin',
                        'mods': { 'size': '10', 'theme': 'white-10', 'progress': 'yes', 'request-key': tariffs.request_key }
                    }
                ]
            else:
                try:
                    phone = segment.display_info.phone
                except AttributeError:
                    phone = None

                if phone:
                    return {
                        'block': 'b-tel',
                        'content': [
                            {
                                'block': 'b-icon',
                            },
                            phone,
                        ]
                    }

        return ''

    def tariffs_places(self, places, segment=None):
        return [
            {
                'block': 'b-price',
                'mods': { 'theme': 'yellow' },
                'mix': [{ 'block': 'b-timetable', 'elem': 'sub-place', 'js': { 'index': i } } ],
                'js': {'partner': place.partner_code} if hasattr(place, 'partner_code') else False,
                'content': [
                    place.name and { 'elem': 'type', 'content': dynamic_gettext(place.name) },
                    place.tariff and [' ', self.tariff(segment, place)]
                ]
            } for i, place in enumerate(places)
        ]

    def sort_row_cell(self, column, content, sort_value=None):
        mods = { 'type': column }

        if column == self.sort.column_code:
            mods['state'] = 'current'

        rv = {
            'elem': 'cell',
            'elemMods': mods,
            'content': content
        }

        if sort_value is not None:
            rv['js'] = { 'sort-value': sort_value }

        return rv

    def days_texts(self, segment, *args, **kwargs):
        shifts = {}

        if segment.departure:
            dt = segment.departure
        else:
            now = self.context.request.now
            dt = now.astimezone(segment.station_from.pytz)

        time_zone = self.context.time_zone
        tz_cities = self.context.tz_cities
        thread = segment.thread

        start_date = segment.start_date or dt.date()

        kwargs['thread_start_date'] = start_date

        if time_zone not in tz_cities:
            tz_cities.append(time_zone)

        for tz in [None] + tz_cities:
            date_ = astimezone(dt, tz).date()

            shift = (date_ - start_date).days

            shifts.setdefault(shift, set()).add(tz)

        for shift, timezones in shifts.items():
            active = time_zone in timezones

            yield timezones, thread.L_days_text(shift, *args, **kwargs), active

    def days(self, segment):
        url = composeurl('thread', args=[segment.thread.uid],
                         params={'station_from': segment.station_from.id})

        extra_link = [
            ', ',
            {
                'block': 'b-link',
                'url': url,
                'content': '…'
            }
        ]

        texts = self.days_texts(
            segment,
            extra_link=extra_link,
            next_plan=self.context.next_plan
        )

        return [
            {
                'elem': 'days',
                'mix': [
                    { 'elemMods': { 'tz': tz and str(tz.id) or 'null' } }
                    for tz in text_timezones
                ] + [not active and { 'block': 'i-hidden' } ],
                'content': text
            }
            for text_timezones, text, active in texts
        ]

    def details(self, key, hide, show, elem='details'):
        show_details = key in self.context.request.GET.getlist('details')

        if show_details:
            trigger = self.base.b_link_change_params(hide, details__remove=key)
        else:
            trigger = self.base.b_link_change_params(show, details__add=key)

        trigger = {
            'elem': elem + '-trigger',
            'elemMods': { 'state': show_details and 'current' or None },
            'content': trigger
        }

        return show_details, trigger

    def ico(self, display_t_code):
        display_t_name = DisplayTCodeMixin.t_code_name(display_t_code)

        return { 'block': 'b-transico', 'mods': { 'type': display_t_code }, 'alt': display_t_name }

    def icons(self, segment):
        display_t_codes = getattr(segment, 'display_t_codes', None)

        if display_t_codes:
            return [self.ico(display_t_code) for display_t_code in display_t_codes]

        return self.ico(getattr(segment, 'display_t_code', None))

    def ico_cell(self, icons):
        return {
            'elem': 'cell',
            'elemMods': { 'type': 'ico', 'size': self.max_icons and unicode(self.max_icons) },
            'content': icons
        }

    def stops(self, segment):
        return {
            'elem': 'cell',
            'elemMods': {'type': 'station'},
            'content': (lambda stops: stops and {
                'elem': 'stations',
                'content': stops,
            })(hasattr(segment, 'L_stops') and segment.L_stops())
        }

    def plain_rows(self):
        sort_rows = self.sort.sort(self.plain_segments, key=lambda s: headnonzerononfail(lambda: s.info.route_key))

        filters = self.context.filters

        departure_formatter = timetable.TimeWithDateIfChangedFormatter(self.base) if self.base.context.date else timetable.TimeOnlyFormatter(self.base)
        arrival_formatter = timetable.TimeWithDateIfChangedFormatter(self.base) if self.base.context.date else timetable.TimeOnlyFormatter(self.base)

        visible_rows_count = 0

        json_rows = []

        json_blablacar_row = self.blablacar_row(self.pop_blablacar_row(sort_rows))
        blablacar_row_added = False

        for row_index, sort_row in enumerate(sort_rows):
            if not blablacar_row_added and visible_rows_count == BLABLACAR_POSITION and json_blablacar_row:
                json_rows.append(json_blablacar_row)
                blablacar_row_added = True

            segment = sort_row.row

            mods = { 'sortable': 'yes' }

            if segment.gone:
                mods['gone'] = 'yes'

            hidden = filters.is_filtered(segment) or segment.gone and self.hide_gone

            if not hidden:
                visible_rows_count += 1

            comment, comment_trigger = self.comment(segment, hidden)

            if comment:
                mods['group'] = 'bus'

            key = unicode(sort_row.key)

            show_details, trigger = self.details(key, gettext('скрыть рейсы'), gettext('показать рейсы'))

            if show_details:
                mods['type'] = 'opened'

            transfers = getattr(segment, 'transfers', None)

            if transfers:
                mods['transfers'] = 'yes'

            stabilizers = []

            if segment.departure:
                if self.mode == 'schedule':
                    stabilizers.append(segment.departure.strftime('%H:%M'))
                else:
                    stabilizers.append(segment.departure.strftime('%Y-%m-%d %H:%M'))

            if transfers:
                # Сегменты пересадок отображаются иконками
                # Eсли иконки не отображаются, то нужно писать число пересадок
                show_transfers_count = not self.show_t_ico

                if get_language() == 'ru':
                    if len(transfers) == 1:
                        transfers_title = xformat('с пересадкой в <points-title case="locative"/>', points_title=transfers[0].L_title)

                    else:
                        def points_list(case):
                            titles = [point.L_title(case) for point in transfers ]

                            if len(titles) > 1:
                                return [punctuate_content(titles[:-1]), ' и&nbsp;', titles[-1] ]

                            return titles[0]

                        transfers_title = xformat('через <points-list case="accusative"/>', points_list=points_list)

                        if show_transfers_count:
                            transfers_title = [
                                transfers_title,
                                ' ',
                                '(',
                                choose_plural(len(transfers), ['%d&nbsp;пересадка', '%d&nbsp;пересадки', '%d&nbsp;пересадок']) % len(transfers),
                                ')'
                            ]

                else:
                    if show_transfers_count:
                        transfers_title = xngettext(
                            len(transfers),
                            '<num/>&nbsp;пересадка: <points-list/>',
                            '<num/>&nbsp;пересадки: <points-list/>',
                            '<num/>&nbsp;пересадок: <points-list/>',
                            num=len(transfers),
                            points_list=punctuate_content([ point.L_title() for point in transfers ])
                        )

                    else:
                        transfers_title = xgettext('пересадки: <points-list/>', points_list=punctuate_content([ point.L_title() for point in transfers ]))

            if segment.gone and not settings.SHOW_TRANSFER_PRICES:
                tariffs = { 'elem': 'gone', 'content': gettext('ушёл') }

            else:
                tariffs = self.tariffs(sort_row)

            json_rows.append({
                'block': 'b-timetable',
                'elem': 'row',
                'elemMods': mods,
                'mix': [hidden and { 'block': 'i-hidden' } ],
                'js': {
                    'key': sort_row.key,
                    'stabilizers': stabilizers,
                    'filter-values': filters.filter_values(segment)
                },
                'content': [

                    self.show_t_ico and self.ico_cell(self.icons(segment)),

                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'trip' },
                        'content': [

                            {
                                'elem': 'transpoint',
                                'content': transfers_title
                            },

                            trigger

                        ] if transfers else [

                            self.tripname(segment),
                            self.show_desc and self.description(segment, hidden),

                        ]
                    },

                    self.sort_row_cell('departure', [
                        segment.departure and departure_formatter(segment.departure, hidden),
                        [
                            self.point(segment.station_from, segment.rtstation_from),
                            self.delay(segment, 'departure')
                        ]
                    ]),

                    self.sort_row_cell('arrival', [
                        segment.arrival and arrival_formatter(segment.arrival, hidden),
                        [
                            self.point(segment.station_to, segment.rtstation_to),
                            self.delay(segment, 'arrival')
                        ]
                    ]),

                    self.sort_row_cell('duration', self.duration_with_express(segment), timedelta2minutes(segment.duration) if segment.duration else None),

                    self.show_stops and self.stops(segment),

                    self.show_days and {
                        'elem': 'cell',
                        'content': [
                            self.days(segment),
                            comment and not self.show_tariffs and comment_trigger
                        ]
                    },

                    self.show_tariffs and self.sort_row_cell('tariff', [
                        settings.SHOW_TRANSFER_PRICES and ['у. %s' % getattr(segment, 'convenience', '0'), tariffs] or tariffs,
                        comment_trigger
                    ]),

                ]
            })

            if transfers:
                json_rows.append({
                    'elem': 'row',
                    'elemMods': { 'key': key, 'details': 'yes', 'visibility': not show_details and 'hidden' },
                    'mix': [hidden and { 'block': 'i-hidden' } ],
                    'content': {
                        'elem': 'cell',
                        'elemMods': { 'type': 'group' },
                        'attrs': { 'colspan': '6' },
                        'content': self.transfers(segment)
                    }
                })

            if comment:
                json_rows.append(comment)

        if json_blablacar_row and not blablacar_row_added:
            json_rows.append(json_blablacar_row)

        return json_rows, visible_rows_count

    def pop_blablacar_row(self, sort_rows):
        blablacar_rows = filter(
            lambda sort_row: getattr(sort_row.row, 't_type', None) and sort_row.row.t_type.id == TransportType.BLABLACAR,
            sort_rows)

        if blablacar_rows:
            blablacar_row = blablacar_rows[0]
            sort_rows.remove(blablacar_row)
            return blablacar_row

        return None

    def blablacar_row(self, row=None):
        if not is_valid_blablacar_direction(self.context['from'], self.context['to']):
            return None

        blablacar_date = self.base.context.date or BLABLACAR_ALL_DAYS_DATE

        blablacar_request_key = 'blablacar_%s' % blablacar_date.isoformat() if type(blablacar_date) is date else None


        if not (row or blablacar_request_key and self.context.ajax_tariffs_info and blablacar_request_key in self.context.ajax_tariffs_info.request_keys):
            return None

        mods = {}

        if row:
            mods['loading'] = 'success'
            segment = row.row
            key = row.key

            # округление продолжительности пути до 10 минут в большую сторону
            duration_in_sec = math.trunc(math.ceil(segment.duration.total_seconds() / 600)) * 600
            duration = timedelta(seconds = duration_in_sec)

            data = {
                'route': mdash_wrappable(segment.L_title()),
                'time': '~%s' % human_duration(duration),
                'link_text': tngettext(
                    segment.number_of_variants,
                    '<n/> предложение',
                    '<n/> предложения',
                    '<n/> предложений'
                ) if hasattr(segment, 'number_of_variants') else None,
            }

            tariffs = segment.display_info.tariffs_info
            if tariffs:
                data['url'] = getattr(tariffs, 'url', None)

                if tariffs.places and len(tariffs.places):
                    data['tariff'] = self.tariff(segment, tariffs.places[0], True)

        else:
            mods['loading'] = 'progress'
            key = 'bla-bla-car-%02d%02d' % (blablacar_date.month, blablacar_date.day)
            data = None

        return {
            'block': 'b-blablacar',
            'columns': self.column_count,
            'mods': mods,
            'mix': [
                {
                    'block': 'b-timetable',
                    'elem': 'row',
                    'elemMods': { 'sortable': 'yes' },
                    'js': {
                        'key': key,
                        'filter-values': {}
                    }
                }
            ],
            'js': {
                'data': data,
                'awaps': {
                    'showUrl': BLABLACAR_AWAPS_SHOW_URL,
                    'clickUrl': BLABLACAR_AWAPS_CLICK_URL
                }
            }
        }

    def transfers_tripname(self, segment):
        # RASP-12721
        lines = []

        if self.show_number:
            strong_number = segment_strong_number(segment)

            if strong_number:
                lines.append(strong_number)

        title = segment.L_title()

        if title:
            lines.append(mdash_wrappable(title))

        if lines:
            link = self.segment_link(segment)

            if link:
                lines[0] = { 'block': 'b-link', 'url': link, 'content': lines[0] }

        content = punctuate_content(lines, '<br />')

        return {
            'elem': 'tripname',
            'content': content
        }

    def transfers(self, segment):
        rows = []

        departure_formatter = timetable.TimeWithDateIfChangedFormatter(self.base) if self.base.context.date else timetable.TimeOnlyFormatter(self.base)
        arrival_formatter = timetable.TimeWithDateIfChangedFormatter(self.base) if self.base.context.date else timetable.TimeOnlyFormatter(self.base)

        convenience = segment.convenience and ', %s' % segment.convenience or ''

        for i, segment in enumerate(segment.segments):
            point_from = segment.station_from.settlement or segment.station_from
            point_to = segment.station_to.settlement or segment.station_to

            if point_from == point_to:
                point_from = segment.station_from
                point_to = segment.station_to

            rows.append({
                'elem': 'row',
                'elemMods': {
                    'type': 'trip',
                    'gone': segment.gone and 'yes' or None,
                    'has-number': 'yes' if self.show_number and segment_strong_number(segment) else 'no'
                },
                'content': [
                    self.show_t_ico and {
                        'elem': 'cell',
                        'elemMods': { 'type': 'ico' },
                        'content': self.ico(segment.display_t_code)
                    },
                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'trip' },
                        'content': [
                            self.transfers_tripname(segment),
                            self.show_desc and self.description(segment),
                        ]
                    },
                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'departure' },
                        'content': [
                            segment.departure and departure_formatter(segment.departure),
                            self.point(segment.station_from, segment.rtstation_from, force_station=True),
                        ]
                    },
                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'arrival' },
                        'content': [
                            segment.arrival and arrival_formatter(segment.arrival),
                            self.point(segment.station_to, segment.rtstation_to, force_station=True),
                        ]
                    },
                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'duration' },
                        'content': self.duration_with_express(segment)
                    },

                    self.show_stops and self.stops(segment),

                    self.show_tariffs and {
                        'elem': 'cell',
                        'elemMods': { 'type': 'tariff' },
                        'content': { 'elem': 'gone', 'content': gettext('ушёл') } if segment.gone else self.tariffs(segment=segment)
                    }
                ]
            })

            transfer = segment.display_info.get('transfer')

            if transfer:
                point = transfer['in']
                station_from = transfer.get('from')
                station_to = transfer.get('to')

                content = ''
                if settings.SHOW_TRANSFER_PRICES:
                    price = transfer.get('price')
                    convenience = transfer.get('convenience')
                else:
                    price = convenience = None

                if price:
                    content = 'ц. %s' % price

                if convenience:
                    if price:
                        content += ', у. %s' % convenience
                    else:
                        content = 'у. %s' % convenience

                if get_language() == 'ru':
                    title = xformat('<strong>Пересадка</strong> <point-title case="phrase_in"/>', point_title=point.L_title)

                else:
                    title = xgettext('<strong>Пересадка</strong>: <point-title/>', point_title=point.L_title)

                description = None

                if station_from:
                    if station_to and station_to != station_from:
                        description = [
                            station_from.L_popular_title() or station_from.L_title(),
                            ' → ',
                            station_to.L_popular_title() or station_to.L_title(),
                        ]
                    else:
                        description = station_from.L_popular_title() or station_from.L_title()

                rows.append({
                    'elem': 'row',
                    'elemMods': { 'type': 'trans-type' },
                    'content': [

                        self.show_t_ico and { 'elem': 'cell' },

                        {
                            'elem': 'cell',
                            'attrs': { 'colspan': '3' },
                            'content': [
                                {
                                    'elem': 'trans',
                                    'content': [
                                        title,
                                        description and [' (', description, ')']
                                    ]
                                },

                                {
                                    'elem': 'cell', 'elemMods': { 'type': 'duration' },
                                    'content': self.pathtime(transfer['duration'])
                                },

                                self.show_stops and {
                                    'elem': 'cell',
                                    'elemMods': { 'type': 'station' },
                                },

                                self.show_tariffs and { 'elem': 'cell', 'elemMods': { 'type': 'price' }, 'content': content },

                            ]
                        },

                    ]
                })

        return {
            'elem': 'groupdec',
            'content': {
                'block': 'b-timetable',
                'js': True,
                'content': rows
            }
        }

    def interval_sort_key(self, s):
        thread = s.thread

        return (
            thread.begin_time is None,
            thread.begin_time,
            thread.end_time is None,
            thread.end_time
        )

    def interval_rows(self):
        json_rows = []

        filters = self.context.filters

        segments = sorted(self.interval_segments, key=self.interval_sort_key)

        visible_rows_count = 0

        for segment in segments:
            hidden = filters.is_filtered(segment) or segment.gone and self.hide_gone

            if not hidden:
                visible_rows_count += 1

            comment, trigger = self.comment(segment, hidden, gettext('Время отправления'))

            mods = { 'group': 'bus', 'fixed': 'yes' }

            if segment.gone:
                mods['gone'] = 'yes'

            json_rows.append({
                'block': 'b-timetable',
                'elem': 'row',
                'mix': [hidden and { 'block': 'i-hidden' } ],
                'elemMods': mods,
                'js': {
                    'filter-values': filters.filter_values(segment)
                },
                'content': [

                    self.show_t_ico and self.ico_cell(self.ico(segment.display_t_code)),

                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'trip' },
                        'content': [
                            self.tripname(segment),
                            self.show_desc and self.description(segment, hidden),
                        ]
                    },

                    self.sort_row_cell('departure', [

                        {
                            'elem': 'time',
                            'content': [
                                {
                                    'tag': 'strong',
                                    'content': segment.thread.begin_time.strftime('%H:%M')
                                },
                                '...',
                                {
                                    'tag': 'strong',
                                    'content': segment.thread.end_time.strftime('%H:%M')
                                }
                            ]
                        },

                        {
                            'elem': 'repeat',
                            'content': segment.thread.density
                        },

                        self.point(segment.station_from, segment.rtstation_from),

                    ]),

                    self.sort_row_cell('arrival', [

                        self.point(segment.station_to, segment.rtstation_to),

                    ]),

                    self.sort_row_cell('duration', [

                        self.pathtime(segment.duration),
                        segment.thread and self.express(segment.thread),

                    ], timedelta2minutes(segment.duration) if segment.duration else None),

                    self.show_stops and self.stops(segment),

                    self.show_days and {
                        'elem': 'cell',
                        'content': [
                            self.days(segment),
                            comment and not self.show_tariffs and trigger,
                        ]
                    },

                    self.show_tariffs and self.sort_row_cell('tariff', [

                        { 'elem': 'gone', 'content': gettext('ушёл') } if segment.gone else self.tariffs(segment=segment),

                        trigger

                    ])

                ]
            })

            if comment:
                json_rows.append(comment)

        return json_rows, visible_rows_count

    def comment(self, segment, hidden, title=None):
        comment = headnonzerononfail(lambda: segment.thread.comment)

        if not comment:
            return None, None

        key = unicode(segment.thread.id)

        show_details, trigger = self.details(key, gettext('Скрыть'), gettext('Подробности'), 'times')

        return {
            'elem': 'row',
            'elemMods': {
                'type': 'bus-schedule',
                'key': key,
                'details': 'yes',
                'visibility': not show_details and 'hidden'
            },
            'mix': [hidden and { 'block': 'i-hidden' }],
            'content': {
                'elem': 'cell',
                'attrs': { 'colspan': '6' },
                'content': {
                    'block': 'b-bus-schedule',
                    'content': [
                        title and { 'elem': 'title', 'content': title },
                        { 'elem': 'content', 'content': comment },
                    ]
                }
            }
        }, trigger

    def delay(self, segment, event):
        z_tablo = getattr(segment, event + '_z_tablo', None)

        if not z_tablo:
            return None

        comment = z_tablo.L_comment()

        if getattr(z_tablo, event + '_cancelled'):
            return {
                'elem': 'delay',
                'content': [
                    {
                        'elem': 'delay-content',
                        'elemMods': {'type': 'cancel'},
                        'content': tgettext('Отмена')
                    },
                    {
                        'elem': 'delay-comment',
                        'content': comment
                    } if comment else None
                ]
            }

        time = getattr(z_tablo, event)
        real_time = getattr(z_tablo, 'real_' + event)

        if not time or not real_time:
            return None

        delta = real_time - time
        minutes = int(delta.total_seconds() / 60)

        if minutes == 0:
            return None

        if minutes > 0:
            delay_content = tngettext(minutes,
                                      'Задержка на <n /> минуту',
                                      'Задержка на <n /> минуты',
                                      'Задержка на <n /> минут')

        elif minutes < 0:
            delay_content = tngettext(-minutes,
                                      'Раньше на <n /> минуту',
                                      'Раньше на <n /> минуты',
                                      'Раньше на <n /> минут')

        return {
            'elem': 'delay',
            'content': [
                {
                    'elem': 'delay-content',
                    'elemMods': {'type': 'delay'},
                    'content': delay_content
                },
                {
                    'elem': 'delay-comment',
                    'content': comment
                } if comment else None
            ]
        }

    def __json__(self):
        notes = [
            ('bottom-return-link', self.base.return_link()),
            ('filtered-out', gettext('К сожалению, ни один из рейсов не подходит к выбранным настройкам фильтров. Попробуйте выбрать другие настройки фильтров или начните новый поиск.')),
            ('add-transfers', { 'block': 'b-link', 'url': self.base.change_params(transfers='add'), 'content': gettext('Добавить рейсы с пересадками') }),
            ('e-ticket', [self.base.b_icon_type_eticket(), ' ',
                          StaticText.get_text_by_code('e-ticket')]),
        ]

        mods = { 'sort': 'yes', 'filter': 'yes', 'preset': self.preset }

        if self.context.ajax_tariffs_info:
            mods['ajax'] = 'yes'

        try:
            self.max_icons = max(len(s.transport_types) for s in self.plain_segments if hasattr(s, 'transport_types'))
        except ValueError:
            pass

        interval_rows, visible_interval_rows = self.interval_rows()

        plain_rows, visible_plain_rows = self.plain_rows()

        rows = interval_rows + plain_rows

        visible_rows_count = visible_interval_rows + visible_plain_rows

        if visible_rows_count > 4:
            self.visible_notes.add('bottom-return-link')

        if not visible_rows_count:
            self.visible_notes.add('filtered-out')

        if self.context.show_add_transfers:
            self.visible_notes.add('add-transfers')

        return [
            {
                'block': 'b-timetable',
                'mods': mods,
                'js': {
                    'ajax-info': self.context.ajax_tariffs_info,
                    'columns': self.sort.js(),
                    'mode': self.mode,
                    'blablacarPosition': BLABLACAR_POSITION,
                    'columnCount': self.column_count,
                    'transportTypes': [t_type.code for t_type in self.transport_types]
                },
                'content': self.hrows() + rows
            },
            {
                'block': 'b-static-text',
                'mods': { 'type': 'timetable-notes' },
                'mix': [not self.visible_notes and { 'block': 'i-hidden' } ] + [{ 'block': 'i-media-print'}],
                'content': [
                    {
                        'elem': 'note',
                        'tag': 'p',
                        'elemMods': {'type': code},
                        'mix': [code not in self.visible_notes and { 'block': 'i-hidden' } ],
                        'content': text
                    } for code, text in notes
                ]
            }
        ]

    @classmethod
    def ajax(cls, request, context, updates):
        t = cls(Base(request, context))

        plain_rows, visible = t.plain_rows()

        return {
            'segments': plain_rows,
            'updates': dict(
                (key, {
                    'et-marker': cooked.et_marker,
                    'places': t.tariffs_places(cooked.places),
                    'filter-values': {
                        'tickets': [p.name for p in cooked.places if p.name],
                        'seats': 'y' if cooked.places else 'n',
                    }
                })

                for key, cooked in updates.items()
             )
        }


class NoRoutesHints(Base):
    page = 'search'

    def page_attrs(self):
        request = self.context.request

        key = ' '.join(
            "%s=%s" % (name, request.GET.get(name, ''))
            for name in ['fromId', 'fromName', 'toId', 'toName', 'when']
        )

        return {
            'onload': self.metrika_reach_goal("no_routes_hints_page_load", {
                'search-params-key': request.path + ' ' + key
            })
        }

    def nearest_search_link(self, point, direction, title):
        point_from = self.context['from']
        point_to = self.context['to']
        when = self.context['date']

        if direction == 'from':
            point_from = point
        else:
            point_to = point

        return {
            'block': 'b-link',
            'url': search_url(point_from, point_to, when),
            'content': title
        }

    def nearest_cities_hint(self):
        if get_language() == 'ru':
            return self.nearest_cities_hint_ru()

        nearest_cities = self.context.nearest_cities
        point_from = self.context['from']
        point_to = self.context['to']

        def nearest_point_link(point, direction):
            return self.nearest_search_link(point, direction, point.L_title())

        if len(nearest_cities['from']) == 1 and not nearest_cities['to']:
            return xgettext(
                'Попробуйте использовать ближайший город отправления: <nearest-city-title/>.',
                nearest_city_title=nearest_point_link(nearest_cities['from'][0], 'from')
            )

        elif len(nearest_cities['to']) == 1 and not nearest_cities['from']:
            return xgettext(
                'Попробуйте использовать ближайший город назначения: <nearest-city-title/>.',
                nearest_city_title=nearest_point_link(nearest_cities['to'][0], 'to')
            )

        variants = []

        if nearest_cities['from']:
            for point in nearest_cities['from']:
                variants.append(self.nearest_search_link(point, 'from', [point.L_title(), '&nbsp;&mdash; ', point_to.L_title() ]))

        if nearest_cities['to']:
            for point in nearest_cities['to']:
                variants.append(self.nearest_search_link(point, 'to', [point_from.L_title(), '&nbsp;&mdash; ', point.L_title() ]))

        return xgettext(
            'Попробуйте использовать следующие варианты поиска: <variants/>.',
            variants=punctuate_content(variants)
        )

    def nearest_cities_hint_ru(self):
        context = self.context

        point_from = context['from']
        point_to = context['to']

        nearest_cities = self.context.nearest_cities

        def nearest_list(direction):
            points = nearest_cities[direction]

            return ru_human_list_ili([
                self.nearest_search_link(point, direction, point.L_title(case='genitive')) for point in points
            ])

        if nearest_cities['from'] and nearest_cities['to']:
            return xformat(
                'Попробуйте поискать от <point-from-title case="genitive"/> до <nearest-cities-to/>, или от <nearest-cities-from/> до <point-to-title case="genitive"/>.',
                point_from_title=point_from.L_title,
                point_to_title=point_to.L_title,
                nearest_cities_to=nearest_list('to'),
                nearest_cities_from=nearest_list('from'),
            )

        elif nearest_cities['from']:
            if len(nearest_cities['from']) == 1:
                return xformat(
                    'Попробуйте поискать маршрут от ближайшего к <point-from-title case="genitive"/> города — <nearest-city-from/>, а далее воспользоваться местным транспортом.',
                    point_from_title=point_from.L_title,
                    nearest_city_from=nearest_list('from')
                )
            else:
                return xformat(
                    'Попробуйте поискать маршрут от ближайших к <point-from-title case="genitive"/> городов — <nearest-cities-from/>, а далее воспользоваться местным транспортом.',
                    point_from_title=point_from.L_title,
                    nearest_cities_from=nearest_list('from')
                )

        elif nearest_cities['to']:
            if len(nearest_cities['to']) == 1:
                return xformat(
                    'Попробуйте поискать маршрут до ближайшего к <point-to-title case="dative"/> города — <nearest-city-to/>, а далее воспользоваться местным транспортом.',
                    point_to_title=point_to.L_title,
                    nearest_city_to=nearest_list('to'),
                )
            else:
                return xformat(
                    'Попробуйте поискать маршрут до ближайших к <point-to-title case="dative"/> городов — <nearest-cities-to/>, а далее воспользоваться местным транспортом.',
                    point_to_title=point_to.L_title,
                    nearest_cities_to=nearest_list('to')
                )

    def bus_station_incomplete_schedule_disclaimer(self):
        point_from = self.context['from']
        point_to = self.context['to']

        point_from_settlement = getattr(point_from, 'settlement', None)

        return StaticText.get_text_by_code(
            'bus_station_incomplete_schedule_disclaimer',
            city_title=point_from_settlement and point_from_settlement.L_title,
            station_title=point_from.L_title,
            point_from_title=self.format_point_title(point_from, point_to),
            point_to_title=self.format_point_title(point_to, point_from),
        )

    def settlement_incomplete_schedule_disclaimer(self):
        point_from = self.context['from']
        point_to = self.context['to']

        return StaticText.get_text_by_code(
            'settlement_incomplete_schedule_disclaimer',
            city_title=point_from.L_title,
            point_from_title=self.format_point_title(point_from, point_to),
            point_to_title=self.format_point_title(point_to, point_from),
        )

    def any_day_any_transport_hint(self):
        context = self.context

        urls = {
            'any_day': self.change_params(when=gettext('на все дни')),
            'any_transport': '/search/?' + context.request.GET.urlencode()
        }

        kwargs = {
            'b_link': lambda content, url: {
                'block': 'b-link',
                'url': urls[url],
                'content': content
            },
        }

        if context.date:
            return xgettext('Попробуйте поискать <b-link url="any_day">на все дни</b-link>, <b-link url="any_transport">любым транспортом</b-link>, или выберите другой пункт назначения.', **kwargs)

        return xgettext('Попробуйте поискать <b-link url="any_transport">любым транспортом</b-link>, или выберите другой пункт назначения.', **kwargs)

    def kwargs(self):
        context = self.context

        return {
            'point_from_title': self.format_point_title(context['from'], context['to']),
            'point_to_title': self.format_point_title(context['to'], context['from']),
            'date': context.date and DateTimeFormatter(context.date).L
        }

    def format_point_title(self, point1, point2):
        """
        RASP-13909 При одноименности пунктов надо в сообщении в результатах поиска выводить не только названия,
        но и префиксы типов
        """
        if normalize(hasattr(point1, 'L_title_with_prefix') and point1.L_title()) == normalize(point2.L_title()):
            return point1.L_title_with_prefix()

        return point1.L_title

    def no_routes_message(self):
        context = self.context

        kwargs = self.kwargs()

        if context.date:
            if context.search_type == 'plane':
                return xgettext('К сожалению, у Яндекс.Расписаний нет данных об авиарейсах <point-from-title/>&nbsp;&mdash; <point-to-title/> на дату <date format="%d %B"/>.', **kwargs)

            elif context.search_type == 'train':
                return xgettext('К сожалению, у Яндекс.Расписаний нет данных о поездах <point-from-title/>&nbsp;&mdash; <point-to-title/> на дату <date format="%d %B"/>.', **kwargs)

            elif context.search_type == 'suburban':
                return xgettext('К сожалению, у Яндекс.Расписаний нет данных о пригородных поездах <point-from-title/>&nbsp;&mdash; <point-to-title/> на дату <date format="%d %B"/>.', **kwargs)
            elif context.search_type == 'bus':
                return xgettext('К сожалению, у Яндекс.Расписаний нет данных об автобусах <point-from-title/>&nbsp;&mdash; <point-to-title/> на дату <date format="%d %B"/>.', **kwargs)

            return xgettext('К сожалению, у Яндекс.Расписаний нет данных о рейсах <point-from-title/>&nbsp;&mdash; <point-to-title/> на дату <date format="%d %B"/>.', **kwargs)

        if context.search_type == 'plane':
            return xgettext('К сожалению, у Яндекс.Расписаний нет данных об авиарейсах <point-from-title/>&nbsp;&mdash; <point-to-title/>.', **kwargs)

        elif context.search_type == 'train':
            return xgettext('К сожалению, у Яндекс.Расписаний нет данных о поездах <point-from-title/>&nbsp;&mdash; <point-to-title/>.', **kwargs)

        elif context.search_type == 'suburban':
            return xgettext('К сожалению, у Яндекс.Расписаний нет данных о пригородных поездах <point-from-title/>&nbsp;&mdash; <point-to-title/>.', **kwargs)

        elif context.search_type == 'bus':
            return xgettext('К сожалению, у Яндекс.Расписаний нет данных об автобусах <point-from-title/>&nbsp;&mdash; <point-to-title/>.', **kwargs)

        return xgettext('К сожалению, у Яндекс.Расписаний нет данных о рейсах <point-from-title/>&nbsp;&mdash; <point-to-title/>.', **kwargs)

    def no_direct_routes_message(self):
        context = self.context

        kwargs = self.kwargs()

        if context.search_type == 'plane':
            return xgettext('К сожалению, у Яндекс.Расписаний нет данных о прямых авиарейсах <point-from-title/>&nbsp;&mdash; <point-to-title/>.', **kwargs)

        elif context.search_type == 'train':
            return xgettext('К сожалению, у Яндекс.Расписаний нет данных о прямых поездах <point-from-title/>&nbsp;&mdash; <point-to-title/>.', **kwargs)

        elif context.search_type == 'suburban':
            return xgettext('К сожалению, у Яндекс.Расписаний нет данных о прямых пригородных поездах <point-from-title/>&nbsp;&mdash; <point-to-title/>.', **kwargs)

        elif context.search_type == 'bus':
            return xgettext('К сожалению, у Яндекс.Расписаний нет данных о прямых автобусах <point-from-title/>&nbsp;&mdash; <point-to-title/>.', **kwargs)

        return xgettext('К сожалению, у Яндекс.Расписаний нет данных о прямых рейсах <point-from-title/>&nbsp;&mdash; <point-to-title/>.', **kwargs)

    def no_routes_message_and_hints(self):
        context = self.context

        transfers_hint = gettext('Для подбора вариантов проезда с пересадками укажите конкретную дату выезда.')

        if context.search_type:
            if context.date or context.search_type == 'suburban':
                return [self.no_routes_message(), ' ', self.any_day_any_transport_hint() ]

            else:
                if context.incomplete_schedule and context.search_type == 'bus':
                    if context.from_is_station:
                        return self.bus_station_incomplete_schedule_disclaimer()
                    else:
                        return self.settlement_incomplete_schedule_disclaimer()
                else:
                    return punctuate_content([
                        self.no_direct_routes_message(),
                        self.any_day_any_transport_hint(),
                        transfers_hint,
                    ], sep=' ')
        else:
            if context.date:
                if context.any_routes:
                    return punctuate_content([
                        self.no_routes_message(),
                        xgettext(
                            'Попробуйте поискать <b-link>на все дни</b-link>, или выберите другой пункт назначения.',
                            b_link=lambda content: {
                                'block': 'b-link',
                                'url': self.change_params(when=gettext('на все дни')),
                                'content': content,
                            }
                        )
                    ], sep=' ')

                else:
                    return punctuate_content([
                        self.no_routes_message(),
                        gettext('Попробуйте выбрать другой пункт назначения.')
                    ], sep=' ')

            else:
                if context.incomplete_schedule:
                    if context.from_is_station:
                        return self.bus_station_incomplete_schedule_disclaimer()

                    else:
                        return self.settlement_incomplete_schedule_disclaimer()

                elif context.search_in_same_city: # RASP-11184
                    return punctuate_content([
                        gettext('К сожалению, сообщения между запрошенными вами станциями не найдено. Попробуйте использовать городской транспорт или уточните ваш запрос.'),
                        transfers_hint,
                    ], sep=' ')

                else:
                    return punctuate_content([
                        self.no_direct_routes_message(),
                        gettext('Попробуйте выбрать другой пункт назначения.'),
                        transfers_hint,
                    ], sep=' ')

    def content(self):
        if self.context.nearest_cities:
            message = [
                xgettext(
                    'К сожалению, у Яндекс.Расписаний нет данных о рейсах <point-from-title/> — <point-to-title/>.',
                    point_from_title=self.context['from'].L_title, point_to_title=self.context['to'].L_title
                ),
                ' ',
                self.nearest_cities_hint()
            ]

            # title = u"%s &mdash; %s: ближайшие города" % (hidden_search_form.cleaned_data['from'].point.title,
            #                                               hidden_search_form.cleaned_data['to'].point.title)
        else:
            message = self.no_routes_message_and_hints()

        return [
            {
                'block': 'l-page',
                'mods': { 'layout': '72-20', 'row': 'multiple' },
                'content': [
                    {
                        'elem': 'row',
                        'content': [
                            {
                                'elem': 'gap'
                            },
                            {
                                'elem': 'left',
                                'content': {
                                    'block': 'b-page-title',
                                    'content': [
                                        {
                                            'elem': 'title',
                                            'content': self.context.page_title,
                                        },
                                        self.context.segments and {
                                            'elem': 'return-trip',
                                            'content': [
                                                self.return_link(),
                                            ]
                                        }
                                    ]
                                }
                            },
                            { 'elem': 'gap-right'},
                            { 'elem': 'right' },
                            { 'elem': 'gap' },
                        ]
                    },
                    {
                        'elem': 'row',
                        'content': [
                            { 'elem': 'gap' },
                            {
                                'elem': 'left',
                                'content': {
                                    'block': 'b-static-text',
                                    'content': {
                                        'tag': 'p',
                                        'content': message
                                    }
                                }
                            },
                            { 'elem': 'gap-right'},
                            {
                                'elem': 'right',
                                'content': [
                                    self.teaser('ahtung')
                                ]
                            },
                            { 'elem': 'gap' },
                        ]
                    },
                ]
            }
        ]
