# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from itertools import cycle

from common.models.disclaimers import StaticText
from common.utils.date import DateTimeFormatter
from common.xgettext.i18n import gettext, xgettext

# Шаблоны
from common.utils.text import mdash_nowrap
from common.utils.bemhtml import space
from travel.rasp.morda.morda.templates.station.base import Template as StationBaseTemplate
from travel.rasp.morda.morda.utils.locations import station_thread_url
from travel.rasp.morda.morda.templates.timetable import TimeOnlyFormatter, TimeWithDateIfChangedFormatter


class Template(StationBaseTemplate):
    page = 'tstation'

    def b_timetable_direction(self):
        show_only_current_direction = self.context.get('print', False)
        show_all_directions = not show_only_current_direction

        def direction_link(d_code, title):
            current = d_code == self.context.direction_code
            url = not current and self.change_params(direction=d_code, subdir=None, title=None)

            return {
                'elem': 'item',
                'elemMods': {'state': current and 'current' or ''},
                'content': (
                    {
                        'block': 'b-link',
                        'url': url,
                        'mods': {'pass-params': 'yes'},
                        'content': title,
                    }
                    if url else
                    title
                )
            }

        return {
            'block': 'b-timetable-direction',
            'content': [
                {
                    'elem': 'cell',
                    'content': [
                        (
                            direction_link(d_code, title)
                            if (
                                show_all_directions or
                                (show_only_current_direction and d_code == self.context.direction_code)
                            ) else
                            ''
                        )

                        for d_code, title, _count in row
                        if d_code
                    ]
                } for row in self.context.directions
            ]
        }

    def hrows(self):
        context = self.context
        station = context.station

        return [
            {
                'elem': 'hrow',
                'content': [
                    { 'elem': 'cell' },
                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'timeselector' },
                        'content': {
                            'block': 'b-timeselector',
                            'mods': { 'type': 'header' },
                            'content': [
                                {
                                    'elem': 'selector',
                                    'content': [
                                        {
                                            'block': 'b-icon'
                                        },
                                        self.timeselector(),
                                    ]
                                },
                            ]
                        }
                    },
                    context.type != 'aeroex' and { 'elem': 'cell' } or '',
                    not context.date and { 'elem': 'cell' } or ''
                ]
            },
            {
                'elem': 'hrow',
                'elemMods': { 'position': 'last' },
                'content': [
                self.hrow_cell(column, name) for column, name in filter(None, [
                        ('trip', gettext('рейс')),
                        ('name', station.L_short_title_with_prefix()),
                        ('station', gettext('остановки')),
                        not context.date and ('days', gettext('дни курсирования')),
                    ])
                ]
            }
        ]

    def rows(self):
        context = self.context
        station = context.station
        schedule = context.schedule
        mark_cycle = cycle([None, 'mark'])
        departure_formatter = TimeWithDateIfChangedFormatter(self) if self.context.date else TimeOnlyFormatter(self)

        if schedule:
            json_rows = []
            for j, schedule_route in enumerate(schedule):
                loop_last = j == len(schedule) - 1
                mark_class = mark_cycle.next()

                thread_url = station_thread_url(schedule_route.thread, schedule_route.start_date, self.context.request)
                thread_link_content = mdash_nowrap(schedule_route.title)

                row = {
                    'elem': 'row',
                    'mods': {
                        'mark': context.get('print', False) and mark_class or '',
                        'type': schedule_route.cancel and 'canceled' or '',
                        'position': loop_last and 'last',
                        },
                    'content': [
                        {
                            'elem': 'cell',
                            'elemMods': { 'type': 'trip' },
                            'content': [
                                {
                                    'elem': 'tripname',
                                    'content': [
                                        context.get('print', False) and [
                                            mdash_nowrap(schedule_route.thread.title_short)
                                        ]
                                        or
                                        {
                                            'block': 'b-link',
                                            'url': thread_url,
                                            'mods': {},
                                            'content': thread_link_content,
                                        },
                                        schedule_route.terminal and {
                                            'elem': 'nostop',
                                            'content': [' ', gettext('посадки нет')],
                                        }
                                    ]
                                },
                                {
                                    'elem': 'description',
                                    'content': getattr(schedule_route, 'ticket_number', '')
                                },
                            ]
                        },
                        {
                            'elem': 'cell',
                            'elemMods': { 'type': 'departure', 'state': 'current' },
                            'content': {
                                'elem': 'time',
                                'content': [
                                    departure_formatter(schedule_route.event_dt),
                                    (schedule_route.thread and (
                                        schedule_route.thread.express_lite and {
                                            'elem': 'express',
                                            'attrs': { 'style': 'color: %s' % schedule_route.thread.express_lite.color },
                                            'content': schedule_route.thread.express_lite.title
                                        } or \
                                        schedule_route.thread.is_express and { 'elem': 'express', 'content': gettext('экспресс') } or \
                                        schedule_route.thread.is_aeroexpress and { 'elem': 'express', 'content': gettext('аэроэкспресс') } or ''
                                    ) or ''),

                                    schedule_route.rtstation.L_platform() and {
                                        'elem': 'platform',
                                        'content': self.station_schema_or_platform(schedule_route.rtstation)
                                    } or ''
                                ]
                            }
                        },
                        context.type != 'aeroex' and {
                            'elem': 'cell',
                            'elemMods': {'type': 'station'},
                            'content': {
                                'elem': 'stations',
                                'content': schedule_route.schedule and schedule_route.schedule.L_stops()
                            }
                        },
                        not context.date and {
                            'elem': 'cell',
                            'elemMods': {'type': 'days'},
                            'content': self.days_text(schedule_route),
                        } or '',
                    ]
                }
                json_rows.append(row)

            return json_rows

        else:
            direction_code = context.direction_code

            # TODO адаптировать параметр direction=arrival в электричках для турецкой версии
            kwargs = {
                'date': self.context.date and DateTimeFormatter(self.context.date).L,
                'b_link': lambda content, url: {
                    'block': 'b-link',
                    'url': {
                        'all_days': self.station_url(span='schedule', type='suburban', direction=direction_code),
                        'trains': self.station_url(type='train', span=(context.span if context.date else 'schedule')),
                    }[url],
                    'content': content
                }
            }

            if context.date:
                no_departure = xgettext('Нет информации о пригородных поездах (электричках), отправляющихся <date format="%d %B"/>.', **kwargs)
                no_arrival = xgettext('Нет информации о пригородных поездах (электричках), прибывающих <date format="%d %B"/>.', **kwargs)

                all_days = xgettext('Попробуйте посмотреть расписание <b-link url="all_days">на все дни</b-link>.', **kwargs)

                if len(context.view_types) == 1:
                    if direction_code != 'arrival':
                        content = space(no_departure, all_days)

                    else:
                        content = space(no_arrival, all_days)

                else:
                    all_days_and_trains = xgettext('Попробуйте посмотреть расписание <b-link url="all_days">на все дни</b-link> или <b-link url="trains">расписание поездов дальнего следования</b-link>.', **kwargs)

                    if direction_code != 'arrival':
                        content = space(no_departure, all_days_and_trains)

                    else:
                        content = space(no_arrival, all_days_and_trains)

            else:
                no_departure = gettext('Нет информации о пригородных поездах (электричках), отправляющихся с данной станции.')
                no_arrival = gettext('Нет информации о пригородных поездах (электричках), прибывающих на данную станцию.')

                if len(context.view_types) == 1:
                    if direction_code != 'arrival':
                        content = no_departure

                    else:
                        content = no_arrival

                else:
                    trains = xgettext('Попробуйте посмотреть <b-link>расписание поездов дальнего следования</b-link>.', b_link=lambda content: {
                        'block': 'b-link',
                        'url': self.station_url(type='train', span='schedule'),
                        'content': content
                    })

                    if direction_code != 'arrival':
                        content = space(no_departure, trains)

                    else:
                        content = space(no_arrival, trains)

            return [
                {
                    'elem': 'row',
                    'elemMods': { 'type': 'no-bottom' },
                    'content': [
                        {
                            'elem': 'cell',
                            'attrs': { 'colspan': '3' if self.context.date else '4' },
                            'content': [
                                '<!-- noindex -->',
                                '<br/>',
                                {
                                    'block': 'b-static-text',
                                    'mix': [{ 'block': 'b-notice' }],
                                    'content': content
                                },
                                '<br/>',
                                '<!-- /noindex -->',
                                ]
                        }
                    ]
                }
            ]

    def b_timetable(self):
        return {
            'block': 'b-timetable',
            'js': {
                'mode': not self.context.date and 'schedule' or ''
            },
            'content': self.hrows() + self.rows()
        }

    def content(self):
        station = self.context.station

        return [
            {
                'block': 'l-page',
                'mods': { 'layout': '72-20', 'row': 'multiple' },
                'content': [
                    self.content_first_row(),
                    {
                        'elem': 'row',
                        'content': [
                            {
                                'elem': 'gap'
                            },
                            {
                                'elem': 'left',
                                'content': [
                                    {
                                        'block': 'b-tablo-triggers',
                                        'content': [
                                            self.b_datelinks(),
                                            self.b_triggers(),
                                        ]
                                    },
                                    {
                                        'block': 'b-layout-table',
                                        'mods': { 'layout': 'tsearch' },
                                        'content': [
                                            {
                                                'elem': 'row',
                                                'content': [
                                                    {
                                                        'elem': 'cell',
                                                        'elemMods': { 'type': 'tabs' },
                                                        'content': self.b_timetable_direction()
                                                    },
                                                ]
                                            }
                                        ]
                                    },
                                    self.b_timetable(),
                                    {
                                        'block': 'b-description-text',
                                        'content': [
                                            {
                                                'elem': 'item',
                                                'content': StaticText.get_text_by_code(
                                                    'suburban_schedule_disclaimer',
                                                    station_title=station.L_popular_title_extra,
                                                    date=DateTimeFormatter(self.context.today).L
                                                )
                                            }
                                        ]
                                    },
                                    {
                                        'elem': 'item',
                                        'cls': 'i-media-print',
                                        'content': self.b_link_place_on_my_site()
                                    },
                                    '<img src="//clck.yandex.ru/click/dtype=stred/pid=168/cid=70208/*http://ya.ru" alt="" />'
                                ]
                            },
                            {
                                'elem': 'gap-right'
                            },
                            {
                                'elem': 'right',
                                'content': self.right_content(),
                            },
                            {
                                'elem': 'gap'
                            }
                        ]
                    }
                ]
            }
        ]
