# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from itertools import cycle

from common.models.disclaimers import StaticText
from common.utils.bemhtml import punctuate_content
from common.utils.date import DateTimeFormatter
from common.utils.locations import composeurl
from common.utils.text import mdash_wrappable
from common.xgettext.i18n import gettext, xgettext

# Шаблоны
from base import Template as Base
from travel.rasp.morda.morda.templates.timetable import (TimeOnlyFormatter, TimeWithDateIfChangedFormatter,
                                       segment_description)


class Template(Base):
    page = 'tstation'

    def hrows(self):
        context = self.context
        station = context.station

        return [
            {
                'elem': 'hrow',
                'content': [
                    { 'elem': 'cell' },
                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'timeselector' },
                        'content': {
                            'block': 'b-timeselector',
                            'mods': { 'type': 'header' },
                            'content': [
                                {
                                    'elem': 'selector',
                                    'content': [
                                        {
                                            'block': 'b-icon'
                                        },
                                        self.timeselector(),
                                    ]
                                },
                            ]
                        }
                    },
                    not context.date and { 'elem': 'cell' } or ''
                ]
            },
            {
                'elem': 'hrow',
                'elemMods': { 'position': 'last' },
                'content': [
                    self.hrow_cell(column, name) for column, name in filter(None, [
                        ('trip', gettext('рейс')),
                        ('name', station.L_short_title_with_prefix()),
                        not context.date and ('days', gettext('дни курсирования')),
                    ])
                ]
            }
        ]

    def rows(self):

        station = self.context.station
        schedule = self.context.schedule
        mark_cycle = cycle([None, 'mark'])

        if schedule:
            time_formatter = TimeWithDateIfChangedFormatter(self) if self.context.date else TimeOnlyFormatter(self)

            json_rows = []

            for j, schedule_route in enumerate(schedule):
                loop_last = j == len(schedule) - 1
                mark_class = mark_cycle.next()

                params = {}

                if not self.context.request.yandex_bot:
                    if schedule_route.start_date:
                        params['departure'] = str(schedule_route.start_date)

                thread_url = composeurl('thread', args=[schedule_route.thread.uid], params=params)

                row = {
                    'elem': 'row',
                    'mods': {
                        'mark': self.context.get('print', False) and mark_class or '',
                        'type': schedule_route.cancel and 'canceled' or '',
                        'position': loop_last and 'last',
                    },
                    'content': [
                        {
                            'elem': 'cell',
                            'elemMods': { 'type': 'trip' },
                            'content': [
                                {
                                    'elem': 'tripname',
                                    'content': [
                                        self.context.get('print', False) and [
                                            schedule_route.thread.number, ' ', mdash_wrappable(schedule_route.thread.title_short)
                                        ],
                                        ' ',
                                        {
                                            'block': 'b-link',
                                            'url': thread_url,
                                            'content': [
                                                {
                                                    'elem': 'text',
                                                    'tag': 'strong',
                                                    'content': schedule_route.thread.number
                                                },
                                                ' ',
                                                mdash_wrappable(schedule_route.title)
                                            ],
                                        },
                                    ]
                                },
                                {
                                    'elem': 'description',
                                    'content': punctuate_content(segment_description(schedule_route))
                                },
                            ]
                        },
                        {
                            'elem': 'cell',
                            'elemMods': {'type': self.context.event},
                            'content': {
                                'elem': 'time',
                                'content': time_formatter(schedule_route.event_dt)
                            }
                        },
                        not self.context.date and {
                            'elem': 'cell',
                            'elemMods': {'type': 'days'},
                            'content': self.days_text(schedule_route),
                        } or '',
                    ]
                }
                json_rows.append(row)

            return json_rows
        else:
            event = self.context.event

            kwargs = {
                'date': self.context.date and DateTimeFormatter(self.context.date).L,
                'b_link': lambda content, url: {
                    'block': 'b-link',
                    'url': {
                        'all_days': self.station_url(span='schedule', type='train', event=event),
                        'suburban': self.station_url(type='suburban', span=self.context.span),
                    }[url],
                    'content': content
                }
            }

            if self.context.date:
                no_departure = xgettext('Нет информации о поездах, отправляющихся <date format="%d %B"/>.', **kwargs)
                no_arrival = xgettext('Нет информации о поездах, прибывающих <date format="%d %B"/>.', **kwargs)

                all_days = xgettext('Попробуйте посмотреть <b-link url="all_days">расписание на все дни</b-link>.', **kwargs)

                if len(self.context.view_types) == 1:
                    if event == 'departure':
                        content = punctuate_content([no_departure, all_days], sep=' ')

                    else:
                        content = punctuate_content([no_arrival, all_days], sep=' ')

                else:
                    all_days_and_suburban = xgettext('Попробуйте посмотреть расписание <b-link url="all_days">на все дни</b-link> или <b-link url="suburban">расписание пригородных поездов (электричек).</b-link>', **kwargs)

                    if event == 'departure':
                        content = punctuate_content([no_departure, all_days_and_suburban], sep= ' ')

                    else:
                        content = punctuate_content([no_arrival, all_days_and_suburban], sep=' ')

            else:
                no_departure = gettext('Нет информации о поездах, отправляющихся с данной станции.')
                no_arrival = gettext('Нет информации о поездах, прибывающих на данную станцию.')

                if len(self.context.view_types) == 1:
                    if event == 'departure':
                        content = no_departure

                    else:
                        content = no_arrival

                else:
                    suburban = xgettext('Попробуйте посмотреть <b-link>расписание пригородных поездов (электричек).</b-link>', b_link=lambda content: {
                        'block': 'b-link',
                        'url': self.station_url(type='suburban'),
                        'content': content,
                    })

                    if event == 'departure':
                        content = punctuate_content([no_departure, suburban], sep=' ')
                    else:
                        content = punctuate_content([no_arrival, suburban], sep=' ')

            return [
                {
                    'elem': 'row',
                    'elemMods': { 'type': 'no-bottom' },
                    'content': [
                        {
                            'elem': 'cell',
                            'attrs': { 'colspan': '2' if self.context.date else '3' },
                            'content': [
                                '<!-- noindex -->',
                                '<br/>',
                                {
                                    'block': 'b-static-text',
                                    'mix': [{ 'block': 'b-notice' }],
                                    'content': content
                                },
                                '<br/>',
                                '<!-- /noindex -->',
                            ]
                        }
                    ]
                }
            ]

    def b_timetable(self):
        return {
            'block': 'b-timetable',
            'mods': { 'preset': 'station' },
            'js': {
                'timecolumns': [self.context.event],
                'mode': not self.context.date and 'schedule' or ''
            },
            'content': self.hrows() + self.rows()
        }

    def content(self):
        station = self.context.station

        return [
            {
                'block': 'l-page',
                'mods': { 'layout': '72-20', 'row': 'multiple' },
                'content': [
                    self.content_first_row(),
                    {
                        'elem': 'row',
                        'content': [
                            {
                                'elem': 'gap'
                            },
                            {
                                'elem': 'left',
                                'content': [
                                    {
                                        'block': 'b-tablo-triggers',
                                        'content': [
                                            self.b_datelinks(),
                                            self.b_triggers(),
                                        ]
                                    },
                                    {
                                        'block': 'b-layout-table',
                                        'mods': { 'layout': 'tsearch' },
                                        'content': [
                                            {
                                                'elem': 'row',
                                                'content': [
                                                    {
                                                        'elem': 'cell',
                                                        'elemMods': { 'type': 'tabs' },
                                                        'content': {
                                                            'block': 'b-timetable-trigger',
                                                            'content': [
                                                                self.e_event_tab('departure', gettext('Отправление'), self.context.event),
                                                                self.e_event_tab('arrival', gettext('Прибытие'), self.context.event),
                                                            ]
                                                        }
                                                    },
                                                ]
                                            }
                                        ]
                                    },
                                    self.b_timetable(),
                                    {
                                        'block': 'b-description-text',
#                                        'mods': { 'type': 'gray' },
                                        'content': [
                                            {
                                                'elem': 'item',
                                                'content': StaticText.get_text_by_code('train_schedule_disclaimer', station_title=station.L_popular_title)
                                            }
                                        ]
                                    },
                                    {
                                        'elem': 'item',
                                        'cls': 'i-media-print',
                                        'content': self.b_link_place_on_my_site()
                                    },
                                    '<img src="//clck.yandex.ru/click/dtype=stred/pid=168/cid=70142/*http://yandex.ru" alt="" />'
                                ]
                            },
                            {
                                'elem': 'gap-right'
                            },
                            {
                                'elem': 'right',
                                'content': self.right_content(),
                            },
                            {
                                'elem': 'gap'
                            }
                        ]
                    }
                ]
            }
        ]
