# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import re

from datetime import timedelta

from common.models.transport import TransportType
from common.utils.bemhtml import loop_over, punctuate_content
from common.utils.date import uni_strftime, human_duration, DateTimeFormatter
from common.utils.locations import composeurl, langify
from common.utils.text import mdash_nowrap, mdash
from common.xgettext.i18n import gettext, xgettext, mark_gettext
from travel.rasp.morda.morda.templates.base import Base
from travel.rasp.morda.morda.templates.station_info import station_info
from travel.rasp.morda.morda.templates.timetable import TimeWithDateIfChangedFormatter, TimeOnlyFormatter
from travel.rasp.morda.morda.templates.utils import wrap_with_link, buy_link

## RASPFRONT-2420 - Список компаний-перевозчиков, для которых не нужно показывать легенду под календарем
HIDDEN_CALENDAR_LEGEND_BY_COMPANY = [59181]

mark_gettext('b-avia-disclaimer_ru')
mark_gettext('b-avia-disclaimer_ua')

def pathtime(duration):
    return duration and human_duration(duration) or '&nbsp;'


def widget_link(base):
    # RASP-12040
    if base.context.request.NATIONAL_VERSION != 'ru':
        return

    widget_link = base.context.widget_link
    widget_link_t_type = base.context.widget_link_t_type

    if not widget_link or not widget_link_t_type:
        return False

    return {
        'block': 'b-bwidget-link',
        'content': {
            'block': 'b-link',
            'url': widget_link,
            'content': [
                {
                    'block': 'b-icon',
                    'url': base.static_url('blocks-desktop/b-bwidget-link/totheblog_%s.png' % widget_link_t_type),
                },
                gettext('Расскажите друзьям о&#0160;своём путешествии')
            ]
        }
    }


class Template(Base):
    page = 'trip'

    def __init__(self, *args, **kwargs):
        super(Template, self).__init__(*args, **kwargs)

    def head_content_extended(self):
        return [
            self.maps_api_js_elem()
        ]

    def plane_price(self):
        display_info = self.context.display_info
        if self.context.thread.t_type.code != 'plane' or not display_info:
            return

        tariff = None
        partner_name = None
        deep_link = '#'
        state = 'loading'
        js = {
            'fetchUrl': self.context.plane_price_fetch_url
        }

        tariffs_info = display_info.tariffs_info
        if tariffs_info and tariffs_info.places:
            place = tariffs_info.places[0]
            tariff = place.tariff
            partner_name = place.partner.title
            deep_link = place.deep_link
            state = 'done'
            js = None

        return {
            'block': 'b-avia-thread-price',
            'mods': {
                'state': state
            },
            'js': js,

            'content': [
                state == 'loading' and {
                    'elem': 'state',
                    'elemMods': {
                        'type': 'loading'
                    },
                    'content': {
                        'block': 'b-spin',
                        'mods': {
                            'size': '45',
                            'theme': 'white-45',
                            'progress': 'yes'
                        }
                    }
                },
                {
                    'elem': 'state',
                    'elemMods': {
                        'type': 'done'
                    },
                    'content': [
                        {
                            'elem': 'left',
                            'content': [
                                {
                                    'tag': 'a',
                                    'elem': 'text',
                                    'attrs': {
                                        'href': deep_link,
                                        'target': '_blank'
                                    },
                                    'content': [
                                        {
                                            'elem': 'header',
                                            'content': gettext('лучшая цена')
                                        },
                                        self.b_currency(tariff, show_cents=False) if tariff else {
                                            'elem': 'price'
                                        }
                                    ],
                                },
                                {
                                    'elem': 'partner-name',
                                    'content': partner_name
                                }
                            ]
                        },
                        {
                            'elem': 'right',
                            'content': [
                                {
                                    'tag': 'a',
                                    'attrs': {
                                        'href': deep_link,
                                        'target': '_blank'
                                    },
                                    'elem': 'button',
                                    'content': {
                                        'elem': 'button-text',
                                        'content': gettext('Купить')
                                    }
                                },
                                {
                                    'elem': 'button-label',
                                    'content': gettext('реклама')
                                }
                            ]
                        }
                    ]
                }
            ]
        }

    def get_price_block(self):
        result = self.plane_price()
        if result:
            return result
        return self.b_bigprice()

    def b_bigprice(self):
        tariffs = []
        if self.context.tariffs:
            tariffs += [
                {
                    'elem': 'cell',
                    'content': [
                        {
                            'elem': 'price',
                            'content': {
                                'block': 'b-link',
                                'url': buy_link(self.context.request, link_params),
                                'content': self.b_currency(tariff, show_cents=False)
                            }
                        },
                        {
                            'elem': 'label',
                            'content': {
                                'elem': 'nowrap',
                                'content': '%s%s' % (label, (' ' + gettext('СВ')) if label == 'люкс' else '')
                            }
                        }
                    ]
                }
                for tariff, label, seats, link_params in self.context.tariffs
            ]

        if self.context.ticket_link:
            tariffs += [
                {
                    'elem': 'cell',
                    'content': {
                        'elem': 'link',
                        'url': self.context.ticket_link,
                        'content': xgettext(
                            '<nowrap>Посмотреть <strong>все цены</strong></nowrap> <nowrap>по направлению</nowrap> <nowrap>на <ticket-date/></nowrap>',
                            nowrap=lambda content: { 'elem': 'nowrap', 'content': content },
                            strong=lambda content: { 'block': 'text', 'tag': 'strong', 'content': content },
                            ticket_date=DateTimeFormatter(self.context.ticket_date).L,
                        )
                    }
                }
            ]

        return tariffs and {
            'block': 'b-bigprice',
            'content': {
                'elem': 'layout',
                'content': punctuate_content(tariffs, { 'elem': 'gap' })
            }
        }

    def b_info_company_link(self):
        company = self.context.thread.thread.company

        if not company:
            return ''

        if not company.strange and company.t_type and \
               company.t_type.code in ('train', 'plane', 'suburban'):
            return {
                'block': 'b-link',
                'url': composeurl('info_object',
                                  kwargs={'object_type': 'company', 'object_id': company.id}),
                'content': company.L_title(),
            }
        else:
            return company.L_title()

    def company_contact_info(self):
        company = self.context.thread.thread.company
        if company and not company.strange and company.t_type and company.t_type.code in ['bus', 'plane', 'water'] and \
                company.contact_info and company.contact_info.strip():
            return [', ', clean_contact_info(company.contact_info)]
        else:
            return ''

    def start_finish_markers(self, st):
        return [
            st.is_first and {
                'elem': 'ico',
                'elemMods': { 'type': 'start' },
            },

            st.is_last and {
                'elem': 'ico',
                'elemMods': { 'type': 'finish' },
            }
        ]

    def b_timetable_first_column(self, st):
        thread = self.context.thread.thread
        station_url = (not st.station.not_in_search() and st.station.interesting() and not st.station.hidden and
                       self.station_url(st.id))

        return {
            'elem': 'cell',
            'elemMods': {'type': 'trip'},
            'content': [

                self.start_finish_markers(st),

                {
                    'elem': 'city',
                    'elemMods': { 'type': st.is_technical_stop and 'tech' or None },
                    'content': [
                        wrap_with_link(st.station.L_railway_title(terminal=st.terminal, thread=thread),
                                       url=station_url),
                        st.l_platform and [
                            ', ',
                            {
                                'elem': 'platform',
                                'elemMods': { 'inline': 'yes' },
                                'content': self.station_schema_or_platform(st)
                            }
                        ],
                        st.is_technical_stop and '*' or '',
                        st.event_states and {
                            'elem': 'event-state',
                            'elemMods': {'status': st.event_states.status},
                            'content': st.event_states.message
                        }
                    ]
                },

                st.station.settlement and thread.t_type_id == TransportType.PLANE_ID and {
                    'elem': 'description',
                    'content': xgettext('г. <city-title/>', city_title=st.station.settlement.L_title),
                },

            ]
        }

    def b_timetable_row_pathtime_column(self, st):
        return {
            'elem': 'cell',
            'elemMods': {'type': 'time'},
            'content': '' if st.arrival == st.departure else {
                'elem': 'pathtime',
                'content': pathtime(st.duration)
            }
        }

    def b_timetable_row_elemMods(self, st, loop):
        return {
            'type': (st.is_first and 'start') or \
                    (st.is_last and 'end') or \
                    ('') or (st.get('class') == 'gray' and 'disable'),
            'position': loop.last and 'last' or ''
        }

    def b_timetable_row(self, stop, loop, arr_formatter, dep_formatter, time_formatter):
        arrival = stop.arrival and arr_formatter(stop.arrival) or ''
        departure = stop.departure and dep_formatter(stop.departure) or ''
        elemMods = self.b_timetable_row_elemMods(stop, loop)

        if self.context.short_path:
            message = None

            kwargs = {
                'arrival': {
                    'elem': 'time',
                    'content': arrival
                },
                'departure': {
                    'elem': 'time',
                    'content': departure
                }
            }

            if arrival and departure:
                message = xgettext('приб. <arrival/>, отпр. <departure/>', **kwargs)

            elif arrival:
                message = xgettext('приб. <arrival/>', **kwargs)

            elif departure:
                message = xgettext('отпр. <departure/>', **kwargs)

            return {
                'elem': 'row',
                'elemMods': elemMods,
                'content': [
                    self.b_timetable_first_column(stop),

                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'departure' },
                        'content': message
                    },

                    self.b_timetable_row_pathtime_column(stop),
                ]
            }
        elif stop.is_combined and not stop.is_first and not stop.is_last:
            return [
                {
                    'elem': 'row',
                    'elemMods': elemMods,
                    'content': [
                        self.b_timetable_first_column(stop),
                        {
                            'elem': 'cell',
                            'elemMods': {'type': 'arrival'},
                            'content': '' if stop.arrival == stop.departure else {
                                'elem': 'time',
                                'content': arrival
                            }
                        },
                        {
                            'elem': 'cell',
                            'elemMods': {'type': 'stand'},
                            'content': ''
                        },
                        {
                            'elem': 'cell',
                            'elemMods': {'type': 'departure'},
                            'content': ''
                        },
                        self.b_timetable_row_pathtime_column(stop),
                    ]
                },
                {
                    'elem': 'row',
                    'elemMods': { 'type': 'change' },
                    'content': {
                        'elem': 'cell',
                        'attrs': { 'colspan': '5' },
                        'content': gettext('пересадка')
                    }
                },
                {
                    'elem': 'row',
                    'elemMods': elemMods,
                    'content': [
                        self.b_timetable_first_column(stop),
                        {
                            'elem': 'cell',
                            'elemMods': {'type': 'arrival'},
                            'content': ''
                        },
                        {
                            'elem': 'cell',
                            'elemMods': {'type': 'stand'},
                            'content': ''
                        },
                        {
                            'elem': 'cell',
                            'elemMods': {'type': 'departure'},
                            'content': '' if stop.arrival == stop.departure else {
                                'elem': 'time',
                                'content': departure
                            }
                        },
                        {'elem': 'cell'}
                    ]
                }

            ]
        else:
            return {
                'elem': 'row',
                'elemMods': elemMods,
                'content': [
                    self.b_timetable_first_column(stop),
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'arrival'},
                        'content': '' if stop.arrival == stop.departure else
                        self.b_time(stop, 'arrival', arrival, time_formatter)
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'stand'},
                        'content': '-' if stop.arrival == stop.departure else {
                            'elem': 'pathtime',
                            'content': pathtime(stop.stop_time)
                        }
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'departure'},
                        'content': '' if stop.arrival == stop.departure else
                        self.b_time(stop, 'departure', departure, time_formatter)
                    },
                    self.b_timetable_row_pathtime_column(stop),
                ]
            }

    def b_time(self, stop, event, formatted_time, time_formatter):
        event_state = stop.event_states and getattr(stop.event_states, event)
        should_show_corrected_time = bool(event_state and event_state.status != 'ok' and event_state.minutes_from)
        result = [
            {
                'elem': 'time',
                'elemMods': {'delay': should_show_corrected_time},
                'content': formatted_time
            }
        ]
        if should_show_corrected_time:
            different_minutes = event_state.minutes_to and event_state.minutes_from != event_state.minutes_to
            delay_time = [time_formatter(getattr(stop, event) + timedelta(minutes=event_state.minutes_from))]
            if different_minutes:
                delay_time += ['...', time_formatter(getattr(stop, event) + timedelta(minutes=event_state.minutes_to))]
            result.append({
                'elem': 'time',
                'elemMods': {'delay-corrected': True},
                'content': delay_time
            })

        return result

    def b_timetable_short_head(self):
        return [
            {
                'elem': 'hrow',
                'elemMods': {'position': 'first'},
                'content': [
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'trip'},
                        'content': gettext('маршрут')
                    },
                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'timeselector' },
                        'content': {
                            'block': 'b-timeselector',
                            'mods': { 'type': 'one' },
                            'content': [
                                {
                                    'elem': 'selector',
                                    'content': [
                                        { 'block': 'b-icon' },
                                        self.timeselector(),
                                    ]
                                },
                            ]
                        }
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'time'},
                        'content': gettext('в пути')
                    }
                ]
            }
        ]

    def b_timetable_head(self):
        return self.context.short_path and self.b_timetable_short_head() or \
        [
            {
                'elem': 'hrow',
                'content': [
                    {
                        'elem': 'cell'
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'timeselector'},
                        'attrs': {'colspan': '2'},
                        'content': {
                            'block': 'b-timeselector',
                            'content': [
                                {
                                    'elem': 'selector',
                                    'content': [
                                        {
                                            'block': 'b-icon'
                                        },
                                        self.timeselector(),
                                    ]
                                },
                                { 'elem': 'line' }
                            ]
                        }
                    },
                    {
                        'elem': 'cell'
                    },
                    {
                        'elem': 'cell'
                    }
                ]
            },
            {
                'elem': 'hrow',
                'elemMods': {'position': 'first'},
                'content': [
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'trip'},
                        'content': gettext('маршрут')
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'arrival'},
                        'content': gettext('прибытие')
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'stand'},
                        'content': gettext('стоянка')
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'departure'},
                        'content': gettext('отправление')
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'time'},
                        'content': gettext('в пути')
                    }
                ]
            }
        ]

    def b_timetable(self):
        arr_formatter = TimeWithDateIfChangedFormatter(self, fuzzy_title=gettext(u'Примерное время прибытия'))
        dep_formatter = TimeWithDateIfChangedFormatter(self, fuzzy_title=gettext(u'Примерное время отправления'))
        time_formatter = TimeOnlyFormatter(self)

        content = self.b_timetable_head()
        for loop, st in loop_over(self.context.thread.stops):
            content.append(self.b_timetable_row(st, loop, arr_formatter, dep_formatter, time_formatter))

        mods = {}

        if self.context.thread.is_interval:
            mods['interval'] = 'yes'

        if self.context.thread.t_type.code == 'bus':
            mods['preset'] = 'bus'

        block = {
            'block': 'b-timetable',
            'mods': mods,
            'js': {
                'timecolumns': [ 'arrival', 'departure' ],
            },
            'content': content
        }

        return block

    def b_map(self):
        return {
            'block': 'b-map',
            'mods': { 'hide': 'yes', 'type': 'route' },
            'js': self.context.route_data,
        }

    def show_thread_selector(self):
        t_type_id = self.context['thread']['t_type'].id

        if t_type_id in TransportType.WATER_TTYPE_IDS:
            return False

        if t_type_id == TransportType.BUS_ID:
            return False

        company = self.context['thread']['thread'].company
        if company and company.id in HIDDEN_CALENDAR_LEGEND_BY_COMPANY:
            return False

        if self.context.request.NATIONAL_VERSION == 'tr':
            if t_type_id == TransportType.PLANE_ID:
                return True

            return False

        return True

    def legend_block(self):
        if not self.show_thread_selector():
            return None

        elems = [self.legend_block_elem(block, i) for i, block in enumerate(self.context.cal_legend_order)]

        elems = filter(lambda x: x, elems)

        return {
            'block': 'b-page-calendar-legend',
            'js': True,
            'content': [
                           {
                               'elem': 'head',
                               'content': gettext('Варианты расписания в&nbsp;разные дни')
                           } if len(elems) > 1 else None
                       ] + elems
        }

    def legend_block_elem(self, block, i):
        data = self.context.cal_legend_blocks[block]

        if block == 'cancel':
            if not data['text']:
                return

            return {
                'elem': 'range',
                'elemMods': {'type': data['type']},
                'content': [
                    {
                        'elem': 'title',
                        'content': data['title']
                    },
                    {
                        'elem': 'date',
                        'mix': [{
                            'block': 'i-date',
                            'js': self.context.cancel_days
                        }],
                        'content': data['text']
                    }
                ]
            }

        if not data['threads']:
            return

        dep = self.context['thread']['stops'][0]['departure']
        arr = self.context['thread']['stops'][-1]['arrival']
        thread_start_date = dep.astimezone(self.context['thread']['thread'].pytz).date()

        def _get_js_params(thread):
            params = {
                'local': thread.L_days_text(
                    self.context['shift'],
                    except_separator=", ",
                    html=False,
                    template_only=False,
                    show_all_days=True,
                    thread_start_date=thread_start_date,
                )
            }

            for city in self.context.tz_cities:
                local_date = dep.astimezone(city.pytz).date()
                shift = (local_date - thread_start_date).days

                params[city.id] = thread.L_days_text(
                    shift,
                    except_separator=", ",
                    html=False,
                    template_only=False,
                    show_all_days=True,
                    thread_start_date=thread_start_date
                )

            return params

        def get_localized_times(thread, tz_cities=None):
            dep_shifts = {}
            arr_shifts = {}

            if tz_cities:
                for city in tz_cities:
                    local_dep_date = dep.astimezone(city.pytz).date()
                    dep_shifts[city.id] = (local_dep_date - thread_start_date).days

                    local_arr_date = arr.astimezone(city.pytz).date()
                    arr_shifts[city.id] = (local_arr_date - thread_start_date).days

            return [
                {
                    'block': 'time',
                    'tag': 'span',
                    'mix': [{
                        'block': 'i-time',
                        'js': {
                            'fuzzy': False,
                            'local': thread.departure.strftime('%d %b %Y %H:%M:%S'),
                            'shifts': dep_shifts
                        }
                    }],
                    'content': thread.departure.strftime('%H:%M')
                },
                ' &ndash; ',
                {
                    'block': 'time',
                    'tag': 'span',
                    'mix': [{
                        'block': 'i-time',
                        'js': {
                            'fuzzy': False,
                            'local': thread.arrival.strftime('%d %b %Y %H:%M:%S'),
                            'shifts': arr_shifts
                        }
                    }],
                    'content': thread.arrival.strftime('%H:%M')
                }
            ]

        return {
            'elem': 'range',
            'elemMods': {'type': data['type'], 'state': not i and 'selected'},
            'content': [
                {
                    'elem': 'title',
                    'content': [
                        {
                            'block': 'b-icon',
                            'mix': [{'block': 'b-page-calendar-legend', 'elem': 'icon', 'elemMods': {'type': data['type']}}]
                        },
                        data['title']
                    ]
                }
            ] + [{
                'elem': 'item',
                'content': [
                    {
                        'elem': 'trip',
                        'content': {
                            'block': 'b-link',
                            'url': composeurl('thread', kwargs={'uid': thread.uid}),
                            'content': mdash(thread.L_title())
                        }
                    },
                    {
                        'elem': 'time',
                        'content': get_localized_times(thread, self.context.tz_cities)
                    } ,
                    {
                        'elem': 'date',
                        'mix': [{
                            'block': 'i-date',
                            'js': _get_js_params(thread)
                        }],
                        'content': thread.L_days_text(
                            self.context['shift'],
                            except_separator=", ",
                            html=False,
                            template_only=False,
                            show_all_days=True,
                            thread_start_date=thread_start_date
                        )
                    }
                ]
            } for thread in data['threads'] if thread]
        }

    def content(self):
        self.context.show_social_share = True

        if self.show_thread_selector():
            show_days = bool(self.context.calendar.template)
            show_groups = bool(self.context.calendar.groups)
            show_vertical_direct = True

        else:
            show_days = show_groups = show_vertical_direct = False

        dep = self.context['thread']['stops'][0]['departure']
        thread_start_date = dep.astimezone(self.context['thread']['thread'].pytz).date()

        caution = xgettext('Данный вариант расписания действует: &lt;strong&gt;<days/>&lt;/strong&gt;',
                           days=self.context['thread']['thread'].L_days_text(
                               self.context['shift'],
                               except_separator=", ",
                               html=False,
                               template_only=False,
                               next_plan=self.context['next_plan'],
                               show_all_days=True,
                               thread_start_date=thread_start_date
                           ))

        return [
            {
                'block': 'b-width',
                'content': [
                    {
                        'block': 'l-page',
                        'mods': { 'layout': '72-20', 'row': 'multiple' },
                        'content': [
                            {
                                'elem': 'row',
                                'content': [
                                    {
                                        'elem': 'gap'
                                    },
                                    {
                                        'elem': 'left',
                                        'content': [
                                            self.get_price_block(),
                                            {
                                                'block': 'b-page-title',
                                                'content': [
                                                    {
                                                        'elem': 'title',
                                                        'content': map(mdash_nowrap, self.context.title),
                                                    },
                                                    {
                                                        'elem': 'text',
                                                        'content': [
                                                            self.context.model and '%s, ' % self.context.model or '',
                                                            self.b_info_company_link(),
                                                            self.company_contact_info(),
                                                        ]
                                                    },
                                                    {
                                                        'elem': 'caution',
                                                        'content': caution
                                                    }
                                                ]
                                            }
                                        ]
                                    },
                                    {
                                        'elem': 'gap-right'
                                    },
                                    {
                                        'elem': 'right',
                                        'content': self.b_metalinks()
                                    },
                                    { 'elem': 'gap' },
                                ]
                            },
                            {
                                'elem': 'row',
                                'content': [
                                    {
                                        'elem': 'gap'
                                    },
                                    {
                                        'elem': 'left',
                                        'content': [
                                            {
                                                'block': 'b-change-tabs',
                                                'js': True,
                                                'content': [
                                                    self.context.route_data and {
                                                        'elem': 'tabs',
                                                        'content': [
                                                            {
                                                                'block': 'b-menu-horiz',
                                                                'js': False,
                                                                'mods': {'layout': 'complex', 'preset': 'trip-type'},
                                                                'content': [
                                                                    {
                                                                        'elem': 'title',
                                                                        'content': gettext('Маршрут&#0160;следования:')
                                                                    },
                                                                    {
                                                                        'elem': 'item',
                                                                        'mods': {'state': 'current'},
                                                                        'content': {
                                                                            'block': 'b-link',
                                                                            'mods': {'pseudo': 'yes', 'type': 'data'},
                                                                            'url': '#',
                                                                            'content': gettext('Таблица')
                                                                        }
                                                                    },
                                                                    {
                                                                        'elem': 'item',
                                                                        'content': {
                                                                            'block': 'b-link',
                                                                            'mods': {'pseudo': 'yes', 'type': 'map'},
                                                                            'url': '#',
                                                                            'content': gettext('Карта')
                                                                        }
                                                                    },
                                                                    {
                                                                        'elem': 'item',
                                                                        'mods': {'type': 'dates'},
                                                                        'content': {
                                                                            'elem': 'text',
                                                                            'content': ''
                                                                        }
                                                                    }
                                                                ]
                                                            }
                                                        ]
                                                    },
                                                    {
                                                        'elem': 'pane',
                                                        'elemMods': {'type': 'data'},
                                                        'content': [
                                                            self.b_timetable(),
                                                            {
                                                                'block': 'b-description-text',
                                                                'content': [
                                                                    self.context.technical_stops and {
                                                                        'elem': 'item',
                                                                        'content': gettext('*&#0160;&#8212; техническая остановка, купить билет до этой станции невозможно.')
                                                                    } or '',
                                                                ]
                                                            }
                                                        ]
                                                    },
                                                    self.context.route_data and {
                                                        'elem': 'pane',
                                                        'elemMods': {'type': 'map'},
                                                        'content': self.b_map()
                                                    } or ''
                                                ]
                                            },
                                            widget_link(self),
                                        ] + \
                                        [self.context.thread.t_type.code == 'plane' and {
                                            'block': 'b-avia-disclaimer',
                                            'content': gettext("b-avia-disclaimer_{}".format(
                                                self.context.request.NATIONAL_VERSION
                                            ))
                                        }] + \
                                        [station_info(self.context.request, station) for station in self.context.weather_stations] + \
                                        [self.context.counter and '<img src="//clck.yandex.ru/click/dtype=stred/pid=168/cid=%s/*http://ya.ru" />' % self.context.counter or ''] + \
                                        [
                                            self.b_route_statistics(),
                                            not show_vertical_direct and [self.b_flat_yadirect()]
                                        ]
                                    },
                                    {
                                        'elem': 'gap-right'
                                    },
                                    {
                                        'elem': 'right',
                                        'content': [
                                            self.b_calendar(True),

                                            self.legend_block(),

                                            show_days and {
                                                'elem': 'text',
                                                'content': {
                                                    'block': 'b-block',
                                                    'tag': 'p',
                                                    'content': [
                                                        gettext('Дни курсирования:'),
                                                        ' ',
                                                        {
                                                            'elem': 'caution-legend',
                                                            'tag': 'strong',
                                                            'content': self.context.calendar.template
                                                        }
                                                    ]
                                                }
                                            },
                                            show_groups and {
                                                'block': 'b-other-trip',
                                                'content': [
                                                    {
                                                        'elem': 'title',
                                                        'content': gettext('Другие рейсы с этим номером')
                                                    }
                                                ] + [
                                                    {
                                                        'elem': 'item',
                                                        'content': {
                                                            'block': 'b-link',
                                                            'url': fit_link(group.link, self.context.request),
                                                            'content': group.title
                                                        }
                                                    } for group in self.context.calendar.groups
                                                ]
                                            },
                                            self.teaser('ahtung'),
                                            show_vertical_direct and self.b_yadirect()
                                        ]
                                    },
                                    { 'elem': 'gap' },
                                ]
                            }
                        ]
                    }
                ]
            }
        ]

    def b_route_statistics(self):
        rtstations_cnt = len(self.context.thread.stops)
        return self.context.stats and {
            'block': 'b-stat-h',
            'content': [
                {
                    'elem': 'title',
                    'content': gettext('Статистика выполнения рейса')
                },
                {
                    'block': 'b-stat',
                    'content': [
                        {
                            'elem': 'colgroup',
                            'content': [
                                {'elem': 'col'},
                                {'elem': 'col'},
                                {'elem': 'col', 'elemMods': {'type': 'line'}}
                            ] + [
                                {'elem': 'col'},
                                {'elem': 'col'},
                                {'elem': 'col'},
                                {'elem': 'col', 'elemMods': {'type': 'line'}}
                            ] * (rtstations_cnt - 2)
                        },
                        {
                            'elem': 'hrow',
                            'content': [{
                                'elem': 'cell',
                                'elemMods': {'position': 'head', 'type': i in (0, rtstations_cnt - 1) and 'marked' or None},
                                'attrs': {'colspan': i in (0, rtstations_cnt - 1) and 2 or 4},
                                'content': [
                                    {
                                        'elem': 'station',
                                        'content': {
                                            'block': 'b-link',
                                            'url': composeurl('station', kwargs={'station_id': stop.station.id}),
                                            'content': stop.station.L_title()
                                        }
                                    },
                                    stop.station.settlement and {
                                        'elem': 'city',
                                        'content': xgettext('г. <title/>', title=stop.station.settlement.L_title)
                                    }
                                ]
                            } for i, stop in enumerate(self.context.thread.stops)]
                        },
                        {
                            'elem': 'row',
                            'content': [
                                {
                                    'elem': 'cell',
                                    'elemMods': {'type': 'dir'},
                                    'attrs': {'colspan': 2},
                                    'content': [
                                        gettext('Отпр. '),
                                        '&nbsp;',
                                        {
                                            'block': 'b-icon',
                                            'mods': {'type': 'arrow'}
                                        }
                                    ]
                                },
                                {
                                    'elem': 'cell',
                                    'elemMods': {'type': 'dir'},
                                    'attrs': {'colspan': 2},
                                    'content': [
                                        {
                                            'block': 'b-icon',
                                            'mods': {'type': 'arrow'}
                                        },
                                        '&nbsp;',
                                        gettext('Приб. ')
                                    ]
                                }
                            ] * (rtstations_cnt - 1)
                        }
                    ] + [{
                        'elem': 'row',
                        'elemMods': {'type': state == 'past' and 'past' or None},
                        'content': self.stat_cells(state)
                    } for state in ('past', 'current', 'future') if self.context.stats.get(state, False)]
                }
            ]
        }

    def stat_cells(self, state):
        cells = []

        for dt in self.context.stats[state]:
            if dt.station.tablo_state == '':
                content = uni_strftime("%H:%M", dt.real)

            elif dt.station.tablo_state == 'statuses':
                content = dt.state_phrase

            elif dt.status == 'nodata':
                content = {'elem': 'grey', 'content': gettext('нет данных')}

            else:
                if state != 'past' and dt.is_bad_status:
                    content = [{'elem': 'mark', 'content': uni_strftime('%H:%M', dt.real)}]
                else:
                    content = [uni_strftime('%H:%M', dt.real)]

                content += [' ']

                if dt.is_bad_status:
                    if dt.status == 'cencelled':
                        content += [{ 'elem': 'mark', 'content': '(' + gettext('отменен') + ')' }]
                    else:
                        if dt.delta:
                            content += xgettext(u"(<status/> на <duration/>)", status=dt.state_phrase, duration=human_duration(dt.delta))
                        else:
                            content += ['(', dt.state_phrase, ')']
                else:
                    content += [dt.state_phrase]


            pair = [
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'date'},
                    'content': uni_strftime("%d&nbsp;%b", dt.real)
                },
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'text'},
                    'content': content
                }
            ]

            cells += pair

        return cells

    def station_url(self, station_id):
        thread = self.context.thread.thread

        station_params = {
            'span': thread.schedule_plan.code if thread.schedule_plan else 'schedule',
            'type': 'suburban'
        } if thread.t_type_id == TransportType.SUBURBAN_ID else {}
        return composeurl('station', kwargs={'station_id': station_id}, params=station_params)


def fit_link(url, request):
    u""" Если в реквесте написано, что пришел робот яндекса, оторвать параметры ссылки """
    if request.yandex_bot and '?' in url:
        return langify(url[:url.index('?')])

    else:
        return url


re_cleaner = re.compile(ur'<.*?>', re.M|re.U|re.S)
def clean_contact_info(contact_info):
    return re_cleaner.sub(' ', contact_info or u'')
