# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from django.template.defaultfilters import capfirst
from django.utils.translation import get_language

from common.models.transport import TransportType
from common.utils.avia import get_avia_thread_url
from common.utils.bemhtml import punctuate_content
from common.utils.locations import composeurl
from common.xgettext.i18n import gettext, xgettext
from travel.rasp.morda.morda.templates.base import Base
from travel.rasp.morda.morda.templates.search import BlockTimetable
from travel.rasp.morda.morda.templates.utils import headnonzerononfail


class BaseTemplate(Base):
    has_filter_block = True

    def content(self):

        filter_block_content = self.has_filter_block and [
            {
                'block': 'b-menu',
                'mods': {'layout': 'horiz-complex', 'preset': 'media-links'},
                'content': [
                    {
                        'elem': 'item',
                        'content': {}
                    }
                ]
            },
            {
                'block': 'b-filter-title',
                'content': self.filter_block_title()
            },
            self.filter_block()
        ] or ''

        return {
            'block': 'l-page',
            'mods': {'layout': '72-20', 'row': 'multiple'},
            'content': [
                {
                    'elem': 'row',
                    'content': [
                        {'elem': 'gap'},
                        {
                            'elem': 'left',
                            'content': self.left_content()
                        },
                        {'elem': 'gap-right'},
                        {
                            'elem': 'right',
                            'content': [
                                filter_block_content,
                                self.teaser('ahtung')
                            ]
                        },
                        {'elem': 'gap'}
                    ]
                }
            ]
        }

    def left_content(self):
        return []

    def filter_block(self):
        pass

    def filter_block_title(self):
        return ''


class SearchBaseTemplate(BaseTemplate):
    page = 'number'

    def filter_block_title(self):
        return gettext('Поиск рейса')

    def filter_block(self):
        number_value = getattr(self.context, 'route_form', None) and \
                        getattr(self.context.route_form, 'cleaned_data', None) and \
                        self.context.route_form.cleaned_data.get('number') or ''

        # RASP-13759 Remove Tramvay and fix names of transports
        transport = (
            ('plane', u'Uçak'),
            ('train', u'Tren'),
            ('bus', u'Otobüs')
        ) if self.context.request.NATIONAL_VERSION == 'tr' else (
            ('plane', gettext(u'самолёты')),
            ('train', gettext(u'поезда')),
            ('suburban', gettext(u'электрички')),
            ('bus', gettext(u'автобусы'))
        )

        return {
            'block': 'b-filter',
            'js': True,
            'content':{
                'elem': 'form',
                'attrs': {'action': '?'},
                'content': {
                    'elem': 'content',
                    'content': [
                        {
                            'block': 'b-menu-vert-mod',
                            'js': True,
                            'mods': {'preset': 'filter'},
                            'content':[
                                {
                                    'elem': 'item',
                                    'content': {
                                        'elem': 'trigger',
                                        'mods': {'state': 'opened'},
                                        'content': [
                                            {
                                                'block': 'b-icon',
                                                'mix': [{'block': 'b-menu-vert-mod', 'elem': 'trigger-icon'}],
                                                'alt': '+'
                                            },
                                            {
                                                'block': 'b-link',
                                                'mods': {'pseudo': 'yes', 'theme': 'black'},
                                                'content': gettext('Рейс или поезд №')
                                            },
                                        ]
                                    },
                                    'item-content': {
                                        'elem': 'item-content',
                                        'mods': {'visibility': 'visible'},
                                        'content': {
                                            'block': 'b-menu-vert-mod',
                                            'mods': {'type': 'submenu'},
                                            'content': [
                                                {
                                                    'elem': 'item',
                                                    'content': {
                                                        'block': 'b-form-input',
                                                        'js': True,
                                                        'mods': {'theme': 'grey', 'size': 'm', 'has-clear': 'yes',},
                                                        'content': [
                                                            {
                                                                'elem': 'input',
                                                                'clear': {
                                                                    'elem': 'clear',
                                                                },
                                                                'attrs': {
                                                                    'id': 'id2562026',
                                                                    'name': 'number',
                                                                    'value': number_value,
                                                                }
                                                            },
                                                        ]
                                                    }
                                                },
                                            ]
                                        }
                                    }
                                },
                                {
                                    'elem': 'item',
                                    'content': {
                                        'elem': 'trigger',
                                        'mods': { 'state': 'opened' },
                                        'content': [
                                            {
                                                'block': 'b-icon',
                                                'mix': [{ 'block': 'b-menu-vert-mod', 'elem': 'trigger-icon' }],
                                                'alt': '+'
                                            },
                                            {
                                                'block': 'b-link',
                                                'mods': { 'pseudo': 'yes', 'theme': 'black' },
                                                # Название раздела фильтра выбора типа транспорта (самолет, поезд,....). По турецки должно звучать как 'Ulaşım türü'
                                                'content': gettext('Транспорт')
                                            }
                                        ]
                                    },
                                    'item-content': {
                                        'elem': 'item-content',
                                        'mods': { 'visibility': 'visible' },
                                        'content': {
                                            'block': 'b-menu-vert-mod',
                                            'mods': { 'type': 'submenu' },
                                            'content': [self.filter_elem(code, content) for code, content in transport]
                                        }
                                    }
                                },
                            ]
                        },
                        {
                            'elem': 'show',
                            'content': {
                                'block': 'b-form-button',
                                'mods': { 'size': 'm', 'theme': 'grey-m' },
                                'type': 'submit', 'content': gettext('Показать'),
                                'attrs': {
                                    'onmousedown': self.metrika_reach_goal("dedicated_thread_search_button_click")
                                },
                            }
                        }
                    ]
                }
            }
        }

    def filter_elem(self, code, content):
        mods = {'theme': 'grey-m', 'size': 'm'}
        attrs = {'id': 'filter_%s_yes' % code, 'name': code, 'value': 'yes'}

        if self.context.route_form and self.context.route_form.data.get(code):
            mods['checked'] = 'yes'
            attrs['checked'] = 'checked'

        return {
            'elem': 'item',
            'content': {
                'block': 'b-form-checkbox',
                'mods': mods,
                'checkboxAttrs': attrs,
                'content': {'elem': 'label', 'content': content}
            }
        }

    def error_left_block(self):
        try:
            number = self.context.route_form.cleaned_data.get('number')
        except AttributeError:
            number = None

        if number:
            content = gettext('Такого рейса не существует. Попробуйте поискать другой рейс или маршрут.')
        else:
            content = gettext('Введите номер рейса или поезда.')

        return [
            {
                'block': 'b-page-title',
                'content': {
                    'elem': 'title',
                    'content': gettext('Поиск рейсов')
                }
            },
            {
                'block': 'b-precise-title',
                'content': '<p>%s</p>' % content
            }
        ]


class OrdinalSearchTemplate(SearchBaseTemplate):
    def desc(self, thread):
        desc = '%s - %s' % (thread.first_station.departure_time.strftime('%H:%M'),
                            thread.last_station.arrival_time.strftime('%H:%M'))

        if thread.first_station.schedule:
            desc += ', %s.' % thread.first_station.schedule

        return ' ' + desc

    def tripdesc(self, thread):
        title_special = thread.L_title_special_short()

        punctuated = punctuate_content(filter(None, [
            title_special and {
                'elem': 'tname',
                'content': title_special
            },
            headnonzerononfail(lambda: thread.company.L_title())
        ]))

        return [' '] + punctuated + [', '] if punctuated else None

    def left_content(self):
        left_content = []

        if self.context.exact:
            left_content += [
                {
                    'block': 'b-subtitle',
                    'content': {
                        'elem': 'title',
                        'content': [
                            {'block': 'b-transico', 'mods': {'type': self.context.exact.t_type.code}, 'alt': ''},
                            gettext('Точное совпадение')
                        ]
                    }
                },
                {
                    'block': 'b-short-timetable',
                    'content': {
                        'elem': 'item',
                        'content': [
                            {
                                'elem': 'tripname',
                                'content': ' ' + self.context.exact.L_title()
                            },
                            '<br />',
                            {
                                'elem': 'trip',
                                'content': {
                                    'block': 'b-link',
                                    'url': get_thread_url(self.context.exact, self.context.request),
                                    'content': self.context.exact.number
                                }
                            },
                            {
                                'elem': 'tripdesc',
                                'content': self.tripdesc(self.context.exact)
                            },
                            self.desc(self.context.exact)
                        ]
                    }
                }
            ]

        if self.context.groups:
            for ttype, type_routes in self.context.groups.iteritems():
                items = []

                if ttype == TransportType.BUS_ID:
                    for direction, courses in self.context.mta_groups.iteritems():
                        for course in ('forward', 'backward'):
                            for route, stations_grp in courses[course].iteritems():
                                for stations, mask_grp in stations_grp.iteritems():
                                    for mask, threads in mask_grp.iteritems():
                                        desc = ''
                                        for thread in threads:
                                            desc += '<a class="b-link" href="{}">{}</a> '.format(
                                                get_thread_url(thread, self.context.request),
                                                thread.first_station.departure_time.strftime('%H:%M')
                                            )
                                        desc += unicode(mask)

                                        item = {
                                            'elem': 'item',
                                            'content': [
                                                {
                                                    'elem': 'tripname',
                                                    'content': ' ' + threads[0].L_title()
                                                },
                                                '<br/>',
                                                thread.number + ' ',
                                                desc
                                            ]
                                        }

                                        items.append(item)

                for route, direction_grp in type_routes.iteritems():
                    if direction_grp and direction_grp[0].supplier_id == 44:  # mta
                        continue

                    for thread in direction_grp:
                        desc = '%s - %s' % (thread.first_station.departure_time.strftime('%H:%M'),
                                            thread.last_station.arrival_time.strftime('%H:%M'))

                        if thread.first_station.schedule:
                            desc += ', %s.' % thread.first_station.schedule

                        item = {
                            'elem': 'item',
                            'content': [
                                {
                                    'elem': 'tripname',
                                    'content': ' ' + thread.L_title()
                                },
                                '<br/>',
                                {
                                    'elem': 'trip',
                                    'content': {
                                        'block': 'b-link',
                                        'url': get_thread_url(thread, self.context.request),
                                        'content': thread.number
                                    }
                                },
                                {
                                    'elem': 'tripdesc',
                                    'content': self.tripdesc(thread)
                                },
                                self.desc(thread)
                            ]
                        }

                        items.append(item)

                query = self.context.route_form.cleaned_data.get('number', '')

                title = None
                if ttype == TransportType.TRAIN_ID:
                    title = xgettext('Поезда по&nbsp;запросу «<query/>»', query=query)
                elif ttype == TransportType.PLANE_ID:
                    title = xgettext('Самолёты по&nbsp;запросу «<query/>»', query=query)
                elif ttype == TransportType.BUS_ID:
                    title = xgettext('Автобусы по&nbsp;запросу «<query/>»', query=query)
                elif ttype in TransportType.WATER_TTYPE_IDS:
                    title = xgettext('Корабли по&nbsp;запросу «<query/>»', query=query)
                elif ttype == TransportType.SUBURBAN_ID:
                    title = xgettext('Электрички по&nbsp;запросу «<query/>»', query=query)
                elif ttype == TransportType.HELICOPTER_ID:
                    title = xgettext('Вертолёты по&nbsp;запросу «<query/>»', query=query)

                if ttype in TransportType.WATER_TTYPE_IDS:
                    t_type_code = 'water'
                else:
                    t_type_code = self.context.types[ttype].code

                group = [
                    '<br />',
                    {
                        'block': 'b-subtitle',
                        'content': {
                            'elem': 'title',
                            'content': [
                                {
                                    'block': 'b-transico',
                                    'mods': { 'type': t_type_code },
                                    'alt': ''
                                },
                                title
                            ]
                        }
                    },
                    {
                        'block': 'b-short-timetable',
                        'content': items
                    },
                ]

                left_content += group

            left_content += ['<br />', { 'tag': 'p', 'content': gettext('время местное') } ]
        else:
            left_content += self.error_left_block()

        return left_content


class FiltersGag(object):

    def is_filtered(self, obj):
        return False

    def filter_values(self, obj):
        return []


class SpecialSearchTemplate(SearchBaseTemplate):

    def left_content(self):
        self.context.filters = FiltersGag()


        if self.context.show_sapsan_link:
            sapsan_link = {
                'block': 'b-link',
                'url': composeurl('info', kwargs={'object_id': 'sapsan'}),
                'content': gettext('Схемы вагонов')
            }
        else:
            sapsan_link = None

        search_results = []
        self.context['mode'] = 'schedule'

        for search_result in self.context.search_results:
            self.context.segments = search_result.segments
            block_t = BlockTimetable(self)

            search_results += [
                {
                    'block': 'fragment',
                    'bem': False,
                    'tag': 'h2',
                    'attrs': {'id': search_result.fragment},
                    'content': [
                        capfirst(search_result['from'].L_title(case='phrase_from')), ' ', search_result['to'].L_title(case='phrase_to')
                    ] if get_language() == 'ru' else [
                        search_result['from'].L_title(), '&nbsp;&mdash; ', search_result['to'].L_title()
                    ]
                },
                block_t
            ]

        left_content = [
            {
                'block': 'b-page-title',
                'content': [

                    {
                        'elem': 'title',
                        'content': xgettext('Расписание поездов «<number/>»', number=self.context.number)
                    },'&nbsp;',

                    sapsan_link

                ]
            },
            {
                'block': 'b-notice',
                'mods': {'type': 'info'},
                'content': [

                    { 'block': 'b-icon', 'mods': { 'type': 'notice' } },

                    gettext('Это расписание на все дни. Для получения информации о стоимости проезда и наличии мест укажите конкретную дату выезда.')

                ]
            },
            {
                'block': 'b-routes-list',
                'content': [{
                    'elem': 'row',
                    'content': [{
                        'elem': 'item',
                        'content': {'block': 'b-link', 'url': '#' + link['fragment'], 'content': link['link_text']}
                    } for link in row if link]
                } for row in self.context.search_navigation if row]
            }
        ] + search_results

        return left_content

    def return_link(self):
        pass

    def b_icon_type_eticket(self):
        pass


def get_thread_url(thread, request, params={}):
    if thread.t_type_id == TransportType.PLANE_ID:
        return get_avia_thread_url(thread, request.tld, utm_medium='main', utm_campaign='flight_search')
    else:
        return composeurl('thread', kwargs={'uid': thread.uid}, params=params)


