# -*- coding: utf-8 -*-

from __future__ import absolute_import

from six.moves.urllib.parse import SplitResult, urlsplit, parse_qsl

from common.models.geo import Settlement
from common.models.transport import TransportType
from common.utils.locations import composeurl, set_lang_param
from common.utils.avia import get_avia_thread_url


def search_url(city_from, city_to, search_type=None, **extra):
    params = get_from_to_params(city_from, city_to)

    params.update(extra)

    args = [search_type] if search_type else []

    return get_search_url(args=args, params=params)


def suburban_search_url(city_from, city_to, **extra):
    params = get_from_to_params(city_from, city_to)

    params.update(extra)

    return get_search_url(args=['suburban'], params=params)


def get_from_to_params(city_from, city_to):
    if isinstance(city_from, tuple):
        title_from, key_from = city_from
    else:
        title_from, key_from = city_from.L_title(), city_from.point_key

    if isinstance(city_to, tuple):
        title_to, key_to = city_to
    else:
        title_to, key_to = city_to.L_title(), city_to.point_key

    return {
        'fromName': title_from,
        'fromId': key_from,
        'toName': title_to,
        'toId': key_to,
    }


def aeroexpress_search_url(airport, reverse=False):
    if not airport.t_type.code == 'plane':
        return None

    if not airport.has_aeroexpress:
        return None

    if not airport.settlement:
        return None

    if reverse:
        point_from = airport
        point_to = airport.settlement
    else:
        point_from = airport.settlement
        point_to = airport

    return suburban_search_url(point_from, point_to, aeroex='y')


def tablo_link(z_tablo, request):
    if not z_tablo.thread:
        return None
    return station_thread_url(z_tablo.thread, z_tablo.start_date, request)


def station_thread_url(thread, departure, request):
    if thread.t_type_id == TransportType.PLANE_ID:
        return get_avia_thread_url(thread, request.tld, departure,
                                   utm_medium='station_landing', utm_campaign='flight_link')
    params = {}
    if departure and not request.yandex_bot:
        params['departure'] = departure
    return composeurl('thread', args=[thread.uid], params=params)


def get_point_url(point):
    if isinstance(point, Settlement):
        return get_city_url(point.id)
    else:
        return get_station_url(point.id)


def get_city_url(city_id):
    return '/city/{}'.format(city_id)


def get_station_url(station_id):
    return '/station/{}'.format(station_id)


def get_search_url(args=None, kwargs=None, params=None, query=''):
    if args:
        path = '/search/{}'.format(args[0])
    elif kwargs:
        path = '/search/{}'.format(kwargs['search_type'])
    else:
        path = '/search'

    params = params or {}
    query = set_lang_param(query or params)

    return SplitResult('', '', path, query, '').geturl()
