# -*- coding: utf-8 -*-

import re
import locale
from itertools import chain, groupby

from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404

from common.models.geo import Settlement, Station
from common.utils.locations import composeurl
from common.xgettext.i18n import xgettext
from travel.rasp.morda.morda.templates.direction_map import DirectionMapTemplate
from travel.rasp.morda.morda.utils.locations import get_city_url
from travel.rasp.morda.morda.views.teasers import TeaserSetMorda


def direction(request, city_id):
    city = get_object_or_404(Settlement.hidden_manager, pk=city_id)

    main_directions, directions = city.get_directions()

    direction_code = request.GET.get('direction')

    direction = None

    all_directions = list(chain(main_directions, directions))

    if not all_directions:
        return HttpResponseRedirect(get_city_url(city_id))

    for d in all_directions:
        # Выбор первого направления, если остальные не совпадут с кодом
        if not direction:
            direction = d

        if d.code == direction_code:
            direction = d
            break

    all_stations = list(Station.objects.filter(hidden=False, externaldirectionmarker__external_direction=direction).all())

    def sorter(station):
        title = station.L_title()
        parts = re.findall('\D+|\d+', title) # сортируем числа по возрастанию, а не как в строках, ставя лидирующие нули

        return "".join(["%020d" % int(p) if p.isdigit() else p for p in parts])

    all_stations.sort(key=sorter, cmp=locale.strcoll)

    def grouper(station):
        title = station.L_title()

        letter = title[0].upper()

        if not letter.isalpha():
            return "0-9"

        if letter == u'Ё':
            return u'Е'

        return letter

    # Высота заголовка (в станциях)
    header_height = 2

    n_columns = 4

    groups = [(letter, list(stations)) for letter, stations in groupby(all_stations, grouper)]

    overall_height = len(all_stations) + header_height * len(groups)

    median_height = (overall_height + n_columns - 1) / n_columns # ceil division

    def block_height(block):
        letter, stations = block
        return len(stations) + header_height

    def divide_variants(groups, n_columns):
        column = []
        column_height = 0

        if n_columns == 1:
            yield [groups]
            return

        # Копия
        groups = groups[:]

        while len(groups):
            block = groups.pop(0)

            b_height = block_height(block)

            if column_height + b_height > median_height:
                # Варианты без переноса блока
                for columns in divide_variants(groups, n_columns - 1):
                    yield [column + [block]] + columns

                # Варианты с переносом блока
                for columns in divide_variants([block] + groups, n_columns - 1):
                    yield [column] + columns

                break

            column.append(block)
            column_height += b_height

        # Предыдущий цикл прошелся по всем группам, но высота колонки не превысила среднюю
        # Возвращаем всю колонку
        if not len(groups):
            yield [column]

    variants = divide_variants(groups, n_columns)

    min_deviation = None
    stations_by_column = None

    for columns in variants:
        deviation = 0.0
        for c in columns:
            column_height = sum(block_height(b) for b in c)
            deviation += (column_height - median_height) ** 2

        avg_deviation = deviation / n_columns

        if min_deviation is None or avg_deviation < min_deviation:
            min_deviation = avg_deviation
            stations_by_column = columns

    # Вкладка
    tab = request.GET.get('tab')

    if tab != 'index':
        tab = None

    schema = direction.schema()

    if not schema:
        tab = 'index'

    kwargs = {
        'city_title': city.L_title,
        'direction_title': direction.L_full_title()
    }

    context = {
        'form': 'suburban',
        'tab': tab,
        'city': city,
        'canonical_url': canonical_url(direction.base_station and direction.base_station.settlement or city, direction),
        'stations_by_column': stations_by_column,
        'direction': direction,
        'schema': schema,
        'main_directions': main_directions,
        'directions': directions,
        'teasers': TeaserSetMorda(request, 'direction', direction),
#        'title': u'%s. Расписание электричек - %s' % (city.title, direction.full_title),
        'title': xgettext(u'<city-title/>. Расписание электричек - <direction-title/>', **kwargs),
        }

    return DirectionMapTemplate.render(request, context)


def canonical_url(main_city, direction):
    if not main_city:
        return None

    return composeurl('city_direction', args=[main_city.id], params={'direction': direction.code})
