# -*- coding: utf-8 -*-

from datetime import date, datetime, timedelta
from itertools import chain, groupby

from django.http import Http404
from django.utils.translation import get_language

from common.models.geo import SuburbanZone, Station, ExternalDirection, ExternalDirectionMarker
from common.models.schedule import RTStation, TrainSchedulePlan
from common.utils.httpresponses import jsonp_response
from common.utils.mysql_try_hard import mysql_try_hard
from common.utils.date import DateTimeFormatter
from common.xgettext.i18n import xgettext, gettext

from travel.rasp.morda.morda.templates.print_schedule.print_settings import Template as SettingsTemplate
from travel.rasp.morda.morda.templates.print_schedule.print_result import Template as ResultTemplate

from route_search.models import ZNodeRoute2


class DisplayRoute(object):
    def __init__(self):
        self.departures = {}
        self.arrivals = {}


def schedule_settings(request, today):
    zones = list(SuburbanZone.objects.order_by('settlement__title'))  # Settlement'ы закешированы

    current_zone = request.client_city.suburban_zone or request.default_city.suburban_zone

    if not current_zone:
        raise Http404

    directions = list(current_zone.externaldirection_set.order_by('title'))

    current_direction = directions[0]

    _, next_plan = TrainSchedulePlan.get_current_and_next(today)

    context = {
        'zones': zones,
        'current_zone': current_zone,
        'directions': directions,
        'stations': list(current_direction.stations.order_by('externaldirectionmarker__order')),
        'next_plan': next_plan,
    }

    return SettingsTemplate.render(request, context)


@mysql_try_hard
def print_schedule(request):
    if get_language() == 'tr':
        raise Http404

    today = date.today()

    if 'print' not in request.GET:
        return schedule_settings(request, today)

    stations_from_ids = [int(i) for i in request.GET.getlist('stationsFrom') if i.isdigit()]
    stations_to_ids = [int(i) for i in request.GET.getlist('stationsTo') if i.isdigit()]

    stations = Station.objects.in_bulk(stations_from_ids + stations_to_ids)

    stations_from = [stations[i] for i in stations_from_ids if i in stations]
    stations_to = [stations[i] for i in stations_to_ids if i in stations]

    current_plan, next_plan = TrainSchedulePlan.get_current_and_next(today)

    plan = next_plan if 'nextPlan' in request.GET else current_plan

    table_forward = getschedule(today, stations_from, stations_to, plan)

    table_backward = getschedule(today, list(reversed(stations_to)), list(reversed(stations_from)), plan)

    context = {
        'stations_from': stations_from,
        'stations_to': stations_to,
        'table_forward': table_forward,
        'table_backward': table_backward,
        'expresses': any(express for _, _, express, _ in chain(table_forward, table_backward)),
        }

    if next_plan or plan and (plan.end_date - today).days < 7:
        if plan.is_current(today):
            context['plan_disclaimer'] = xgettext(u'Расписание актуально до&nbsp;<date format="%d&nbsp;%B %Y"/>&nbsp;г.',
                                                  date=DateTimeFormatter(plan.end_date).L)
        else:
            context['plan_disclaimer'] = xgettext(u'Расписание актуально с&nbsp;<date format="%d&nbsp;%B %Y"/>&nbsp;г.',
                                                  date=DateTimeFormatter(plan.start_date).L)

    return ResultTemplate.render(request, context)


def getschedule(today, stations_from, stations_to, plan):
    noderoutes = list(ZNodeRoute2.objects.filter(
        station_from__in=stations_from,
        station_to__in=stations_to,
        thread__route__t_type__id=6,
        thread__type__code='basic',
        ).select_related('thread', 'thread__route', 'rtstation_from', 'rtstation_to'))

    noderoutes.sort(key=lambda nr: nr.thread_id)

    displayroutes = []

    for thread, tnoderoutes in groupby(noderoutes, key=lambda nr: nr.thread):
        # Пропускаем треды из других графиков
        if plan and thread.schedule_plan_id and thread.schedule_plan_id != plan.id:
            continue

        dr = DisplayRoute()
        displayroutes.append(dr)

        for nr in tnoderoutes:
            dr.thread = nr.thread
            dr.departures[nr.station_from_id] = nr.rtstation_from
            dr.arrivals[nr.station_to_id] = nr.rtstation_to

    # Для определения проездов без остановки
    rtstations = RTStation.objects.filter(
        thread__in=[_dr.thread for _dr in displayroutes],
        station__in=list(chain(stations_from, stations_to))).only('id', 'thread__id', 'station__id')

    rtstations = set((s.thread_id, s.station_id) for s in rtstations)

    table = []

    # Формируем таблицу
    for r in displayroutes:
        first_run = r.thread.first_run(today)

        if first_run is None:
            continue

        naive_start_dt = datetime.combine(first_run, r.thread.tz_start_time)

        def format_stops(stations, event):
            first_stop = None
            stops = []

            class Stop:
                def __init__(self, time, l_platform=None):
                    self.time = time
                    self.l_platform = l_platform

            for s in stations:
                try:
                    rtstation = getattr(r, event + 's')[s.id]

                    local = rtstation.get_event_dt(event, naive_start_dt, out_tz=s.pytz)

                    stop = Stop(local.strftime("%H:%M"))

                    if first_stop is None:
                        first_stop = local

                    if rtstation.L_platform():
                        stop.l_platform = rtstation.L_platform()

                except KeyError:
                    # Если нет записи в noderoute, но есть rtstation —
                    # значит поезд не останавливается на этой станции

                    if (r.thread.id, s.id) in rtstations:
                        stop = Stop('&ndash;')
                    else:
                        stop = Stop('')

                stops.append(stop)

            return first_stop, stops

        first_departure, departures = format_stops(stations_from, 'departure')
        _, arrivals = format_stops(stations_to, 'arrival')

        shift = (first_departure.date() - naive_start_dt.date()).days
        schedule_text = r.thread.plain_days_text(shift, get_language())

        if schedule_text is None or len(schedule_text) > 23:
            # расписание хождения электричек: по отдельным дням
            schedule_text = gettext(u'по&nbsp;отд. дням')

        table.append((first_departure, schedule_text, departures, r.thread.is_express, arrivals))

    # Сортируем по времени отправления с первой станции
    table.sort(key=lambda r: r[0].time())

    return [r[1:] for r in table]


@jsonp_response
def directions(request):
    zone = SuburbanZone.objects.get(id=request.GET['zone'])

    response = []

    for direction in zone.externaldirection_set.order_by('title'):
        response.append({
            'id': direction.id,
            'name': direction.L_title(),
        })

    return response


@jsonp_response
def stations(request):
    zone_id = request.GET['zone']

    direction_id = request.GET.get('direction')

    response = []

    for station in get_stations(zone_id, direction_id):
        d_title = getattr(station, 'direction_title', None)

        if d_title:
            name = u"%s (%s)" % (station.L_suburban_title(), d_title)
        else:
            name = station.L_suburban_title()

        response.append({
            'id': station.id,
            'name': name,
        })

    return response


def get_stations(zone_id, direction_id):
    if direction_id:
        stations = ExternalDirection.objects.get(id=direction_id).stations.order_by('externaldirectionmarker__order')
    else:
        stations = Station.objects.filter(hidden=False, externaldirectionmarker__external_direction__suburban_zone__id=zone_id).order_by('title').distinct()

        add_directions(stations)

    return stations


def add_directions(stations):
    groups = dict()

    for s in stations:
        groups.setdefault(s.title, []).append(s)

    dmarkers = ExternalDirectionMarker.objects.filter(station__in=[s for g in groups.values() if len(g) > 1 for s in g])

    directions_mapping = {}

    all_directions = set()

    for dm in dmarkers:
        directions_mapping.setdefault(dm.station_id, []).append(dm.external_direction_id)
        all_directions.add(dm.external_direction_id)

    all_directions = ExternalDirection.objects.in_bulk(all_directions)

    for group in groups.values():
        if len(group) > 1:
            for i, s in enumerate(group):
                station_directions = set(directions_mapping.get(s.id, []))

                for other in chain(group[:i], group[i + 1:]):
                    station_directions.difference_update(directions_mapping.get(other.id, []))

                if station_directions:
                    station_direction = all_directions.get(station_directions.pop())

                    if station_direction:
                        s.direction_title = station_direction.L_cutted_full_title()
