# -*- coding: utf-8 -*-

from __future__ import unicode_literals


import random
from urllib import quote, unquote

from django.conf import settings

from common.models.geo import Point, Settlement, Station
from common.xgettext.i18n import gettext


TRIP_COOKIE_NAME = b'trip_params'


def store_params(request, response, search_type, form):
    stored_params = restore_params(request)

    stored_params[search_type or 'all'] = (
        form.cleaned_data['from'].point.L_title(),
        form.cleaned_data['to'].point.L_title(),
        form.data['when']
    )

    serialized = ";".join([
        "%s:%s" % (st, ','.join(params))
        for st, params in stored_params.items()
    ])

    cookie = quote(serialized.encode('utf8'))

    response.set_cookie(TRIP_COOKIE_NAME, cookie)


def restore_params(request):
    """Формат куки: 'None:Москва,Питер,10 ноября;train:Самара,Саратов,завтра'"""

    try:
        cookie = request.COOKIES[TRIP_COOKIE_NAME]
    except KeyError:
        return {}

    unqouted = unquote(cookie)

    try:
        serialized = unqouted.decode('utf-8')
    except UnicodeDecodeError:
        return {}

    params = {}

    for pair in serialized.split(';'):
        try:
            st, st_params = pair.split(':')
        except ValueError:
            continue

        params[st] = st_params.split(',')

    return params


def search_samples(request):
    client_city = request.client_city

    params = restore_params(request)

    for search_type in params.keys():
        params[search_type] = dict(zip(('from', 'to', 'when'), params[search_type]))

    city_from = client_city

    city_to_id, backup_city_to_id = \
        settings.NATIONAL_VERSION_DEFAULT_SEARCH_CITIES.get(request.NATIONAL_VERSION)

    if city_from.id == city_to_id:
        city_to_id = backup_city_to_id

    city_to = Settlement.objects.get(id=city_to_id)

    base_sample = {
        'from': city_from.L_title(),
        'to': city_to.L_title()
    }

    suburban_sample = base_sample.copy()

    sample_title = Station.sample_title(client_city.region_id, client_city.country_id)

    if sample_title:
        suburban_sample['to'] = sample_title

    default_params = {
        'all': base_sample,
        'train': base_sample,
        'plane': base_sample,
        'suburban': suburban_sample,
        'bus': get_bus_sample(request),
        'water': get_water_sample(request),
    }

    for search_type in default_params.keys():
        if not search_type in params:
            params[search_type] = default_params[search_type]

    return params


def _sample_id_to_title(from_id, to_id):
    points = Point.in_bulk([from_id, to_id])

    point_from = points.get(from_id)
    point_to = points.get(to_id)

    return point_from and point_from.L_title(), point_to and point_to.L_title()


def _get_bus_sample(request):
    country = request.NATIONAL_VERSION

    # RASP-9691
    if country == 'tr':
        # Когда в базе появятся турецкие автобусные рейсы, нужно будет выбрать штук 30
        # городов с автобусным сообщением для подсказок
        return gettext(u'Анкара'), gettext(u'Стамбул')

    if country == 'ua':
        from_id = 'c143' # Киев

        to_id = random.choice([
            'c147', # Харьков
            'c145', # Одесса
            'c141', # Днепропетровск
            'c142', # Донецк
            'c960', # Запорожье
            'c144', # Львов
        ])

        return _sample_id_to_title(from_id, to_id)

    if country == 'ru':
        return random.choice([
            (u'Ижевск', u'Сарапул'),
            (u'Ульяновск', u'Димитровград'),
            (u'Санкт-Петербург', u'Подпорожье'),
            (u'Москва', u'Хвастовичи'),
            (u'Москва', u'Козельск'),
            (u'Волгоград', u'Москва'),
            (u'Майкоп', u'Москва'),
            (u'Волгоград', u'Ростов-на-Дону'),
            (u'Глазов', u'Пудем'),
            (u'Великий Новгород', u'Старая Русса'),
            (u'Ульяновск', u'Чердаклы'),
            (u'Сенгилей', u'Цемзавод Сенгилеевский'),
            (u'Москва', u'Иваново'),
            (u'Москва', u'Тамбов'),
            (u'Москва', u'Тула'),
            (u'Москва', u'Серебряные Пруды'),
            (u'Екатеринбург', u'Алапаевск'),
            (u'Екатеринбург', u'Лесной'),
            (u'Екатеринбург', u'Кировград'),
            (u'Екатеринбург', u'Серов'),
            (u'Ижевск', u'Чайковский'),
            (u'Санкт-Петербург', u'Великий Новгород'),
            (u'Санкт-Петербург', u'Сланцы'),
            (u'Санкт-Петербург', u'Выборг'),
            (u'Санкт-Петербург', u'Приморск'),
            (u'Санкт-Петербург', u'Тихвин'),
            (u'Великий Новгород', u'Санкт-Петербург'),
            (u'Великий Новгород', u'Луга'),
            (u'Великий Новгород', u'Красные Станки'),
        ])

    return None, None


def _get_sea_sample(request):
    country = request.NATIONAL_VERSION

    if country == 'tr':
        from_id, to_id = random.choice([
            ('c11508', 'c11504'), # İstanbul, Bursa
            ('c11508', 'c28842'), # İstanbul, Yalova
        ])

        return _sample_id_to_title(from_id, to_id)

    return None, None


def get_bus_sample(request):
    from_, to = _get_bus_sample(request)

    return {'from': from_, 'to': to}


def get_water_sample(request):
    from_, to = _get_sea_sample(request)

    return {'from': from_, 'to': to}
