# -*- coding: utf-8 -*-

from datetime import time
from itertools import groupby

from common.models.tariffs import CLS_MAPPING
from common.xgettext.i18n import mark_gettext


class BaseFilter(object):
    def __init__(self, request_get, objects):
        self.values = dict(
            (obj, self.get_value(obj))
            for obj in objects
        )

        self.objects = objects

        self.prepare()

    def filters(self, obj):
        return not self.passes(obj)

    def update_options(self, objects, filtered_by):
        pass

    @property
    def data(self):
        return {
           'param': self.param_name,
           'handler': self.handler,
           }


class SimpleFilter(BaseFilter):
    template = 'simple'
    handler = 'simple'

    dynamic = False
    restrict = False
    always_open = False

    def __init__(self, request_get, segments):
        self.selected = set(request_get.getlist(self.param_name))

        BaseFilter.__init__(self, request_get, segments)

    def get_options(self):
        return self.options

    def prepare(self):
        self._options = self.get_options()

        self.active = set()

        for value in self.values.values():
            if value:
                if isinstance(value, list):
                    self.active.update(value)
                else:
                    self.active.add(value)

        self.selected = self.selected.intersection(self.active)

        if len(self.active) == 1:
            self.hide()

    def passes(self, segment):
        if not self.selected:
            return True

        value = self.values[segment]

        if value is None:
            return True

        if isinstance(value, list):
            if self.restrict:
                return self.selected.issuperset(value)
            else:
                return self.selected.intersection(value)

        return value in self.selected

    def hide(self):
        self.selected = set()
        self.active = set()

    def disable(self):
        self.selected = set()

    def __iter__(self):
        for o in self._options:
            checked = o.value in self.selected

            hidden = (not self.active.intersection(o.value)) if isinstance(o.value, list) else (o.value not in self.active)

            yield o, checked, hidden

    @property
    def opened(self):
        return self.always_open or self.selected

    @property
    def data(self):
        return {
           'param': self.param_name,
           'restrict': self.restrict,
           'handler': self.handler,
           }


class FilterOption(object):
    def __init__(self, value, name, counter=None, tooltip=None, attrs={}):
        self.value = value
        self.name = name
        self.counter = counter
        self.attrs = attrs
        self.tooltip = tooltip


class TransportFilter(SimpleFilter):
    options = [
        FilterOption('plane', mark_gettext(u'самолёты'), counter="70141"),
        FilterOption('train', mark_gettext(u'поезда'), counter="70142"),
        FilterOption('suburban', mark_gettext(u'электрички'), counter="70208"),
        FilterOption('bus', mark_gettext(u'автобусы'), counter="70881"),
        FilterOption('river', mark_gettext(u'речной транспорт'), counter="72126"),
        FilterOption('sea', mark_gettext(u'теплоход'), counter="72126"),
        FilterOption('water', mark_gettext(u'водный транспорт'), counter="72126"),
    ]

    restrict = True

    always_open = True

    name = mark_gettext(u'Транспорт')

    param_name = 'transport'

    def get_value(self, segment):
        if hasattr(segment, 't_type'):
            return segment.t_type.code

        if hasattr(segment, 'transport_types'):
            return [t_type.code for t_type in segment.transport_types]


class TicketsFilter(SimpleFilter):
    dynamic = True

    always_open = True

    options = [
        FilterOption('y', mark_gettext(u'только с билетами')),
        ]

    name = mark_gettext(u'Билеты')

    param_name = 'seats'

    def get_value(self, segment):
        if getattr(segment, 'thread', False) and segment.thread.is_interval:
            return 'n'

        if not hasattr(segment, 't_type') or segment.t_type.code in ['suburban', 'bus']:
            return None

        tariffs = segment.display_info.get('tariffs_info')

        if tariffs and tariffs.places:
            return 'y'

        return 'n'


class TrainTicketsFilter(SimpleFilter):
    dynamic = True

    always_open = True

    # Групбаем удаляем повторяющиеся имена тарифов (а они там есть, потому-что название "купе" - общее)
    options = [FilterOption(name, name) for name, codes in groupby(CLS_MAPPING, key=lambda t: t[0])]

    name = mark_gettext(u'Билеты')

    param_name = 'tickets'

    def get_value(self, segment):
        if 'tariffs_info' in segment.display_info:
            return [p.name for p in segment.display_info['tariffs_info'].places if p.name]

        return []


class ExpressFilter(SimpleFilter):
    dynamic = True

    always_open = True

    options = [
        FilterOption('y', mark_gettext(u'только экспрессы'))
        ]

    name = mark_gettext(u'Экспресс')

    param_name = 'express'

    def get_value(self, segment):
        return 'y' if getattr(segment, 'is_express', None) else 'n'


class AeroexFilter(SimpleFilter):
    options = [
        FilterOption('y', mark_gettext(u'аэроэкспрессы')),
        FilterOption('n', mark_gettext(u'электрички')),
        ]

    name = mark_gettext(u'Аэроэкспресс')

    param_name = 'aeroex'

    def get_value(self, segment):
        return 'y' if getattr(segment, 'thread', None) and segment.thread.is_aeroexpress else 'n'


class CompanyFilter(SimpleFilter):
    name = mark_gettext(u'Авиакомпания')

    param_name = 'carrier'

    def get_options(self):
        companies = set(s.company for s in self.objects if getattr(s, 'company', None))

        options = [FilterOption(str(c.id), c.L_title()) for c in companies]

        return options

    def get_value(self, segment):
        if getattr(segment, 'company', None):
            return str(segment.company.id)


class StationFilter(SimpleFilter):
    def get_options(self):
        stations = set(self.get_station(s) for s in self.objects)

        options = [FilterOption(self.value_from_st(s, t), s.L_title_with_terminal(terminal=t), attrs={'station': s}) for s, t in stations]
        options.sort(key=lambda o: o.name)

        for name, opts in groupby(options, lambda o: o.name):
            opts = list(opts)

            if len(opts) > 1:
                for o in opts:
                    prefix = o.attrs['station'].station_type.L_prefix()

                    if prefix:
                        o.name = prefix + u' ' + name

        return options

    def value_from_st(self, station, terminal):
        if terminal:
            return "%s-%s" % (station.id, terminal.id)

        return str(station.id)

    def get_value(self, segment):
        station, terminal = self.get_station(segment)

        return self.value_from_st(station, terminal)


class DepartureStationFilter(StationFilter):
    param_name = 'stationFrom'

    name = mark_gettext(u'Станция отправления')

    def get_station(self, segment):
        return segment.station_from, segment.rtstation_from and segment.rtstation_from.terminal


class ArrivalStationFilter(StationFilter):
    param_name = 'stationTo'

    name = mark_gettext(u'Станция прибытия')

    def get_station(self, segment):
        return segment.station_to, segment.rtstation_to and segment.rtstation_to.terminal


class DepartureAirportFilter(DepartureStationFilter):
    name = mark_gettext(u'Аэропорт отправления')


class ArrivalAirportFilter(ArrivalStationFilter):
    name = mark_gettext(u'Аэропорт прибытия')


class DepartureTrainStationFilter(DepartureStationFilter):
    name = mark_gettext(u'Вокзал отправления')


class ArrivalTrainStationFilter(ArrivalStationFilter):
    name = mark_gettext(u'Вокзал прибытия')


class DepartureBusStationFilter(DepartureStationFilter):
    name = mark_gettext(u'Вокзал отправления')


class ArrivalBusStationFilter(ArrivalStationFilter):
    name = mark_gettext(u'Вокзал прибытия')


class TransfersFilter(SimpleFilter):
    options = [
        FilterOption('d', mark_gettext(u'прямые')),
        FilterOption('t', mark_gettext(u'с пересадками')),
    ]

    param_name = u'transfers'
    name = mark_gettext(u'Пересадки')

    def get_value(self, segment):
        if hasattr(segment, 'transfers'):
            return 't'

        return 'd'


class FilterSet(object):
    def __init__(self, filters):
        self.filters = filters

    class Data(object):
        def __init__(self):
            self.filtered = set()
            self.values = {}

        def filter_values(self, obj):
            return self.values[obj]

        def is_filtered(self, obj):
            return obj in self.filtered

    def apply(self, request_get, objects):
        filters = [f(request_get, objects) for f in self.filters]

        data = self.Data()

        filtered_by = {}

        for obj in objects:
            data.values[obj] = dict((f.param_name, f.values[obj]) for f in filters)

            obj_filtered_by = filtered_by[obj] = set(f for f in filters if f.filters(obj))

            if obj_filtered_by:
                data.filtered.add(obj)

        # GET-параметры исходного запроса
        params = request_get.copy()

        data.params = request_get.copy()

        # убираем параметры фильтров
        for f in filters:
            if f.param_name in params:
                del data.params[f.param_name]

        filters = [f for f in filters if f.active or f.dynamic]

        for f in filters:
            f.update_options(objects, filtered_by)

        data.filters = dict((f.param_name, f) for f in filters)
        data.ordered = [f for f in filters if (f.active or f.dynamic) and f.template]

        data.active = any(f.active for f in filters)

        return data


class HighSpeedTrainFilter(SimpleFilter):
    always_open = True

    options = [
        FilterOption('lastochka', mark_gettext(u'Ласточка')),
        FilterOption('sapsan', mark_gettext(u'Сапсан')),
        FilterOption('strizh', mark_gettext(u'Стриж')),
    ]

    name = mark_gettext(u'Скоростной поезд')

    param_name = 'highSpeedTrain'

    def get_value(self, segment):
        thread = getattr(segment, 'thread', None)

        if thread:
            deluxe_train = segment.thread.deluxe_train

            if deluxe_train:
                # Сюда же попадает переходный вариант - 'Сапсан по 31 мая/Стриж с 1 июня'
                if deluxe_train.title_ru.startswith(u'Сапсан'):
                    return 'sapsan'

                if deluxe_train.title_ru == u'Ласточка':
                    return 'lastochka'

                if deluxe_train.title_ru == u'Стриж':
                    return 'strizh'

        return 'none'


class TimeFilter(SimpleFilter):
    options = [
        FilterOption('morning', mark_gettext(u'утро'), tooltip=u'06:00 - 12:00'),
        FilterOption('day', mark_gettext(u'день'), tooltip=u'12:00 - 18:00'),
        FilterOption('evening', mark_gettext(u'вечер'), tooltip=u'18:00 - 24:00'),
        FilterOption('night', mark_gettext(u'ночь'), tooltip=u'00:00 - 06:00'),
    ]

    def get_value(self, segment):
        t = self.get_time(segment)

        if t is None:
            return None

        if getattr(segment, 'thread', False) and segment.thread.is_interval:
            return 'none'

        if t < time(6, 0):
            return 'night'
        elif t < time(12, 0):
            return 'morning'
        elif t < time(18, 0):
            return 'day'
        else:
            return 'evening'


class DepartureTimeFilter(TimeFilter):
    name = mark_gettext(u'Время отправления')

    param_name = u'depTime'

    def get_time(self, segment):
        return segment.departure and segment.departure.time()


class ArrivalTimeFilter(TimeFilter):
    name = mark_gettext(u'Время прибытия')

    param_name = u'arrTime'

    def get_time(self, segment):
        return segment.arrival and segment.arrival.time()


FILTERS = {
    None: FilterSet([
        TransportFilter,
        HighSpeedTrainFilter,
        ExpressFilter,
        TicketsFilter,
        TransfersFilter,
        DepartureTimeFilter,
        ArrivalTimeFilter,
        DepartureStationFilter,
        ArrivalStationFilter,
   ]),
    'plane': FilterSet([
        TicketsFilter,
        CompanyFilter,
        DepartureTimeFilter,
        ArrivalTimeFilter,
        DepartureAirportFilter,
        ArrivalAirportFilter,
    ]),
    'train': FilterSet([
        HighSpeedTrainFilter,
        ExpressFilter,
        TrainTicketsFilter,
        DepartureTimeFilter,
        ArrivalTimeFilter,
        DepartureTrainStationFilter,
        ArrivalTrainStationFilter,
    ]),
    'suburban': FilterSet([
        ExpressFilter,
        AeroexFilter,
        DepartureTimeFilter,
        ArrivalTimeFilter,
    ]),
    'bus': FilterSet([
        DepartureTimeFilter,
        ArrivalTimeFilter,
        DepartureTrainStationFilter,
        ArrivalTrainStationFilter,
    ]),
    'water': FilterSet([
        DepartureTimeFilter,
        ArrivalTimeFilter,
        DepartureTrainStationFilter,
        ArrivalTrainStationFilter,
    ]),
}


def apply(request, search_type, segments):
    return FILTERS[search_type].apply(request.GET, segments)
