# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.conf import settings
from django.http import HttpResponseRedirect, HttpResponsePermanentRedirect, HttpResponse
from django.shortcuts import get_object_or_404
from django.utils.translation import get_language

from common.models.geo import Country, Settlement, Station
from common.models.transport import TransportType
from common.utils.date import parse_date
from common.utils.locations import get_choice, composeurl
from common.utils.mysql_try_hard import mysql_try_hard
from common.views.timezones import fill_tz_context
from common.xgettext.i18n import mark_gettext, xgettext, dynamic_gettext
from geosearch.forms import StationForm
from travel.rasp.morda.morda.models import Redirect
from travel.rasp.morda.morda.templates.station.hidden import Template as HiddenStationTemplate
from travel.rasp.morda.morda.utils.domodevo_news import get_domodedovo_news
from travel.rasp.morda.morda.views.station.common_schedule import schedule
from travel.rasp.morda.morda.views.station.locations import get_type_url
from travel.rasp.morda.morda.views.station.plane import plane_schedule
from travel.rasp.morda.morda.views.station.suburban import suburban
from travel.rasp.morda.morda.views.station.tablo import tablo
from travel.rasp.morda.morda.views.station.title import construct_titles
from travel.rasp.morda.morda.views.station.train import train

from travel.rasp.library.python.common_old.settlement import get_stations_by_type


log = logging.getLogger(__name__)


class Event(object):
    def __init__(self, code, title):
        self.code = code
        self.title = title


EVENTS = [
    Event('departure', mark_gettext('Отправление')),
    Event('arrival', mark_gettext('Прибытие')),
]


@mysql_try_hard
def station(request, station_id=None, esr_code=None, express_code=None):
    """Просмотр табло для указанной станции"""

    today = request.now.date()

    # Расписание на заданный день
    try:
        today = parse_date(request.GET.get('date')) or today
    except (ValueError, TypeError):
        pass

    if esr_code:
        station = get_object_or_404(Station.objects, code_set__system__code='esr',
                                    code_set__code=esr_code)
    elif express_code:
        station = get_object_or_404(Station.objects, code_set__system__code='express',
                                    code_set__code=express_code)
    else:
        station = get_object_or_404(Station.objects, pk=station_id)

    context = {
        'station': station,
        'today': today,
        'show_print': True,
        'month': 'FIXME',
        'span': request.GET.get('span')
    }

    city = _get_city_from_request(request, station)
    types = type_choices(station)

    response = _redirect_or_fail_response(request, station, city, bool(types))
    if isinstance(response, HttpResponse):
        return response

    view_type = get_choice(request.GET, 'type', types).choice
    event = (
        'departure'
        if view_type.code == 'suburban' else
        get_choice(request.GET, 'event', EVENTS).choice.code
    )

    context.update({
        'city': city,
        'other_stations': city and get_stations_by_type(city, plane_only=city.is_foreign()),
        'view_type': view_type,
        'view_types': types,
        'event': event,
        'events': EVENTS,
        'canonical_url': canonical_url(station, view_type)
    })

    if 'title' in request.GET:
        context['station_form'] = StationForm(initial=request.GET)

    fill_tz_context(request, context, [station], dates=[today])

    # Новости Домодедово. Когда появятся новости других станций, сделать универсально
    # Показываем новости Домодедово только при отображении на русском языке
    if station.id == 9600216 and get_language() == 'ru':
        try:
            context['stations_news'] = get_domodedovo_news()
        except Exception:
            # TODO: если новости нужны, нужно вернуть log.exception
            # https://st.yandex-team.ru/RASPFRONT-5009
            log.warning('Не смогли получить новости', exc_info=True)

    context.update(construct_titles(today, station, view_type.code, event))

    context['counter'] = settings.COUNTERS.get(station.t_type_id, 0)

    # RASP-11202
    if view_type.code in ('tablo', 'train', 'plane'):
        context['fast_search_points'] = fast_search_points(station, context)

    if view_type.code == 'tablo':
        return tablo(request, station, event, context)
    elif view_type.code == 'plane':
        return plane_schedule(request, context, station, today, event)
    elif view_type.code == 'suburban':
        return suburban(request, context, station, city, view_type.code, today)
    elif view_type.code == 'aeroex':
        return HttpResponsePermanentRedirect(get_type_url(request, view_type.code, station))
    elif view_type.code == 'train':
        return train(request, context, station, today, event)

    return schedule(request, context, station, today, event, view_type.code)


def canonical_url(station, view_type):
    params = {}
    if view_type.code != 'tablo':
        params['type'] = view_type.code
    return composeurl('station', args=[station.id], params=params)


def hidden_station(request, station, city):
    if station.t_type_id == 1:
        if station.popular_title:
            title = xgettext(u"Расписание &mdash; <station-popular-title/>",
                             station_popular_title=station.L_popular_title)
        else:
            title = xgettext(u"Расписание по станции &laquo;<station-title/>&raquo;", station_title=station.L_title)
    else:
        if station.station_type_id == 2:
            title = xgettext(u"Расписание платформы &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 3:
            title = xgettext(u"Расписание остановочного пункта &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 4:
            title = xgettext(u"Расписание блок-поста &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 5:
            title = xgettext(u"Расписание поста &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 6:
            title = xgettext(u"Расписание разъезда &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 7:
            title = xgettext(u"Расписание обгонного пункта &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 8:
            title = xgettext(u"Расписание вокзала &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 9:
            title = xgettext(u"Расписание аэропорта &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 10:
            title = xgettext(u"Расписание автовокзала &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 11:
            title = xgettext(u"Расписание автобусной остановки &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 13:
            title = xgettext(u"Расписание порта &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 14:
            title = xgettext(u"Расписание портпункта &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 15:
            title = xgettext(u"Расписание пристани &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 16:
            title = xgettext(u"Расписание речного вокзала &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        elif station.station_type_id == 17:
            title = xgettext(u"Расписание морского вокзала &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)
        else:
            title = xgettext(u"Расписание станции &laquo;<station-popular-title/>&raquo;",
                             station_popular_title=station.L_popular_title_extra)

    context = {
        'station': station,
        'city': city,
        'title': title,
        'page_title': title,
    }

    return HiddenStationTemplate.render(request, context)


class StationViewType(object):
    def __init__(self, code, title):
        self.code = code
        self._title = title

    @property
    def title(self):
        return dynamic_gettext(self._title)

    def get_url(self, request, station):
        return get_type_url(request, self.code, station)


_STATION_VIEW_TYPES = [
    StationViewType('tablo', mark_gettext('табло')),
    StationViewType('plane', mark_gettext('самолёты')),
    StationViewType('suburban', mark_gettext('электрички')),
    StationViewType('aeroex', mark_gettext('аэроэкспрессы')),
    StationViewType('train', mark_gettext('поезда')),
    StationViewType('schedule', mark_gettext('расписание')),
    StationViewType('water', mark_gettext('теплоходы')),
]

STATION_VIEW_TYPES = {view_type.code: view_type for view_type in _STATION_VIEW_TYPES}


def type_choices(station):
    """Режимы, отображения, доступные для станции"""

    if station.hidden or station.not_in_search() or not station.type_choices:
        return []

    type_list = station.type_choices.split(',')

    if not station.tablo_state:
        type_list = filter(lambda t: t != 'tablo', type_list)

        if station.t_type.code == 'plane':
            type_list.insert(0, 'plane')

    if 'aeroex' in type_list and not station.has_aeroexpress:
        type_list = filter(lambda t: t != 'tablo', type_list)

    return [STATION_VIEW_TYPES.get(view_type_code, mark_gettext('Неизвестное направление'))
            for view_type_code in type_list]


def fast_search_points(station, context):
    u"""
    Возвращает города, на поиск до(от) которых нужно сформировать ссылки
    RASP-11202
    """

    if station.country_id == Country.RUSSIA_ID:
        city_ids = [cid for cid in (213, 2) if station.settlement_id != cid]
        return Settlement.objects.filter(id__in=city_ids)

    elif station.country_id in settings.OUR_COUNTRIES:
        context['not_russia'] = True
        capital = station.country.get_capital()

        if not capital:
            return

        if station.settlement and capital.id == station.settlement.id:
            return

        return [capital]


def _get_city_from_request(request, station):
    try:
        city_id = int(request.GET.get('city'))
    except (ValueError, TypeError):
        city_id = None

    if city_id:
        try:
            return Settlement.hidden_manager.get(pk=city_id)
        except Settlement.DoesNotExist:
            pass
    elif station.settlement_id:
        try:
            return Settlement.hidden_manager.get(pk=station.settlement_id)
        except Settlement.DoesNotExist:
            pass


def _redirect_or_fail_response(request, station, city, has_view_types):
    if station.hidden:
        try:
            redirect = Redirect.objects.get(old_url=request.META['PATH_INFO'])
            return HttpResponseRedirect(redirect.get_new_url())
        except Redirect.DoesNotExist:
            return hidden_station(request, station, city)

    if station.is_foreign() and station.t_type_id != TransportType.PLANE_ID:
        return HttpResponseRedirect(composeurl('info_object', args=['station', station.id]))

    if station.hidden:
        return hidden_station(request, station, city)

    # RASP-9057 МТА станции, не попадающие в поиск
    if station.not_in_search() and not station.is_mta:
        return HttpResponseRedirect(composeurl('info_object', args=['station', station.id]))

    if not has_view_types:
        # Ничего не ходит через станцию
        return hidden_station(request, station, city)
