# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from locale import strcoll
from collections import defaultdict

from common.utils import tracer
from common.utils.text import NBSP
from common.utils.title_generator import DASH
from common.views.thread import get_interval_string
from stationschedule import get_schedule_class
from stationschedule.type.base import BaseIntervalSchedule

from travel.rasp.morda.morda.views.station.utils import fill_spans_context, get_schedule_date_by_span, awaps_params
from travel.rasp.morda.morda.views.teasers import TeaserSetMorda
from travel.rasp.morda.morda.views.thread_search import Sorter
from travel.rasp.morda.morda.templates.station.schedule import Template as StationScheduleTemplate
from travel.rasp.morda.morda.templates.station.schedule_bus_block import Template as ScheduleBusBlockTemplate


def schedule(request, context, station, today, event, type_=None):
    # RASP-9856 блочное расписание автобусов по станции
    if 'bus' in (type_, station.t_type.code) and station.is_block_schedule:
        return schedule_bus_block(request, context, station, today)

    schedule_cls = get_schedule_class(station)
    schedule = schedule_cls(station, event=event)

    current_plan, next_plan = schedule.current_next_plans(today)

    current_span = fill_spans_context(request, station, context, current_plan=current_plan,
                                      next_plan=next_plan)

    local_date = get_schedule_date_by_span(request, station, current_span)
    if local_date:
        context['date'] = local_date

    schedule.build(schedule_date=local_date, schedule_plan=current_span.schedule_plan)

    context['schedule'] = schedule.schedule_routes

    # На все дни сортировать по времени, а на дату - по полному datetime,
    # чтобы корректно захватить рейсы на следуюий день
    columns = {
        "time": ('datetime', (lambda s: s.event_dt) if context.get('date', False) else (lambda s: s.event_dt.time())),
    }

    sort = Sorter(columns, request.GET, default='+time')

    context['sort'] = sort

    sort.sort(context['schedule'])

    context['teasers'] = TeaserSetMorda(request, 'tablo', (context['station'], context['schedule']))
    context['awaps_params'] = awaps_params('tablo', station, context.get('date'))

    return StationScheduleTemplate.render(request, context)


@tracer.wrap
def schedule_bus_block(request, context, station, today):
    tracer.tag('enable schedule_bus_block')

    schedule_cls = get_schedule_class(station)
    schedule = schedule_cls(station, all_days_next_days=60, limit=100000)
    interval_schedule = BaseIntervalSchedule(station, all_days_next_days=60)

    current_span = fill_spans_context(request, station, context)

    local_date = get_schedule_date_by_span(request, station, current_span)
    if local_date:
        context['date'] = local_date

    schedule.build(schedule_date=local_date)
    interval_schedule.build(schedule_date=local_date)

    trace_section = tracer.enter('schedule blocks')

    keyfunc = lambda s: (s.number, s.title)

    schedule_blocks = defaultdict(list)

    for sr in schedule.schedule_routes:
        schedule_blocks[keyfunc(sr)].append(sr)

    for sr in interval_schedule.schedule_routes:
        schedule_blocks[keyfunc(sr)].append(sr)

    title_separator = '{}{} '.format(NBSP, DASH)

    schedule_blocks = [
        {'number': number,
         'title': title
             .replace(' - ', title_separator)
             .replace(' {} '.format(DASH), title_separator)
             .replace('{}- '.format(NBSP), title_separator),
         'schedule_routes': list(schedule_routes)}

        for (number, title), schedule_routes in schedule_blocks.items()
    ]

    trace_section.leave()

    def calendar_sort_keyfunc(calendar):
        if calendar['days_text_ru'] == u'ежедневно':
            return 1

        elif calendar['days_text_ru'].startswith(u'ежедневно'):
            return 2

        else:
            return 3

    trace_section = tracer.enter('schedule calendars')

    for block in schedule_blocks:
        interval_routes = []

        block['station_is_first_in_route'] = block['schedule_routes'][0].rtstation.tz_arrival is None

        schedule_routes_by_days_texts = defaultdict(list)
        for schedule_route in block['schedule_routes']:
            if schedule_route.thread.is_interval:
                interval_routes.append(schedule_route)

            else:
                if context['is_for_all_days']:
                    # В расписании на все дни показываем строку дней курсирования. группируем по ней
                    days_texts = schedule_route.L_days_text()
                else:
                    # А в расписании на дату все времена хождения собираем в одну группу
                    days_texts = ''

                schedule_routes_by_days_texts[days_texts].append(schedule_route)

        calendars = []
        for days_texts, schedule_routes in schedule_routes_by_days_texts.items():
            days_text_ru = schedule_routes[0].L_days_text(lang='ru')
            calendars.append({
                'days_text_ru': days_text_ru,
                'days_text': days_texts,
                'times': sorted(
                    [
                        {
                            'time': schedule_route.event_dt.time(),
                            'is_fuzzy': schedule_route.rtstation.is_fuzzy,
                            'route': schedule_route
                        } for schedule_route in schedule_routes
                    ],
                    key=lambda t: t['time']
                )
            })

        block['calendars'] = calendars

        # Добавим интервальные нитки
        block['calendars'] += [
            {
                'days_text_ru': route.L_days_text(lang='ru'),
                'days_text': route.L_days_text(),
                'times': [],
                'interval': {
                    'begin_time': route.thread.begin_time,
                    'end_time': route.thread.end_time,
                    'density': get_interval_string(route.thread),
                    'comment': route.thread.comment
                },
                'route': route,
            } for route in interval_routes
        ]

        # Сортируем только на все дни. На дату не выводим строку дней курсирования
        if context['is_for_all_days']:
            block['calendars'].sort(key=calendar_sort_keyfunc)

        stations_titles = map(unicode.strip, block['title'].split(title_separator))
        if len(stations_titles) == 2:
            block['station_from'], block['station_to'] = stations_titles

    trace_section.leave()

    trace_section = tracer.enter('sort blocks')

    schedule_blocks.sort(
        key=lambda b: (('0' * 10) + (b['number'] or 'z' * 10 ))[-10:] + b.get('station_to', b['title']),
        cmp=strcoll
    )

    trace_section.leave()

    context['schedule_blocks'] = schedule_blocks
    context['show_blocks_numbers'] = any([block.get('number', False) for block in schedule_blocks])

    context['teasers'] = TeaserSetMorda(request, 'tablo', (
        station, schedule.schedule_routes + interval_schedule.schedule_routes
    ))

    return ScheduleBusBlockTemplate.render(request, context)
