# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging
from datetime import datetime, timedelta

from django.db.models import Q
from django.conf import settings
from django.shortcuts import get_object_or_404
from django.utils import translation

from common.models.geo import StationTerminal, Station
from common.utils.date import MSK_TZ, astimezone
from common.utils.httpresponses import jsonp_wrap, jsonp_response
from common.utils.text import transliterate, punto_variants
from common.xgettext.i18n import xgettext
from travel.rasp.morda.morda.views.mapping.suggests.utils import variants_to_response

from stationschedule import station_schedule, TABLO

from travel.rasp.morda.morda.views.station.slider import slider
from travel.rasp.morda.morda.views.station.title import construct_titles
from travel.rasp.morda.morda.views.station.utils import awaps_params
from travel.rasp.morda.morda.views.teasers import TeaserSetMorda

from travel.rasp.morda.morda.templates.station.tablo import Template as TabloTemplate


log = logging.getLogger(__name__)


def tablo(request, station, event, context):
    try:
        current = datetime.strptime(
            request.GET.get('start', '').split('.')[0],
            '%Y-%m-%dT%H:%M:%S'
        )
    except (ValueError, TypeError):
        current = None

    try:
        span_hrs = int(request.GET.get('span', ''))
    except ValueError:
        span_hrs = None
    else:
        # не учитываем диапазон больше суток
        span_hrs = min(span_hrs, 24)

    if current:
        tablo_datetime = station.localize(loc=current)
    else:
        # Вычитаем SEE_MINUTES_PAST здесь, и сдвигаем на начало текущего часа,
        # чтобы выставить слайдер туда, куда нужно (RASP-2870)
        # FIXME: кажется, описание не соответствует строке
        tablo_datetime = astimezone(request.now - timedelta(minutes=settings.SEE_MINUTES_PAST),
                                    station)

    log.debug('tablo_datetime: %r', tablo_datetime)

    context['current'] = tablo_datetime

    try:
        terminal = StationTerminal.objects.get(id=request.GET['terminal'])
    except (ValueError, KeyError, StationTerminal.DoesNotExist):
        terminal = None

    context['show_terminals'] = not terminal and len(station.stationterminal_set.all())

    context['terminal'] = terminal

    today = request.now.date()

    if terminal:
        context.update(construct_titles(today, station, 'tablo', event, terminal))

    query = request.GET.get('query')

    if span_hrs:
        tablo = station_schedule(station, TABLO, terminal=terminal, event=event,
                                 start_datetime_limit=tablo_datetime,
                                 end_datetime_limit=tablo_datetime + timedelta(hours=span_hrs),
                                 query=query)

    else:
        tablo = station_schedule(station, TABLO, terminal=terminal, event=event,
                                 start_datetime_limit=tablo_datetime,
                                 # Показываем не больше 12 часов
                                 end_datetime_limit=tablo_datetime + timedelta(minutes=settings.SEE_MINUTES_FUTURE),
                                 min_datetime_limit=tablo_datetime + timedelta(hours=4),
                                 min_items_number_limit=settings.DEFAULT_TABLO_LIMIT,
                                 query=query)

        if tablo.amount < settings.MIN_TABLO_AMOUNT:
            # полночь по указанной часовой зоне
            midnight = astimezone(tablo_datetime, context['time_zone']).replace(hour=0, minute=0, second=0)

            tablo_datetime = astimezone(midnight, station)

            tablo = station_schedule(station, TABLO, terminal=terminal, event=event,
                                     start_datetime_limit=tablo_datetime,
                                     end_datetime_limit=tablo_datetime + timedelta(hours=24),
                                     query=query)

    if query and not tablo.schedule:
        # Берем интервал, начиная с полуночи по местному времени
        tablo_datetime = tablo_datetime.replace(hour=0, minute=0, second=0, microsecond=0)

        tablo = station_schedule(station, TABLO, terminal=terminal, event=event,
                                 start_datetime_limit=tablo_datetime,
                                 end_datetime_limit=tablo_datetime + timedelta(hours=24),
                                 query=query)

    log.debug("tablo_datetime: %r", tablo_datetime)

    msk_tablo_datetime = tablo_datetime.astimezone(MSK_TZ).replace(tzinfo=None)

    context['slider'] = slider(request, station, terminal, event, msk_tablo_datetime,
                               tablo.slider_width, context['time_zone'] or station)

    log.debug("len(tablo.schedule) = %r", len(tablo.schedule))

    context['routes'] = tablo.schedule
    context['query'] = query
    context['routes_amount'] = tablo.amount

    if not len(tablo.schedule):
        station_name = '%s-%s' % (station.t_type_id, event)

        kwargs = {
            'feedback_link': lambda content: {
                'block': 'b-link',
                'url': request.feedback_url,
                'content': content
            }
        }

        if station_name == '1-departure':
            if station.in_tablo:
                context['message'] = xgettext(
                    'По данным Яндекс.Расписаний, в выбранное вами время рейсов от вокзала нет.'
                    ' Попробуйте выбрать другое время или'
                    ' <feedback-link>сообщите об ошибке</feedback-link>.',
                    **kwargs
                )

            else:
                context['message'] = xgettext(
                    'По данным Яндекс.Расписаний, на протяжении семи дней до или после запрошенной'
                    ' вами даты рейсов от вокзала нет. Попробуйте уточнить свой запрос или'
                    ' <feedback-link>сообщите об ошибке</feedback-link>.',
                    **kwargs
                )

        elif station_name == '2-departure':
            if station.in_tablo:
                context['message'] = xgettext(
                    'По данным Яндекс.Расписаний, в выбранное вами время рейсов из аэропорта нет.'
                    ' Попробуйте выбрать другое время или'
                    ' <feedback-link>сообщите об ошибке</feedback-link>.',
                    **kwargs
                )

            else:
                context['message'] = xgettext(
                    'По данным Яндекс.Расписаний, на протяжении семи дней до или после запрошенной'
                    ' вами даты рейсов из аэропорта нет. Попробуйте уточнить свой запрос или'
                    ' <feedback-link>сообщите об ошибке</feedback-link>.',
                    **kwargs
                )

        elif station_name == '1-arrival':
            if station.in_tablo:
                context['message'] = xgettext(
                    'По данным Яндекс.Расписаний, в выбранное вами время рейсов до вокзала нет.'
                    ' Попробуйте выбрать другое время или'
                    ' <feedback-link>сообщите об ошибке</feedback-link>.',
                    **kwargs
                )

            else:
                context['message'] = xgettext(
                    'По данным Яндекс.Расписаний, на протяжении семи дней до или после запрошенной'
                    ' вами даты рейсов до вокзала нет. Попробуйте уточнить свой запрос или'
                    ' <feedback-link>сообщите об ошибке</feedback-link>.',
                    **kwargs
                )

        elif station_name == '2-arrival':
            if station.in_tablo:
                context['message'] = xgettext(
                    'По данным Яндекс.Расписаний, в выбранное вами время рейсов из аэропорта нет.'
                    ' Попробуйте выбрать другое время или'
                    ' <feedback-link>сообщите об ошибке</feedback-link>.',
                    **kwargs
                )

            else:
                context['message'] = xgettext(
                    'По данным Яндекс.Расписаний, на протяжении семи дней до или после запрошенной'
                    ' вами даты рейсов из аэропорта нет. Попробуйте уточнить свой запрос или'
                    ' <feedback-link>сообщите об ошибке</feedback-link>.',
                    **kwargs
                )

    if request.is_ajax():
        return jsonp_wrap(request, TabloTemplate.ajax(request, context))

    context['teasers'] = TeaserSetMorda(request, 'tablo', (context['station'], context['routes']))
    context['awaps_params'] = awaps_params('tablo', station, tablo_datetime.date())

    return TabloTemplate.render(request, context)


@jsonp_response
def suggests(request, station_id):
    station = get_object_or_404(Station, pk=station_id)
    event = str(request.GET.get('event', 'departure'))
    query = request.GET.get('part', '').lower()
    variants = list(set([
        query,
        transliterate(query, 'cyr-lat'),
        transliterate(query, 'lat-cyr'),
    ] + punto_variants(query)))

    day_start = request.now.astimezone(station.pytz).replace(hour=0, minute=0, second=0, microsecond=0, tzinfo=None)

    tablo = station.ztablo2_set.filter(
        Q(**{
            '%s__gte' % event: day_start,
            '%s__lte' % event: day_start + timedelta(days=1),
        }) | Q(**{
            'real_%s__gte' % event: day_start,
            'real_%s__lte' % event: day_start + timedelta(days=1),
        })
    )

    suggests = []

    lang = translation.get_language()

    for route in tablo:
        for_search = getattr(route, 'for_search_' + lang)

        suggests.extend(for_search.split("|||")[0].split("|"))

    variants = list(set(suggest for suggest in suggests
                        for variant in variants
                        if variant in suggest.lower()))
    variants.sort(key=lambda s: (s.lower().startswith(query) and 1 or 2, s))

    return variants_to_response(variants[:10])
