# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from datetime import timedelta

from common.utils.locations import change_params
from common.xgettext.i18n import gettext


def smart_sort(context):
    schedule = context['schedule']
    span_code = context['span']

    if span_code not in ['day', 'tomorrow']:
        schedule.sort(key=lambda r: r.event_dt.time())
    else:
        schedule.sort(key=lambda r: r.event_dt)


class Span(object):
    def __init__(self, code, title, once_hint=None, plan=None, **extra):
        self.code = code
        self.title = title
        self.once_hint = once_hint
        self.extra = extra
        self.schedule_plan = plan

    @classmethod
    def schedule_plan_once_hint(cls, schedule_plan):
        """
        Подсказку придумывали про Россию. Когда делали Украину, то не обратили внимания.
        Меняется раз в год, весной, поэтому следующее расписание называтся летним.
        """
        return schedule_plan.code, u'летнее расписание электричек'

    @classmethod
    def get_spans(cls, current_plan=None, next_plan=None):
        spans = [
            Span('day', gettext(u'сегодня')),
            Span('tomorrow', gettext(u'завтра')),
        ]

        if current_plan and next_plan:
            spans.append(Span(current_plan.code, current_plan.name, plan=current_plan))
            spans.append(
                Span(next_plan.code, next_plan.name,
                     once_hint=cls.schedule_plan_once_hint(next_plan), plan=next_plan)
            )

        else:
            # RASP-4405, выбираем текущий график движения по дефолту
            spans.append(Span('schedule', gettext(u'на все дни'), plan=current_plan))

        return spans

    @classmethod
    def get_span_choice(cls, request, spans, default='schedule'):
        spans_by_code = {
            s.code: s for s in spans
        }

        span_code = request.GET.get('span', default)
        default_span = spans_by_code[default]

        span_choice = spans_by_code.get(span_code, default_span)

        return span_choice


class SpanLink(object):
    def __init__(self, span, request, extra_params=None, once_hint=None, current=False):
        self.span = span.code
        self.title = span.title

        params = dict(span=span.code)
        if extra_params:
            params.update(extra_params)

        self.url = change_params(request.GET, params)

        self.current = current

        self.once_hint = once_hint

    @classmethod
    def date_links(cls, request, spans, current_span=None, extra_params=None):
        return [
            cls(_s, request, current=current_span.code == _s.code, extra_params=extra_params,
                once_hint=_s.once_hint)
            for _s in spans
        ]


def fill_spans_context(request, station, context, direction=None,
                       current_plan=None, next_plan=None,
                       default_span_code=None):
    context['next_plan'] = next_plan

    spans = Span.get_spans(current_plan, next_plan)

    if default_span_code:
        pass
    elif current_plan and next_plan:
        default_span_code = current_plan.code
    else:
        default_span_code = 'schedule'

    span = Span.get_span_choice(request, spans, default=default_span_code)

    date_links = SpanLink.date_links(request, spans, current_span=span,
                                     extra_params={'direction': direction, 'subdir': None})

    context.update({
        'span': span.code,
        'date_links': date_links,
        'is_for_all_days': span.code == 'schedule'
    })

    if span.schedule_plan:
        plan = span.schedule_plan
        context['title'] = [context['title'], ' ', plan.name]
        context['page_title'] = [context['page_title'], ' ', plan.name]

    return span


def get_schedule_date_by_span(request, station, span):
    if span.code in ['day', 'tomorrow']:
        local_today = station.local_time(request.now).date()

        return local_today if span.code == 'day' else local_today + timedelta(days=1)


def awaps_params(page, station, date):
    params = {
        'page': page,
        'station_id': station.id,
        'station_title': station.title.encode('utf8'),
        'ttype': station.t_type.code,
        'station_majority': station.majority_id
    }

    settlement = station.settlement
    if settlement:
        params.update({
            'settlement_id': settlement._geo_id,
            'settlement_title': settlement.title,
            'settlement_majority' : settlement.majority_id
        })

    if date:
        params['when'] = date.strftime('%Y-%m-%d')

    return params
