# -*- coding: utf-8 -*-

from django.conf import settings
from django.http import HttpResponseRedirect
from django.utils.translation import get_language

from common.models.geo import Country, Station
from common.utils.mysql_try_hard import mysql_try_hard
from common.xgettext.i18n import gettext, xgettext

from geosearch.forms import StationForm
from geosearch.views.station_search import get_cities, get_station_variants

from travel.rasp.morda.morda.templates.station_search import Template, CountryTemplate
from travel.rasp.morda.morda.utils.locations import get_point_url


@mysql_try_hard
def search(request):
    """ Поиск станции по названию """

    form = StationForm(
        request.GET,
        country_id=settings.LANG_COUNTRY_MAP.get(request.LANGUAGE_CODE, 225)
    )

    context = {
        'station_form': form,
        'title': gettext(u'Поиск станций'),
        # 'teasers': TeaserSet(request, 'rasp', 'other'),
        'city_from': request.GET.get('cityFrom', ''),
        'variants': None
    }

    country_id = settings.NATIONAL_VERSION_COUNTRY_MAP.get(request.NATIONAL_VERSION, 225)

    if country_id == 225:
        current_country_name = gettext(u'РФ')
    else:
        current_country = Country.objects.get(id=country_id)
        current_country_name = current_country.L_title()

    context['current_country_name'] = current_country_name

    if not form.is_valid():
        errors = form.non_field_errors()

        if errors:
            context['error'] = errors[0]

        return Template.render(request, context)

    points = form.cleaned_data['points']
    point = points.point

    if point is None:
        # Поиск по названию
        variants = get_station_variants(
            points,
            request.client_city,
            implicit=request.GET.get('implicit'),
            point_url_func=get_point_url
        )

        if len(variants) == 1 and len(variants[0][1]) == 1:
            point = variants[0][1][0]
        else:
            context['variants'] = variants

            ttypes = set(
                point.t_type.code for title, stations in variants for point in stations if isinstance(point, Station))
            context['awaps_params'] = {'page': 'station_search', 'query': form.cleaned_data['cityFrom'],
                                       'ttypes': ','.join(ttypes)}

            return Template.render(request, context)

    if isinstance(point, Country):
        return show_country_points(request, context, point)

    return HttpResponseRedirect(get_point_url(point))


def show_country_points(request, context, country):
    context['airports'] = get_cities(country)
    context['country'] = country

    # Пример фразы "Наиболее популярные города, вокзалы и аэропорты Великобритании"
    airports = gettext(u'аэропорты')
    # Пример фразы "Наиболее популярные города, вокзалы и аэропорты Великобритании"
    auto_stations = gettext(u'автовокзалы')
    # Пример фразы "Наиболее популярные города, вокзалы и аэропорты Великобритании"
    train_stations = gettext(u'вокзалы')
    # Пример фразы "Наиболее популярные города, вокзалы и аэропорты Великобритании"
    ports = gettext(u'порты')

    s_types = {airports: (9,), auto_stations: (10,), train_stations: (8,), ports: (15, 16, 17,)}
    current_s_types_ids = set([s.station_type_id for v in context['airports'] for s in v[1]])
    s_names = [name for name, ids in s_types.items() if any(id in current_s_types_ids for id in ids)]

    s_names = [xgettext(u', <name/>', name=name) for name in s_names[:-1]] +\
              [xgettext(u' и <name/>', name=name) for name in s_names[-1:]]

    if get_language() != 'ru':
        # Пример фразы "Наиболее популярные города, вокзалы и аэропорты Великобритании"
        context['precise_title'] = xgettext(
            u'Наиболее популярные города <names/> <title case="locative"/>',
            names=s_names, title=country.L_title)
    else:
        # Пример фразы "Наиболее популярные города, вокзалы и аэропорты в Великобритании"
        context['precise_title'] = xgettext(
            u'Наиболее популярные города <names/> <title case="preposition_v_vo_na"/> <title case="locative"/>',
            names=s_names, title=country.L_title)

    return CountryTemplate.render(request, context)
