# -*- coding: utf-8 -*-

from __future__ import absolute_import

import logging
import re
from collections import defaultdict
from datetime import datetime, date, timedelta, time
from itertools import groupby

from django.conf import settings
from django.http import HttpResponse, HttpResponseRedirect, HttpResponsePermanentRedirect, Http404
from django.utils.http import urlencode
from django.utils.translation import get_language, ugettext

from common.forms.thread import RouteForm, ThreadForm
from common.models.compatibility import RThreadUidMap
from common.models.schedule import Company, RTStation, RThread, RThreadType, Route, RunMask, TrainSchedulePlan
from common.models.transport import TransportType
from common.models_utils import fetch_related
from common.models_utils.i18n import RouteLTitle
from travel.rasp.library.python.common23.date import environment
from common.utils.date import human_date, human_date_without_year, group_days, MSK_TZ
from common.utils.fields import ThreadCalendarWidget
from common.utils.locations import composeurl
from common.utils.mysql_try_hard import mysql_try_hard
from common.utils.text import transliterate
from common.views.currency import fetch_currency_info
from common.views.thread import (
    get_thread_context, get_thread_departure_date, ThreadDataError,
    ThreadWithoutDaysError, ThreadWithoutStationsError, calc_thread_start_date
)
from common.utils.httpresponses import jsonp_response
from common.views.tariffs import DisplayInfo
from common.views.timezones import fill_tz_context
from common.xgettext.i18n import xgettext, gettext, mark_gettext, dynamic_gettext
from mapping.models import RouteMapBlacklist
from mapping.views.paths import draw_path
from travel.rasp.morda.morda.tariffs.views import thread_tariffs, fill_tariffs_info
from travel.rasp.morda.morda.templates.thread.trip import Template as TripTemplate
from travel.rasp.morda.morda.templates.thread.trip_bus import Template as TripBusTemplate
from travel.rasp.morda.morda.templates.thread.trip_interval import Template as TripIntervalTemplate
from travel.rasp.morda.morda.templates.thread_search import OrdinalSearchTemplate
from travel.rasp.morda.morda.utils.locations import get_search_url
from common.utils.avia import get_avia_thread_url
from travel.rasp.morda.morda.views.teasers import TeaserSetMorda
from route_search.models import RThreadSegment
from stationschedule.models import ZTablo2
from stationschedule.utils import EVENT_ARRIVAL, EVENT_DEPARTURE


log = logging.getLogger(__name__)


def _thread_context(request, uid):
    # RASPFRONT-114 В урле должна быть информация о типе транспорта
    redirect = add_ttype_info_in_url(request, uid)

    if redirect is not None:
        return redirect, False

    form = ThreadForm(request.GET)
    form.is_valid()
    context = form.cleaned_data

    if 'number' in request.GET:
        context['route_form'] = RouteForm(initial=request.GET)

    result = get_thread_data_or_fail_response(request, uid, context, form)
    if isinstance(result, HttpResponse):
        return result, False

    context['thread'] = result

    fill_thread_context(context, request, form)

    # TODO включить когда будет url('widget_route')
    #if context['first_station'] and context['last_station'] and context['segment_departure'] and not context['thread'].get('is_interval'):
    #    context['widget_link'] = url('widget_route') + '?' + urlencode({
    #        'station_from': context['first_station'].id,
    #        'station_to': context['last_station'].id,
    #        'thread': context['thread']['thread_uid'],
    #        'date': context['segment_departure'].date()
    #    })

    #    context['widget_link_t_type'] = context['thread']['route'].t_type.code

    return context, True


@mysql_try_hard
def thread(request, uid):
    """Страница информации о нитке"""

    context, has_context = _thread_context(request, uid)

    if not has_context:
        return context

    if context['thread'].get('is_interval', False):
        # Интервальные нитки показываем только в местном времени (во времени станции отправления)
        context['timezones'] = context['timezones'][:1]
        context['tz_cities'] = context['tz_cities'][:1]
        return TripIntervalTemplate.render(request, context)

    # RASP-9032
    if context['thread']['thread'].supplier.code == 'mta':
        return TripBusTemplate.render(request, context)

    return TripTemplate.render(request, context)


def add_ttype_info_in_url(request, uid):
    if 'tt' in request.GET:
        return None

    try:
        thread = RThread.objects.all().select_related('t_type').get(uid=uid)
    except RThread.DoesNotExist:
        return None

    #  для самолетов делаем редирект на авиа, поэтому этот шаг пропускаем
    if thread.t_type.id == TransportType.PLANE_ID:
        return None

    params = {k: v for k, v in request.GET.iteritems()}
    params['tt'] = thread.t_type.code

    url = composeurl('thread', args=[uid], params=params)

    return HttpResponseRedirect(url)


def get_thread_data_or_fail_response(request, uid, context, form):
    request_log = logging.getLogger('django.request')
    try:
        thread = RThread.objects.all().select_related().get(uid=uid)
        if thread.type.code == 'cancel':
            raise Http404

        if context.get('departure_from') and context.get('station_from'):
            context['departure'] = calc_thread_start_date(
                thread, context['station_from'], context['departure_from']
            )
        all_days = not context['departure']
        departure_date = get_thread_departure_date(context['departure'], thread)

        if request.GET.get('from', '').startswith('wrasp'):
            context['departure'] = datetime.combine(departure_date, time(0))

        if thread.t_type_id == TransportType.PLANE_ID:
            return HttpResponsePermanentRedirect(get_avia_thread_url(
                thread, tld=request.tld,
                day=context['departure'] if context['departure_from'] else None,
                utm_medium='flight_landing', utm_campaign='redirect301'
            ))

        thread_data = get_thread_context(thread, departure_date, all_days)

        return thread_data

    except (ThreadWithoutDaysError, ThreadWithoutStationsError):
        request_log.warning(u"Не ошибка в нитке", exc_info=1)

        return OrdinalSearchTemplate.render(request, context)

    except (RThread.DoesNotExist, ThreadDataError):
        # Рейс не найден
        redirect = get_redirect_by_old_thread_uid(request, uid)
        if redirect is not None:
            return redirect

        redirect = get_redirect_to_similar_existed_thread(request, uid)
        if redirect is not None:
            return redirect

        redirect = get_redirect_to_plane_thread_number_search(uid)
        if redirect is not None:
            return redirect

        request_log.warning(u"Не нашли старый маршрут или похожий по uid %s", uid, exc_info=1)

        if form.cleaned_data['station_from'] and form.cleaned_data['station_to']:
            # В поиск
            station_from = form.cleaned_data['station_from']
            station_to = form.cleaned_data['station_to']

            return HttpResponseRedirect(search_url(station_from, station_to, request.GET.get('departure')))

        else:
            if 'from_search' in request.GET:
                return OrdinalSearchTemplate.render(request, context)

            else:
                # на страницу поиска рейсов
                number = request.GET.get('number')
                if not number:
                    if '_' in uid:
                        number = uid.split('_')[0]
                if not number or number == 'empty':
                    number = uid

                return HttpResponseRedirect(composeurl('search_threads', params={'number': number}))


def search_url(city_from, city_to, date_forward=None):
    params = {
        'fromName': city_from.L_title(),
        'fromId': city_from.point_key,
        'toName': city_to.L_title(),
        'toId': city_to.point_key,
    }

    if date_forward:
        if isinstance(date_forward, date) or isinstance(date_forward, datetime):
            params['when'] = human_date_without_year(date_forward)

        else:
            params['when'] = date_forward

    # Убираем None
    params = dict((k, v) for k, v in params.items() if v is not None)

    return get_search_url(params=params)


def get_redirect_to_similar_existed_thread(request, uid):
    route_uid = get_route_uid_by_thread_uid(uid)
    if not route_uid:
        return

    try:
        route = Route.objects.filter(route_uid=route_uid, hidden=False)[0]
    except IndexError:
        return

    for thread in route.rthread_set.all():
        if not thread.hidden and thread.type_id in (RThreadType.BASIC_ID, RThreadType.INTERVAL_ID):
            log.info(u'Нашли маршрут похожий по uid. Старый uid %s, новый uid %s.', uid, thread.uid)
            return HttpResponsePermanentRedirect(composeurl('thread', args=[thread.uid], params=request.GET))


def get_route_uid_by_thread_uid(thread_uid):
    if '_' not in thread_uid:
        return

    parts = thread_uid.split('_')

    if 'as-' in parts[-1]:
        if len(parts) > 2:
            parts = parts[:-2]
        else:
            return

    return '_'.join(parts[:1] + parts[2:])


def get_redirect_to_plane_thread_number_search(uid):
    def is_plane_uid(uid):
        return (
            uid.endswith('agent') or uid.endswith('_5') or
            uid.endswith('oag') or uid.endswith('_12')
        )

    def get_company(uid):
        companies = Company.objects.all()

        for c in companies:
            c.lat_sirena = transliterate(c.sirena_id or '', 'cyr-lat')

        for i in xrange(1, 5):
            for c in companies:
                if c.sirena_id == uid[:i] or c.lat_sirena == uid[:i] :
                    return c

        return None

    def make_new_number(uid):
        company = get_company(uid)
        if not company:
            return

        number = uid.split('_')[0][len(company.lat_sirena):]
        number = re.sub('\D', '', number)

        return gen_avia_number(company, number)

    if is_plane_uid(uid):
        new_number = make_new_number(uid)
        if new_number:
            log.info(u'Нашли маршрут по самолетному номеру. Старый uid %s, новый номер %s.', uid, new_number)
            return HttpResponseRedirect(composeurl('search_threads', params={'number': new_number}))


def gen_avia_number(company, number):
    if isinstance(company, Company):
        code = company.iata or company.icao or company.sirena_id
    else:
        code = company

    number = code + u" " + number

    return number


def get_redirect_by_old_thread_uid(request, old_uid):
    try:
        new_uid = RThreadUidMap.objects.filter(old_uid=old_uid)[0].new_uid
        log.info(u'Нашли маршрут по RThreadUidMap. Старый uid %s, новый uid %s.', old_uid, new_uid)
        return HttpResponsePermanentRedirect(composeurl('thread', args=[new_uid], params=request.GET))

    except IndexError:
        pass


def build_segment(thread, loc_departure_dt, loc_arrival_dt,
                  thread_start_date, station_from, station_to, rtstation_from, rtstation_to):
    segment = RThreadSegment()

    segment.station_from = station_from
    segment.station_to = station_to

    segment.thread = thread

    segment.departure = loc_departure_dt
    segment.arrival = loc_arrival_dt

    segment.start_date = thread_start_date

    segment.rtstation_from = rtstation_from
    segment.rtstation_to = rtstation_to

    segment.stops_translations = []

    segment.now = MSK_TZ.localize(environment.now())

    segment.display_info = DisplayInfo()

    segment._init_data()

    return segment


def fill_thread_context(context, request, form):
    context['thread']['many_threads'] = len(context['thread']['route_threads']) > 1

    t_type = TransportType.objects.get(pk=context['thread']['transportType'])

    context['canonical_url'] = composeurl('thread', args=[context['thread']['thread_uid']], params={'tt': t_type.code})

    if t_type.code != 'plane':
        add_tz_context(request, context)

    # Убираем лишние станции
    if context['thread']['thread'].express_type:
        context['thread']['stops'] = [st for st in context['thread']['stops'] if st['arrival'] != st['departure']]

    context['first_station'] = context['station_from'] or context['thread']['stops'][0]['station']
    context['last_station'] = context['station_to'] or context['thread']['stops'][-1]['station']

    # В интервальных нитках не учитываем поисковый контекст
    if context['thread'].get('is_interval', False):
        if context['first_station'] != context['thread']['stops'][0]['station']:
            context['first_station'] = context['thread']['stops'][0]['station']
            context['last_station'] = context['thread']['stops'][-1]['station']

    first_station_index = None
    last_station_index = None

    for i, st in enumerate(context['thread']['stops']):
        if st['station'] == context['first_station']:
            first_station_index = i

        if first_station_index is not None and st['station'] == context['last_station']:
            last_station_index = i
            break

    if first_station_index is None:
        first_station_index = 0
    if last_station_index is None:
        last_station_index = len(context['thread']['stops']) - 1

    round_trip = first_station_index == last_station_index

    # Время отправления сегмента
    segment_departure = context['thread']['stops'][first_station_index]['departure']
    segment_arrival = context['thread']['stops'][last_station_index]['arrival']

    for i, st in enumerate(context['thread']['stops']):
        st['duration_from_start'] = st['duration']
        st['class'] = ''

        if i == first_station_index:
            st['class'] = 'mark'
            st['is_first'] = True

        elif i == last_station_index:
            st['class'] = 'mark'
            st['is_last'] = True

        else:
            in_interval = first_station_index <= i <= last_station_index

            if not in_interval and not round_trip:
                st['class'] = 'gray'

        if i > first_station_index and segment_departure:
            st['duration'] = st['arrival'] and st['arrival'] - segment_departure
        else:
            st['duration'] = ''

        if st['station'].time_zone and st['station'].time_zone != 'Europe/Moscow':
            context['different_zones'] = True

        if st['is_technical_stop']:
            context['technical_stops'] = True

    # Положение поезда
    stations = context['thread']['stops']

    now_aware = environment.now_aware()

    if (
        stations[0]['departure'] - now_aware < timedelta(hours=1) and
        now_aware - stations[-1]['arrival'] < timedelta(hours=1)
    ):
        found = -1
        for i, st in enumerate(stations):
            st_time = st['departure'] or st['arrival']
            if now_aware < st_time:
                found = i
                break

        time_class = {0: 'station-from-current',
                      len(stations)-1: 'station-to-current',
                      -1: 'station-to-finish',
                      }.get(found, 'station-current')
        stations[found]['time_class'] = time_class

    if not round_trip:
        found = False
        for st in reversed(context['thread']['stops']):
            if st['station'] == context['first_station'] and not found:
                found = True
            elif found:
                st['duration'] = ''
                st['class'] = 'gray'

    # RASP-3032
    if t_type.code == 'plane' and context['thread']['thread'].supplier.code != 'oag': # RASP-3521
        add_route_stats(context, request.now)

    if (t_type.code == 'plane' and
       form.cleaned_data['station_from'] and
       form.cleaned_data['station_to'] and
       form.cleaned_data['departure']):

        thread = context['thread']['thread']
        station_from = form.cleaned_data['station_from']
        station_to = form.cleaned_data['station_to']

        when = context['departure']
        start_dt = datetime.combine(when, thread.tz_start_time)

        rts_from = stations[first_station_index]['rtstation']
        rts_to = stations[last_station_index]['rtstation']

        loc_dep_dt_aware = rts_from.get_loc_departure_dt(start_dt)
        loc_arr_dt_aware = rts_to.get_loc_arrival_dt(start_dt)

        segment = build_segment(
            thread=thread,
            loc_departure_dt=loc_dep_dt_aware,
            loc_arrival_dt=loc_arr_dt_aware,
            thread_start_date=start_dt,
            station_from=station_from,
            station_to=station_to,
            rtstation_from=rts_from,
            rtstation_to=rts_to
        )

        fill_tariffs_info(segments=[segment],
                          currency_info=fetch_currency_info(request),
                          supplement=[],
                          point_from=form.cleaned_data['point_from'],
                          point_to=form.cleaned_data['point_to'],
                          when=when,
                          request=request,
                          allow_blablacar=False)

        context['display_info'] = segment.display_info
        context['plane_price_fetch_url'] = composeurl('plane_thread_price', args=[
            thread.uid], params=dict(request.GET)
        )

    current_thread = context['thread']['thread']

    if RouteMapBlacklist.is_thread_mapped(context['thread']['thread']):
        thread = context['thread']['thread']

        context['route_data'] = draw_path(
            thread,
            context['thread']['stops'][0]['departure'],
            list(thread.path.select_related('station')),
            first=context['first_station'],
            last=context['last_station']
        )

    # Легенда для календаря
    thread_tz_today = now_aware.astimezone(current_thread.pytz).date()

    is_basic_thread = current_thread.type.code == 'basic'

    first_station = context['thread']['stops'][0]['station']

    start_dt = context['thread']['stops'][0]['departure']

    departure_date = start_dt.astimezone(current_thread.pytz).date()
    calendar_date = start_dt.astimezone(first_station.pytz).date()
    shift = (calendar_date - departure_date).days
    context['shift'] = shift
    lang = get_language()

    cancel_days = u''
    cancel_dates = set()

    cancel_threads = [t for t in context['thread']['route_threads'] if t.type.code == 'cancel']

    for cancel_thread in cancel_threads:
        for d in RunMask(cancel_thread.year_days, today=thread_tz_today).dates():
            cancel_dates.add(d + timedelta(days=shift))

    cancel_dates = sorted(cancel_dates)

    if cancel_dates:
        cancel_days = group_days([(d.day, d.month) for d in cancel_dates], lang)

    context['cancel_days'] = {
        'local': cancel_days
    }

    context['current_thread_days'] = [d + timedelta(days=shift) for d in
                                      RunMask(current_thread.year_days, today=thread_tz_today).dates()]

    dep = context['thread']['stops'][0]['departure']
    for tz_city in context.get('tz_cities', []):
        local_date = dep.astimezone(tz_city.pytz).date()

        shift = (local_date - departure_date).days

        context['current_thread_days_%s' % tz_city.id] = \
            [d + timedelta(days=shift) for d in RunMask(current_thread.year_days, today=thread_tz_today).dates()]

        cancel_dates = set()

        for cancel_thread in cancel_threads:
            for d in RunMask(cancel_thread.year_days, today=thread_tz_today).dates():
                cancel_dates.add(d + timedelta(days=shift))

        cancel_dates = sorted(cancel_dates)

        if cancel_dates:
            context['cancel_days'][str(tz_city.id)] = group_days([(d.day, d.month) for d in cancel_dates], lang)

    def _sort_key(thread):
        try:
            return thread.get_run_date_list()[0]
        except IndexError:
            return

    change_other_threads = sorted([t for t in context['thread']['route_threads'] if context['thread']['thread_id'] != t.id and
                                                                                    t.type.code in ('change', 'assignment') and
                                                                                    len(t.get_run_date_list())], key=_sort_key)

    base_other_threads = sorted([t for t in context['thread']['route_threads'] if t.id not in (context['thread']['thread_id'], current_thread.basic_thread_id) and
                                                                                  t.type.code == 'basic' and
                                                                                  len(t.get_run_date_list())], key=_sort_key)

    through_train_threads = sorted([t for t in context['thread']['route_threads'] if t.type.code == 'through_train' and
                                                                                     t.id != current_thread.id and
                                                                                     len(t.get_run_date_list())], key=_sort_key)

    legend_threads = filter(None, [
        current_thread,
    ] + base_other_threads + change_other_threads + [
        current_thread.basic_thread,
    ] + through_train_threads)

    edge_rtstations = threads_add_bulk_departure_arrival(legend_threads, context['departure'])

    RouteLTitle.fetch(thread.L_title for thread in legend_threads)

    if is_basic_thread:
        if t_type.code in ('plane', 'train', 'helicopter'):
            current_block_title = gettext(u'Данное расписание')
        else:
            current_block_title = gettext(u'Основное расписание')
    elif current_thread.type.code == 'through_train':
        current_block_title = gettext(u'Беспересадочные вагоны')
    else:
        current_block_title = gettext(u'Данное изменение')

    cal_legend_blocks = {
        'current': {  # блок основной нитки
            'type': is_basic_thread and 'base' or 'change',
            'title': current_block_title,
            'threads': [current_thread]
        },
        'basic': {  # блок других базовых ниток
            'type': current_thread.type.code != 'through_train' and 'base-other' or 'base',
            'title': current_thread.type.code in ('through_train', 'assignment') and gettext(u'Основное расписание') or gettext(u'Другие варианты расписания'),
            'threads': base_other_threads
        },
        'change': {  # блок изменений
            'type': 'change-other',
            'title': is_basic_thread and gettext(u'Изменения') or gettext(u'Другие изменения'),
            'threads': change_other_threads
        },
        'cancel': {  # блок отмен
            'type': 'cancel',
            'title': gettext(u'Отменен <br/>на полном маршруте'),
            'text': cancel_days
        },
        'change-basic': {  # блок базовой нитки изменения
            'type': 'base',
            'title': gettext(u'Основное расписание'),
            'threads': not is_basic_thread and current_thread.basic_thread and [current_thread.basic_thread] or []
        },
        'through_train': {
            'type': 'change-other',
            'title': current_thread.type.code == 'through_train' and gettext(u'Другие беспересадочные вагоны') or gettext(u'Беспересадочные вагоны'),
            'threads': through_train_threads,
        }
    }

    if is_basic_thread:
        # блок текущего базового расписания, блоки других базовых расписаний, блок отмен, блоки изменений
        cal_legend_order = ('current', 'basic', 'cancel', 'change', 'through_train')
    else:
        # блок текущего изменения, блоки других изменений, блок отмен, блок базовой нитки изменения,
        # блоки других базовых ниток рейса, блок беспересадочных ниток
        cal_legend_order = ('current', 'change', 'cancel', 'change-basic', 'basic', 'through_train',)

    context['cal_legend_order'] = cal_legend_order
    context['cal_legend_blocks'] = cal_legend_blocks

    start_rtstations = [rt for rt in edge_rtstations if rt.tz_arrival is None]

    context['calendar'] = thread_calendar(
        thread_tz_today,
        context['thread']['t_type'].code,
        context['thread']['thread'],
        context['thread']['route_threads'],
        calendar_date,
        12,
        rts=start_rtstations
    )

    if len(context.get('timezones', [])) >= 2:
        context['tz_calendars'] = {}
        for tz_city in context.get('tz_cities', []):
            context['tz_calendars']['calendar_%s' % tz_city.id] = thread_calendar(
                thread_tz_today,
                context['thread']['t_type'].code,
                context['thread']['thread'],
                context['thread']['route_threads'],
                start_dt.astimezone(tz_city.pytz).date(),
                12,
                rts=start_rtstations,
                city=tz_city
            )

    # RASP-12577
    if not context['departure']:
        context['calendar']['current'] = None
        for key in context.get('tz_calendars', []):
            context['tz_calendars'][key]['current'] = None

    context['title'] = get_thread_title(context['thread'])

    context['model'] = get_thread_model(context['thread'])
    context['segment_departure'] = segment_departure
    context['weather_stations'] = list(get_weather_stations(context))
    context['teasers'] = TeaserSetMorda(
        request,
        'thread_%s' % context['thread']['t_type'].code,
        (context['thread']['thread'], context['thread']['thread'].company)
    )
    context['show_print'] = True

    context['short_path'] = len(context['thread']['stops']) == 2

    context['currency_info'] = fetch_currency_info(request)

    context['counter'] = settings.COUNTERS.get(context['thread']['thread'].t_type_id, 0)

    context['awaps_params'] = awaps_params(context, t_type.code,
                                           form.cleaned_data['station_from'], form.cleaned_data['station_to'], segment_departure)

    if segment_departure and form.cleaned_data.get('point_from') and form.cleaned_data.get('point_to')\
       and form.cleaned_data.get('station_from') and form.cleaned_data.get('station_to'): # Без этого цену вычислить нельзя
        context['tariffs'] = thread_tariffs(
            request,
            form.cleaned_data.get('point_from'),
            form.cleaned_data.get('point_to'),
            form.cleaned_data.get('station_from'),
            form.cleaned_data.get('station_to'),
            context['thread']['thread'],
            segment_departure,
            )

    if segment_departure and t_type.code == 'plane':
        context.update(ticket_link(request, context['first_station'], context['last_station'], segment_departure))

    context['next_plan'] = TrainSchedulePlan.get_current_and_next(thread_tz_today)[1]


def awaps_params(context, t_type_code, station_from, station_to, segment_departure):
    stop_count = len(context['thread']['stops'])

    if not station_from and stop_count == 2:
        station_from = context['first_station']

    if not station_to and stop_count == 2:
        station_to = context['last_station']

    params = {
        'page': 'thread',
        'number': context['thread']['thread'].number,
        'ttype': t_type_code,
        'when': segment_departure and segment_departure.strftime('%Y-%m-%d') or '',
    }

    if station_from:
        params.update(station_awaps_params(station_from, 'from'))

    if station_to:
        params.update(station_awaps_params(station_to, 'to'))

    return params


def station_awaps_params(station, direction):
    params = {
        'station_%s_id' % direction:  station.id,
        'station_%s_title' % direction: station.title,
        'station_%s_ttype' % direction: station.t_type.code,
        'station_%s_majority' % direction: station.majority_id
    }

    settlement = station.settlement

    if settlement:
        params.update({
            'settlement_%s_id' % direction: settlement._geo_id,
            'settlement_%s_title' % direction: settlement.title,
            'settlement_%s_majority' % direction: settlement.majority_id
        })

    return params


TRANSPORT_STATUSES = {
    'current': {
         'departure': {
             'wait': mark_gettext(u'ожидается'),
             'late': mark_gettext(u'задерживается'),
             'departure': mark_gettext(u'отправлен'),
             'arrive': mark_gettext(u'прибыл'),
             'cancelled': mark_gettext(u'отменён'),
             'unknown': mark_gettext(u'неизвестен'),
             'rasch': mark_gettext(u'отправлен'),
             'nodata': mark_gettext(u'нет данных')
         },
         'arrival': {
             'wait': mark_gettext(u'ожидается'),
             'late': mark_gettext(u'опаздывает'),
             'departure': mark_gettext(u'отправлен'),
             'arrive': mark_gettext(u'прибыл'),
             'cancelled': mark_gettext(u'отменён'),
             'unknown': mark_gettext(u'неизвестен'),
             'rasch': mark_gettext(u'прибыл'),
             'nodata': mark_gettext(u'нет данных')
         }
    },
    'future': {
        'departure': {
            'wait': mark_gettext(u'ожидается'),
            'late': mark_gettext(u'задержится'),
            'departure': mark_gettext(u'отправится'),
            'arrive': mark_gettext(u'прибудет'),
            'cancelled': mark_gettext(u'отменён'),
            'unknown': mark_gettext(u'неизвестен'),
            'rasch': mark_gettext(u'отправлен'),
            'nodata': mark_gettext(u'нет данных')
        },
        'arrival': {
            'wait': mark_gettext(u'ожидается'),
            'late': mark_gettext(u'опоздает'),
            'departure': mark_gettext(u'отправится'),
            'arrive': mark_gettext(u'прибудет'),
            'cancelled': mark_gettext(u'отменён'),
            'unknown': mark_gettext(u'неизвестен'),
            'rasch': mark_gettext(u'прибыл'),
            'nodata': mark_gettext(u'нет данных')
        }
    }
}


def add_route_stats(context, now_aware):
    u"""Статистика рейса на три дня"""
    records = list(context['thread']['thread'].ztablo2_set.order_by('utc_start_datetime').prefetch_related('station'))
    if not records:
        return

    state_records = defaultdict(list)
    for _utc_start_datetime, records in groupby(records, key=lambda r: r.utc_start_datetime):
        records = list(records)
        state_records[get_thread_state(records, now_aware)].append(records)

    stats = {}
    for state in ('past', 'current', 'future'):
        if not state_records[state]:
            continue

        times = []
        for record in sorted(
            state_records[state][-1 if state == 'past' else 0],
            key=lambda r: r.get_arrival_dt() or r.get_departure_dt()
        ):
            for event in (EVENT_ARRIVAL, EVENT_DEPARTURE):
                planned_dt = record.get_event_dt(event, ZTablo2.PLANNED)
                if planned_dt is None:
                    continue

                actual_dt = record.get_event_dt(event, ZTablo2.ACTUAL)
                status_code = record.get_status(event, now_aware)[0]
                times.append({
                    'real': actual_dt,
                    'delta': actual_dt - planned_dt,
                    'event': event,
                    'state': state,
                    'status': status_code,
                    'state_phrase': dynamic_gettext(
                        TRANSPORT_STATUSES['future' if actual_dt > now_aware else 'current'][event][status_code]
                    ),
                    'is_bad_status': status_code in ('late', 'cancelled', 'unknown'),
                    'station': record.station,
                })
        stats[state] = times

    if all(t['status'] == 'nodata' for times in stats.values() for t in times):
        # нет никаких данных
        return

    stats['amount'] = len(context['thread']['stops'])
    if len(context['thread']['stops']) == 3:
        context['thread']['stops'][1]['marked'] = True

    context['stats'] = stats


def get_thread_state(rows, now_aware):
    u"""Определяет состяние рейса: прошедший, текущий, будущий"""
    if len(rows) < 2:
        return 'unknown'

    departure = min(r.get_departure_dt() for r in rows if r.departure)
    arrival = max(r.get_arrival_dt() for r in rows if r.arrival)

    if departure <= now_aware <= arrival:
        return 'current'

    elif now_aware < departure:
        return 'future'

    else:
        return 'past'


def get_thread_title(context_thread):
    thread = context_thread['thread']
    number = thread.number

    kwargs = {
        'number': number,
        'title': context_thread['l-title']
    }

    if thread.is_combined and number:
        title = xgettext(u'№<number/> <title/>', **kwargs)

    elif thread.t_type_id == TransportType.BUS_ID:
        if thread.supplier.code == 'mta' and thread.number:
            title = xgettext(u"Маршрут автобуса №<number/> <title/>", **kwargs)

        else:
            title = xgettext(u"Маршрут автобуса <title/>", **kwargs)

    elif thread.t_type_id in (TransportType.PLANE_ID, TransportType.HELICOPTER_ID):
        kwargs['number'] = number
        stations = [st['station'].settlement and st['station'].settlement.L_title() or st['station'].L_title()
                                for st in context_thread['stops']]
        kwargs['stations'] = u"&nbsp;&mdash; ".join(stations)

        if context_thread.get('merged_number'):
            kwargs['merged_number'] = context_thread['merged_number']

            title = xgettext(u"Рейс <number/> &nbsp;&mdash; <merged-number/>, <stations/>", **kwargs)

        else:
            title = xgettext(u"Рейс <number/>, <stations/>", **kwargs)

    elif thread.t_type_id in TransportType.WATER_TTYPE_IDS:
        title = xgettext(u"Теплоход <title/>", **kwargs)

    else:
        title_special = thread.L_title_special()

        kwargs['special_title'] = title_special

        if thread.type.code == 'through_train':
            title = xgettext(u"Беспересадочный вагон <number/>, <title/>", **kwargs)

        elif title_special and thread.is_deluxe:
            if number:
                title = xgettext(u"Фирменный поезд «<special-title/>» <number/>, <title/>", **kwargs)

            else:
                title = xgettext(u"Фирменный поезд «<special-title/>», <title/>", **kwargs)

        elif thread.is_deluxe:
            if number:
                title = xgettext(u"Фирменный поезд <number/>, <title/>", **kwargs)

            else:
                title = xgettext(u"Фирменный поезд, <title/>", **kwargs)

        elif title_special:
            if number:
                title = xgettext(u"Поезд «<special-title/>» <number/>, <title/>", **kwargs)

            else:
                title = xgettext(u"Поезд «<special-title/>», <title/>", **kwargs)

        else:
            if number:
                title = xgettext(u"Поезд <number/>, <title/>", **kwargs)

            else:
                title = xgettext(u"Поезд <title/>", **kwargs)

    return title


def get_thread_model(thread):
    if thread['one_t_model']:
        return thread['t_model'] and thread['t_model'].title or ''

    else:
        models = []
        for st in thread['stops'][:len(thread['stops'])-1]:
            if st['departure_t_model']:
                models.append(st['departure_t_model'].title)

        return ' / '.join(models)


def get_weather_stations(context):
    u"""Выбор станций, погода которых будет показываться"""
    def context_stations(context):
        if context['station_from']:
            yield context['station_from']
        if context['station_to']:
            yield context['station_to']

    if context['thread']['transportType'] == TransportType.PLANE_ID:
        if context['station_from'] or context['station_to']:
            return context_stations(context)
        elif len(context['thread']['stops']) <= 3:
            return [st['station'] for st in context['thread']['stops']]

    if context['station_from'] or context['station_to']:
        return context_stations(context)

    return []


# TODO: возможно нужно перенести в tickets
def ticket_link(request, station_from, station_to, departure):
    if not request.show_ticket_links:
        return {}

    point_from = station_from.settlement or station_from
    point_to = station_to.settlement or station_to

    local_today = point_from.localize(msk=request.now).date()

    when = departure.date()

    if when <= local_today:
        when = local_today + timedelta(days=1)

    params = {
        'fromName': point_from.L_title(),
        'fromId': point_from.point_key,
        'toName': point_to.L_title(),
        'toId': point_to.point_key,
        'adult_seats': 1,
        'children_seats': 0,
        'infanf_seats': 0,
        'klass': 'economy',
        'when': human_date(when),
        }

    url = request.ticket_url + 'variants/?' + urlencode(params)

    return {
        'ticket_date': when,
        'ticket_link': url,
    }


def add_tz_context(request, context):
    stations = []
    for direction in ('from', 'to'):
        # Поисковый контекст в приоритете
        if context['station_%s' % direction]:
            stations.append(context['station_%s' % direction])
    for index in (0, -1):
        # Конечные станции
        stations.append(context['thread']['stops'][index]['station'])

    countries = get_uniq(station['station'].country for station in context['thread']['stops'])

    fill_tz_context(request, context,
                    cities=stations, countries=countries,
                    dates=[context['thread']['departure'].date()])


def threads_add_bulk_departure_arrival(threads, day=None):
    today = environment.today()

    edge_rtstations = list(RTStation.objects.filter(
        thread__in=threads,
    ).exclude(
        tz_departure__isnull=False, tz_arrival__isnull=False, # Берем, только началную, или конечную станцию
    ).select_related('station'))

    threads_by_id = dict((t.id, t) for t in threads)

    for rts in edge_rtstations:
        thread = threads_by_id[rts.thread_id]

        start_date = thread.first_run(today) or day

        naive_start_dt = datetime.combine(start_date, thread.tz_start_time)

        if rts.tz_arrival is None:
            thread.departure = rts.get_loc_departure_dt(naive_start_dt)

        if rts.tz_departure is None:
            thread.arrival = rts.get_loc_arrival_dt(naive_start_dt)

    return edge_rtstations


class CalendarDay(object):
    def __init__(self, thread, msk_date, other_group, color=None):
        self.thread = thread
        self.msk_date = msk_date
        self.other_group = other_group
        self.color = color

    @property
    def link(self):
        return composeurl('thread', args=[self.thread.uid], params={'departure': self.msk_date})


def thread_calendar(thread_tz_today, t_type_code, main_thread, route_threads, calendar_date, month_count, station=None,
                    event='departure', rts=None, city=None):

    # страницу автобусных и водных рейсов не меняем
    if t_type_code in ['bus'] + TransportType.WATER_TTYPE_CODES:
        return thread_bus_calendar(thread_tz_today, t_type_code, main_thread, route_threads, calendar_date, month_count,
                                   station, event, rts)

    calendar_day_by_local_date = {}

    if rts:
        rtstations = rts
    else:
        # RASP-12577
        if main_thread.type.code == 'through_train':
            threads = [main_thread]
        else:
            threads = [main_thread] + [t for t in route_threads if t.type.code != 'through_train']

        rtstations = RTStation.objects.filter(
            thread__in=threads,
            ).select_related('station')

        if station:
            # Если указана станция, берем её
            rtstations = rtstations.filter(station=station)
        else:
            # В противном случае - первую
            rtstations = rtstations.filter(tz_arrival__isnull=True)

    rtstations_by_thread_id = dict((rts.thread_id, rts) for rts in rtstations)

    threads_by_dates = {}

    for thread in route_threads:
        # Нитку, которую смотрим обрабатываем отдельно
        if (thread.type.code in ('cancel', 'through_train')) or (thread.id == main_thread.id) or \
            ((main_thread.route.t_type.code != 'suburban') and (thread.type.code in ('change', 'assignment'))):
            continue

        color = 'base'

        # Если смотрим основную нитку электрички - даты всех ниток-изменений и ниток-назначений рейса светло-красного цвета
        # или Если смотрим нитку-изменение или нитку-назначение - даты всех остальных ниток-изменений и ниток-назначений рейса светло-красного цвета
        if (main_thread.route.t_type.code == 'suburban' and thread.type.code in ('change', 'assignment')) or \
            (main_thread.type.code in ('change', 'assignment') and thread.type.code in ('change', 'assignment')):
            color = 'change'

        if main_thread.type.code in ('change', 'assignment') and main_thread.basic_thread == thread:
            color = 'base-dark'

        start_mask = RunMask(thread.year_days, thread_tz_today)

        try:
            rtstation = rtstations_by_thread_id[thread.id]
        except KeyError:
            # Пропускаем треды, которые не ходят через эту станцию
            continue

        delta = getattr(rtstation, 'tz_' + event)
        if delta is None:
            # Пропускаем, если не делает нужное действие на этой станции
            continue

        for start_date in start_mask.dates():
            naive_start_dt = datetime.combine(start_date, thread.tz_start_time)

            if city:
                local_dt = rtstation.get_event_dt(event, naive_start_dt, city.pytz)
            else:
                local_dt = rtstation.get_loc_event_dt(event, naive_start_dt)

            other_group = False

            # Вставляем только если такого ещё нет
            calendar_day_by_local_date.setdefault(local_dt.date(), CalendarDay(thread, local_dt.date(), other_group, color))
            threads_by_dates.setdefault(local_dt.date(), thread)

    # Обрабатываем нитку, которую смотрим
    main_thread_mask = RunMask(main_thread.year_days, thread_tz_today)

    color = 'base-dark'

    if main_thread.type.code in ('change', 'assignment', 'through_train'):
        color = 'change-dark'

    def get_event_dt(thread, rtstation, start_date, start_dt_if_none=False):
        naive_start_dt = datetime.combine(start_date, thread.tz_start_time)

        if city:
            local_dt = rtstation.get_event_dt(event, naive_start_dt, city.pytz)
        else:
            local_dt = rtstation.get_loc_event_dt(event, naive_start_dt)

        if local_dt is None and start_dt_if_none:
            local_dt = thread.pytz.localize(naive_start_dt)

            if city:
                local_dt = local_dt.astimezone(city.pytz)
            else:
                local_dt = local_dt.astimezone(rtstation.pytz)

        return local_dt

    try:
        rtstation = rtstations_by_thread_id[main_thread.id]

        for start_date in main_thread_mask.dates():
            local_dt = get_event_dt(main_thread, rtstation, start_date, start_dt_if_none=True)

            other_group = False

            # Вставляем только если такого ещё нет
            calendar_day_by_local_date[local_dt.date()] = CalendarDay(main_thread, local_dt.date(), other_group, color)
            threads_by_dates[local_dt.date()] = main_thread
    except KeyError:
        pass

    route_months = []

    for _, dates in groupby(RunMask(today=thread_tz_today).all_days, key=lambda d: d.month):
        dates = list(dates)
        dates_in = sorted([d for d in dates if d in calendar_day_by_local_date])
        last_date = dates_in and dates_in[-1]

        month_data = [None] * 31
        for event_date in dates:

            if event_date in threads_by_dates:
                thread = threads_by_dates[event_date]

            if event_date not in calendar_day_by_local_date:
                continue

            calendar_day = calendar_day_by_local_date[event_date]

            # костыль
            if event_date.month > event_date.month:
                continue

            if station:
                # Если указана станция, то не делать ссылок
                month_data[event_date.day - 1] = True
            else:
                month_data[event_date.day - 1] = calendar_day

        if len(route_months) == month_count:
            if route_months[0][1] == [None] * 31:
                route_months.pop(0)
                route_months.append((dates[0], month_data))
            else:
                break
        else:
            route_months.append((dates[0], month_data))

    template_days = main_thread.L_days_text(
        shift=0,
        except_separator=", ",
        html=False,
        template_only=True,
        thread_start_date=thread_tz_today
    )

    template_days_all = main_thread.L_days_text(
        shift=0,
        except_separator=", ",
        html=False,
        template_only=False,
        thread_start_date=thread_tz_today
    )

    return {
        'months': ThreadCalendarWidget.make_calendar(route_months),
        'title': '',
        'current': calendar_date,
        'current_month': calendar_date.month,
        'template': template_days,
        'template_days': template_days_all,
    }


def thread_bus_calendar(thread_tz_today, t_type, main_thread, route_threads, calendar_date,
                        month_count, station=None, event='departure',
                        rts=None):
    calendar_day_by_local_date = {}

    # Нужно-ли группировать нитки по маршрутам
    # group_by_path = t_type == 'plane'
    group_by_path = t_type == 'plane'

    if group_by_path:
        threads = [main_thread] + route_threads
    else:
        threads = [main_thread]

    if rts:
        rtstations = rts
    else:
        rtstations = RTStation.objects.filter(
            thread__in=threads,
            ).select_related('station')

    if station:
        # Если указана станция, берем её
        if not rts:
            rtstations = rtstations.filter(station=station)

        if t_type == 'plane':
            if event == 'departure':
                title = ugettext(u'Даты вылета из %s') % station.L_title_with_prefix()
            else:
                title = ugettext(u'Даты прилета в %s') % station.L_title_with_prefix()
        else:
            if event == 'departure':
                title = ugettext(u'Даты отправления от %s') % station.L_title_with_prefix()
            else:
                title = ugettext(u'Даты прибытия на %s') % station.L_title_with_prefix()
    else:
        # В противном случае - первую
        if not rts:
            rtstations = rtstations.filter(departure=0)

        if t_type == 'plane':
            title = ugettext(u'Даты вылета')
        elif t_type == 'train':
            title = ugettext(u'Даты следования')
        else:
            title = ugettext(u'Даты курсирования')

    rtstations_by_thread_id = dict((rts.thread_id, rts) for rts in rtstations)

    groups_by_thread = {}
    groups = []
    if group_by_path:
        groups = list(group_plane_threads(main_thread, threads))

        for group in groups:
            for thread_id in group.threads:
                groups_by_thread[thread_id] = group

    for thread in threads:
        thread_mask = RunMask(thread.year_days, thread_tz_today)

        try:
            rtstation = rtstations_by_thread_id[thread.id]
        except KeyError:
            # Пропускаем треды, которые не ходят через эту станцию
            continue

        delta = getattr(rtstation, 'tz_' + event)

        if delta is None:
            # Пропускаем, если не делает нужное действие на этой станции
            continue

        if group_by_path:
            group = groups_by_thread.get(thread.id)

        for start_date in thread_mask.dates():
            naive_start_dt = datetime.combine(start_date, thread.tz_start_time)

            local_dt = rtstation.get_loc_event_dt(event, naive_start_dt)

            if group_by_path and group:
                # Выбор ближайшей даты отправления с этой станции
                # в будущем или прошлом
                if group.dt is None or group.dt <= thread_tz_today:
                    group.dt = local_dt.date()
                    group.start_date = start_date
                    group.thread = thread

                other_group = True
            else:
                other_group = False

            # Вставляем только если такого ещё нет
            calendar_day_by_local_date.setdefault(local_dt.date(), CalendarDay(thread, start_date, other_group))

    mask = RunMask(today=thread_tz_today)

    route_months = []

    for _, dates in groupby(mask.all_days, key=lambda d: d.month):
        dates = list(dates)

        month_data = [None] * 31

        for d in dates:
            try:
                calendar_day = calendar_day_by_local_date[d]

                if station:
                    # Если указана станция, то не делать ссылок
                    month_data[d.day - 1] = True
                else:
                    month_data[d.day - 1] = calendar_day
            except KeyError:
                pass

        if len(route_months) == month_count:
            if route_months[0][1] == [None] * 31:
                route_months.pop(0)
                route_months.append((dates[0], month_data))
            else:
                break
        else:
            route_months.append((dates[0], month_data))

    shift = 0

    template_days = main_thread.L_days_text(
        shift,
        except_separator=", ",
        html=False,
        template_only=True,
        thread_start_date=thread_tz_today
    )

    template_days_all = main_thread.L_days_text(
        shift,
        except_separator=", ",
        html=False,
        template_only=False,
        thread_start_date=thread_tz_today
    )

    context = {
        'months': ThreadCalendarWidget.make_calendar(route_months),
        'title': title,
        'current': calendar_date,
        'template': template_days,
        'current_month': calendar_date.month,
        'template_days': template_days_all,
    }

    if group_by_path:
        context['groups'] = groups

    return context


class Group(object):
    def __init__(self, stations, threads):
        self.stations = stations
        self.threads = threads
        self.dt = None
        self.start_date = None
        self.thread = None

    @property
    def title(self):
        return "&nbsp;&mdash; ".join(s.settlement.L_title() if s.settlement else s.L_title() for s in self.stations)

    @property
    def link(self):
        return composeurl('thread', self.thread.uid, params={'departure': self.start_date})


def group_plane_threads(main, threads):
    # Вытягиваем все rstations сразу
    rtstations = list(RTStation.objects.filter(thread__in=threads))

    fetch_related(rtstations, 'station', model=RTStation)

    # Сортируем по нитке и порядку станции
    rtstations.sort(key=lambda o: (o.thread_id, o.id))

    # Маршруты
    routes = {}

    for thread_id, rtss in groupby(rtstations, lambda rts: rts.thread_id):
        stations = tuple(rts.station for rts in rtss)

        routes.setdefault(stations, []).append(thread_id)

    for stations, route_threads in routes.items():
        # Пропускаем группу выбранной нитки
        if main.id in route_threads:
            continue

        yield Group(stations, route_threads)


def get_uniq(array):
    """
    Возвращает список, в котором каждый элемент встречается только 1 раз
    с сохранением порядка элементов
    """

    if not array:
        return []

    uniq = []

    for obj in array:
        if not obj in uniq:
            uniq.append(obj)

    return uniq
