# -*- coding: utf-8 -*-

import json
import operator
import re
from datetime import datetime
from itertools import groupby
from collections import OrderedDict

from django.http import HttpResponseRedirect, HttpResponsePermanentRedirect
from django.utils.html import escape
from django.utils.http import urlquote

from common.forms.thread import RouteForm
from common.models_utils.i18n import RouteLTitle
from common.models.transport import TransportType
from common.utils.date import astimezone
from travel.rasp.library.python.common23.date import environment
from common.utils.locations import composeurl
from common.utils.mysql_try_hard import mysql_try_hard
from common.views.timezones import fill_tz_context
from common.xgettext.i18n import xgettext, gettext
from common.views.tariffs import DisplayInfo
from common.views.thread_search import process_threads

from route_search.by_number import find_threads_by_query
from route_search.models import RThreadSegment
from route_search.shortcuts import search_routes

from travel.rasp.morda.morda.templates.thread_search import OrdinalSearchTemplate, SpecialSearchTemplate
from common.utils.avia import get_avia_thread_url
from travel.rasp.morda.morda.views.search.counters import calculate_counters


@mysql_try_hard
def search_threads(request):
    """ Поиск маршрута по номеру """

    form = RouteForm(request.GET.copy())

    context = {
        'route_form': form,
        'title': gettext(u'Поиск рейсов'),
    }

    if not form.is_valid():
        return OrdinalSearchTemplate.render(request, context)

    query = form.cleaned_data['number']

    context['title'] = xgettext(u'Поиск рейсов - <query/>', query=escape(query))

    result = find_threads_by_query(query, form.get_ttypes())

    threads = process_threads(result.threads, environment.now_aware())

    if threads and result.is_special_train:
        return special_search_result(request, context, form, threads)

    return ordinal_search_result(request, context, form, threads)


def special_search_result(request, context, form, threads):
    u"""Результаты поиска фирменного рейса"""

    number = form.cleaned_data['number']

    searches = list(set((thread.first_station, thread.last_station) for thread in threads))
    searches.sort(key=lambda pair: sorted([pair[0].settlement_id, pair[1].settlement_id]))

    directions = {}
    for point_from, point_to in searches:
        key = (point_from.settlement or point_from, point_to.settlement or point_to)
        if key not in directions:
            directions[key] = []
        directions[key].append((point_from, point_to))

    context['searches'] = searches
    context['search_results'] = []
    context['number'] = threads[0].L_title_special()

    all_segments = []

    for direction, stations in directions.iteritems():
        segments_res = []

        for search_pair in stations:
            segments, _days, _any_date = search_routes(search_pair[0], search_pair[1])
            segments = [segment for segment in segments
                        if segment.thread.L_title_special.contains(number)]

            fill_tz_context(
                request,
                context,
                cities=search_pair,
                dates=[environment.today()],
            )

            segments_res.extend(segments)
            all_segments.extend(segments)

        time_zone = context['time_zone']

        # Сортировка
        columns = {
            "departure": ("time", lambda s: astimezone(s.departure, time_zone).time()),
            "arrival": ("time", lambda s: astimezone(s.arrival, time_zone).time()),
            "duration": ("raw", lambda s: s.duration),
        }

        sort = Sorter(columns, request.GET, default='+departure')
        sort.sort(segments_res)

        try:
            fragment = segments_res[0].title
        except IndexError:
            fragment = u''

        context['search_results'].append({
            'segments': segments_res,
            'sort': sort,
            'from': direction[0],
            'to': direction[1],
            'link_text': xgettext(u"<title-from/> &mdash; <title-to/>", title_from=direction[0].L_title, title_to=direction[1].L_title, mdash=u'&mdash;'),
            'fragment': fragment,
            'fragment_quoted': urlquote(fragment),
            })
        context['sort'] = sort

    RThreadSegment.fetch_titles(all_segments)

    for s in all_segments:
        s.display_info = DisplayInfo()

    context['search_results'].sort(key=lambda res: sorted((res['from'].id, res['to'].id)))

    search_navigation = [list(v)
        for k, v in groupby(context['search_results'], key=lambda result: sorted([result['from'].id, result['to'].id]))
    ]

    if len(search_navigation) == 1:
        context['search_navigation'] = map(lambda item: (item,), search_navigation[0])
    else:
        context['search_navigation'] = map(None, *search_navigation)

    if form.cleaned_data['number'].lower() == u'сапсан':
        context['show_sapsan_link'] = True

    return SpecialSearchTemplate.render(request, context)


def ordinal_search_result(request, context, form, threads):
    u"""Обычные результаты поиска рейсов по номеру"""

    number = form.cleaned_data['number']

    if form.get_ttypes():
        threads = list(filter_by_ttype(threads, form.get_ttypes()))

    matched_number_threads = filter(lambda t: t.number.lower() == number.lower(), threads)

    if len(threads) == 1 or (threads and len(matched_number_threads) == len(threads)):
        if threads[0].t_type_id == TransportType.PLANE_ID:
            return HttpResponsePermanentRedirect(get_avia_thread_url(
                threads[0], tld=request.tld,
                utm_medium='flight_landing', utm_campaign='redirect301')
            )
        return HttpResponseRedirect(composeurl('thread', kwargs={'uid': threads[0].uid},
                                               params={'number': number, 'from_search': '1'}))

    if matched_number_threads:
        matched_number_thread = matched_number_threads[0]
        threads = [thread for thread in threads if thread.route_id != matched_number_thread.route_id]
    else:
        matched_number_thread = None

    # Группировка по видам транспорта
    types_order = dict((v, k) for k, v in enumerate((2, 1, 6, 3)))

    threads.sort(key=lambda t: types_order.get(t.t_type_id, 99))

    ttype_groups = [
        (t_type_id, list(t_type_threads))
        for t_type_id, t_type_threads in groupby(threads, key=lambda t:t.t_type_id)
    ]

    mta_threads = filter(lambda t: t.supplier_id == 44, threads)
    mta_result = group_mta_threads(mta_threads)

    route_group_by_ttype = {}

    for t_type_id, t_type_threads in ttype_groups:
        # Группируем по рейсу
        t_type_threads[:] = group_routes(t_type_threads)

        route_group_by_ttype[t_type_id] = {}

        for route, route_threads in t_type_threads:
            if route.supplier_id == 44:  # mta
                continue

            # Выбираем первую отправляющуюся нитку
            # RASP-12564 - При поиске по номеру рейса попадаем не на ближайшую дату отправления  (Content error (ichernyshov))
            route_threads.sort(key=lambda t: t.first_station.departure_time)

            route_group_by_ttype[t_type_id][route] = route_threads[:1]

            continue

    types = dict((t.id, t) for t in TransportType.objects.all())

    awaps_params = {
        'page': 'route_search',
        'query': number,
        'ttypes': ','.join([types[t_type_id].code for t_type_id, _ in ttype_groups])
    }

    raw_mta_threads = [
        threads_grp[0]
        for courses in mta_result.values()
        for course in ('forward', 'backward')
        for stations_grp in courses[course].values()
        for mask_grp in stations_grp.values()
        for threads_grp in mask_grp.values()
    ]

    RouteLTitle.fetch([
        thread.L_title
        for thread in (
            [matched_number_thread] +
            raw_mta_threads
        )
        if thread is not None
    ])

    def pad_number_part(number):
        return re.sub('\d+', lambda m: '%010d' % int(m.group(0)), number)

    # Сортируем по номеру рейса, числа в строках сортируются как числа.
    route_group_by_ttype = {
        k: OrderedDict(
            sorted(v.iteritems(),
                   key=lambda x: pad_number_part(x[1][0].number)))
        for k, v in route_group_by_ttype.iteritems()
    }

    context.update({
        'groups': route_group_by_ttype,
        'mta_groups': mta_result,
        'different_plans': len(set((t.schedule_plan_id for t in threads if t.schedule_plan_id is not None))) > 1,
        'types': types,
        'counters': calculate_counters(threads),
        'exact': matched_number_thread,
        'awaps_params': awaps_params
    })

    return OrdinalSearchTemplate.render(request, context)


def group_mta_threads(threads):
    groups = {}
    forward = {}
    result = {}

    # группируем по направлениям
    for thread in threads:
        title_parts = thread.title.split(' - ')
        direction = u' - '.join(sorted(title_parts))

        if direction not in groups:
            groups[direction] = {
                'forward': [],
                'backward': []
            }

        if direction not in forward:
            forward[direction] = []

        if thread.title in forward[direction]:
            groups[direction]['forward'].append(thread)
        else:
            backward_title = u' - '.join(reversed(title_parts))
            if backward_title in forward[direction]:
                groups[direction]['backward'].append(thread)
            else:
                groups[direction]['forward'].append(thread)
                forward[direction].append(thread.title)

    for direction, courses in groups.iteritems():
        result[direction] = {
            'forward': {},
            'backward': {}
        }

        for course in ('forward', 'backward'):
            # группируем по рейсам
            route_groups = group_routes(courses[course])

            for route, threads in route_groups:
                result[direction][course][route] = {}

                # группируем по станциям ниток
                rts_groups = {}
                for thread in threads:
                    rts = thread.rtstation_set.all().order_by('id')
                    thread.rts = rts
                    stations = tuple([rt.station_id for rt in rts])

                    if stations not in rts_groups:
                        rts_groups[stations] = []
                        result[direction][course][route][stations] = {}

                    rts_groups[stations].append(thread)

                # группируем по дням хождений
                for stations, threads in rts_groups.iteritems():
                    for thread in threads:
                        mask = thread.L_days_text(0)

                        if mask not in result[direction][course][route][stations]:
                            result[direction][course][route][stations][mask] = []

                        result[direction][course][route][stations][mask].append(thread)

                    for mask in result[direction][course][route][stations]:
                        result[direction][course][route][stations][mask] = sorted(result[direction][course][route][stations][mask],
                                                                                  key=lambda t: t.tz_start_time)

    return result


def filter_by_ttype(threads, ttypes):
    for thread in threads:
        if thread.t_type_id in ttypes:
            yield thread


def group_routes(threads):
    threads.sort(key=lambda t: (
        t.route_id,
        t.title
    ))

    return [
        (route, list(route_threads)) for route, route_threads in groupby(threads, key=lambda r:r.route)
    ]


class Sorter(object):
    """Сортировщик таблиц"""

    def __init__(self, columns, request_get, table_index=0, default=None):
        self._columns = columns
        self.request_get = request_get
        self.table_index = table_index

        # Сортировочная группа
        # FIXME: если на странице будет больше одного типа сортируемых
        # таблиц, нужно будет его как-то распределять
        self.group_index = 0

        # Попробуем вытащить из запроса
        try:
            requested_sort = request_get.getlist('sortBy')[self.group_index]
            if request_get.get('all_days_search') and 'tariff' in requested_sort:
                requested_sort = None
        except IndexError:
            requested_sort = None

        self.column = None
        self.reverse = False

        def parse(sort_by):
            return sort_by.startswith('-'), sort_by.lstrip('+-')

        for sort_by in [requested_sort, default]:
            if sort_by:
                reverse, column = parse(sort_by)

                try:
                    _, py_extractor = columns[column]

                    self.column = column
                    self.reverse = reverse

                    if callable(py_extractor):
                        self.extractor = py_extractor
                    else:
                        self.extractor = operator.itemgetter(py_extractor)

                    break

                except KeyError:
                    # нет такой колонки, ищем дальше
                    continue

        # Не нашли колонку, значит дефолтовая неправильная
        if not self.column:
            raise Exception("Invalid default sort specified")

        self.columns = self.ColumnAccessor(self)

    class ColumnAccessor(dict):
        """Как словарь возвращает данные для колонок"""

        def __init__(self, sorter):
            self.sorter = sorter

        def __getitem__(self, column):

            # KeyError вылезет наружу, что нам и нужно
            js_extractor, _ = self.sorter._columns[column]

            link = self.sorter.get_sort_params(column)
            direction = self.sorter.get_direction(column)

            data = {
                'tableIndex': self.sorter.table_index,
                'column': column,
                'extractor': js_extractor,
            }

            return {
                'direction': direction,
                'link': link,
                'table_index': self.sorter.table_index,
                'metadata': json.dumps(data, ensure_ascii=False).replace('"', "'"),
            }

    def get_direction(self, column):
        if column == self.column:
            if self.reverse:
                return 'desc'
            else:
                return 'asc'

    def get_sort_params(self, column):
        # Если текущая сортировка является такой-же и прямой, то меняем направление
        if not self.reverse and self.column == column:
            new_sort = '-' + column
        else:
            # Иначе просто задаем нашу сортировку
            new_sort = '+' + column

        # Список сортировок для всех таблиц
        sort_list = self.request_get.getlist('sortBy')[:]

        # Удлинняем массив, если в него не входит нужная сортировка
        if len(sort_list) < self.group_index + 1:
            sort_list.extend([''] * (self.group_index + 1 - len(sort_list)))

        sort_list[self.group_index] = new_sort

        # Копируем параметры, чтобы внести изменения
        new_params = self.request_get.copy()

        new_params.setlist('sortBy', sort_list)

        return '?' + new_params.urlencode().replace('&', '&amp;')

    def sort(self, l):
        l.sort(key=self.extractor, reverse=self.reverse)

        # Значения с ключом None всегда должны быть в конце списка (RASP-1984)
        # поэтому если нет reverse, то нужно их туда переместить

        if not self.reverse:
            # Найдем, куда простирается граница None-значение (после сортировки
            # они оказываются все в начале списка)
            index = None
            for i, item in enumerate(l):
                if self.extractor(item) is not None:
                    # Первое не-None значение
                    index = i
                    break

            # Если не нашли не-None значение, то ничего менять не надо
            if index:
                # рокировка
                # [:] означает, что мы сам объект меняем
                l[:] = l[index:] + l[:index]

        # Поднимем интервальные нитки вверх
        def split_by_cond(arr, cond):
            pos = []
            neg = []
            for j in arr:
                if cond(j):
                    pos.append(j)
                else:
                    neg.append(j)
            return pos, neg

        interval, simple = split_by_cond(l, lambda s: getattr(s, 'thread', False) and s.thread.is_interval)
        interval.sort(key=lambda s: getattr(s, 'thread', False) and s.thread.begin_time or datetime(2001, 1, 1).time())
        l[:] = interval + simple
