# -*- coding: utf-8 -*-
import re
from locale import strcoll

from django.conf import settings
from django.db.models import Q
from django.http import Http404, HttpResponseRedirect, HttpResponsePermanentRedirect
from django.shortcuts import get_object_or_404
from django.utils.translation import get_language

import jinja2

from common.models.geo import Country, Region, Settlement, Station
from common.models.schedule import Company
from common.models.staticpages import Partner
from common.models.staticpages import StaticPage
from common.models.tariffs import AeroexTariff
from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from common.utils.locations import langify, composeurl
from common.utils.mysql_try_hard import mysql_try_hard
from common.views.currency import fetch_currency_info
from common.xgettext.i18n import xgettext, gettext
from travel.rasp.morda.morda.templates import static as static_templates
from travel.rasp.morda.morda.templates.partners import Template as PartnersTemplate
from travel.rasp.morda.morda.views.teasers import TeaserSetMorda
from travel.rasp.morda.morda.utils.locations import get_city_url
from travel.rasp.morda.staticpages.utils import divide_list, get_tablo_url

from travel.rasp.library.python.common23.utils.caching import cached

###############################################################################
# Классы статических страниц

def static_page_factory(page_type, page_id):
    u""" По указанному типу и коду страницы возвращает соответствующий объект """
    if page_type == 'static':
        if page_id.isdigit():
            page = get_object_or_404(StaticPage, pk=page_id, is_published__in=(1, 2), is_ticket_page=False)
        else:
            try:
                page = StaticPage.get_page(slug=page_id, is_published__in=(1, 2))
            except StaticPage.DoesNotExist:
                raise Http404

        if page.type == 'geo':
            return GeoPage()
        if page.type == 'carrier':
            return CarrierListPage(page)
        return FlatPage(page)

    elif page_type == 'ticket_static':
        if page_id.isdigit():
            page = get_object_or_404(StaticPage, pk=page_id, is_published__in=(1,2), is_ticket_page=True)
        else:
            try:
                page = StaticPage.get_ticket_page(slug=page_id, is_published__in=(1,2), is_ticket_page=True)
            except StaticPage.DoesNotExist:
                raise Http404
        return TicketFlatPage(page)

    elif page_type == 'country':
        return CountryPage(page_id)

    elif page_type == 'region':
        return RegionPage(page_id)

    elif page_type == 'settlement':
        return SettlementPage(page_id)

    elif page_type == 'station':
        return StationPage(page_id)

    elif page_type == 'carrier_country':
        return CarrierCountryPage(page_id)

    elif page_type == 'company':
        return CompanyPage(page_id)

    raise Http404


class BaseStaticPage(object):
    title = ''
    url = ''
    template = ''
    content = ''

    def parents(self): raise NotImplementedError
    def children(self): raise NotImplementedError
    @property
    def is_redirect(self): return False
    def tree(self): return []


VAR_RE = re.compile(r'\{[{%] (MEDIA_URL|MARKUP_URL) [%}]\}')

def var_repl(m):
    var = m.group(1)

    return getattr(settings, var)


class TicketFlatPage(BaseStaticPage):
    u"""Страниа из базы для КБ"""

    def __init__(self, page):
        if isinstance(page, StaticPage):
            self.object = page
        elif isinstance(page, int):
            self.object = get_object_or_404(StaticPage, pk=page, is_published__in=(1, 2))
        else:
            self.object = StaticPage.get_ticket_page(slug=page, is_published__in=(1, 2))

        self.title = self.object.L_title()
        self.content = VAR_RE.sub(var_repl, self.object.L_content())
        self.slug = self.object.slug
        if self.object.type == 'link':
            self.url = self.object.L_content()
        else:
            self.url = composeurl('info', args=[self.object.slug or self.object.pk])
        super(TicketFlatPage, self).__init__()

    def breadcrumbs(self):
        return [{
            'code': page.slug,
            'title': page.title
        } for page in self.parents()]

    def parents(self):
        pages = self.object.ticket_parents()
        if pages:
            return [TicketFlatPage(page) for page in pages ]
        else:
            return []

    def children(self):
        return [{
            'title': page.title,
            'code': page.slug
        } for page in self.object.ticket_children()]

    def tree(self):
        parents = list(self.object.ticket_parents()) + [self.object]
        return (TicketFlatPage(page) for page in
                StaticPage.ticket_objects.filter(Q(parent__in=parents[:1]) | Q(parent__isnull=True)
                    ).filter(is_published=1).order_by('materialized_path'))

    def indent(self):
        return len(self.object.materialized_path) / 3 - 1

    def __eq__(self, obj):
        return self.object == obj.object


class FlatPage(BaseStaticPage):
    u""" Обычная страница из базы """
    def __init__(self, page):
        if isinstance(page, StaticPage):
            self.object = page
        elif isinstance(page, int):
            self.object = get_object_or_404(StaticPage, pk=page, is_published__in=(1, 2), is_ticket_page=False)
        else:
            self.object = StaticPage.get_page(slug=page, is_published__in=(1, 2), is_ticket_page=False)
        self.title = self.object.L_title()
        self.content = VAR_RE.sub(var_repl, self.object.L_content() or '')
        self.template = self.object.template or 'static.html'
        self.slug = self.object.slug
        if self.object.type == 'link':
            self.url = self.object.L_content()
        else:
            self.url = composeurl('info', args=[self.object.slug or self.object.pk])
        super(FlatPage, self).__init__()

    @property
    def is_redirect(self):
        return self.object.type == 'link'

    @property
    def links(self):
        return self.object.staticlink_set.all()

    def parents(self):
        pages = self.object.parents()
        if pages:
            return [FlatPage(page) for page in pages ]
        else:
            return []

    def children(self):
        return ( FlatPage(page) for page in self.object.children() )

    def tree(self):
        parents = list(self.object.parents()) + [self.object]
        return (FlatPage(page) for page in
                StaticPage.objects.filter(Q(parent__in=parents[:1]) | Q(parent__isnull=True)
                    ).filter(is_published=1).order_by('materialized_path'))

    def indent(self):
        return len(self.object.materialized_path) / 3 - 1

    def __eq__(self, obj):
        return self.object == obj.object


@cached(lambda self, amount=settings.CITIES_IN_COUNTRY_SEARCH: '/stations_density/%s/%s' % (self.id, amount),
        timeout=settings.CACHES['default']['LONG_TIMEOUT'])
def get_biggest_cities_by_traffic(country, amount=settings.CITIES_IN_COUNTRY_SEARCH):
    cities = list(Settlement.hidden_manager.filter(country=country).order_by('-threads_amount')[:amount])
    stations = Station.objects.filter(settlement__in=cities, t_type=2, hidden=False).order_by('title')
    airports = {}
    for s in stations:
        airports.setdefault(s.settlement.id, []).append(s)

    return [(city, airports[city.id], len(airports[city.id])) for city in cities if city.id in airports]


class GeoPage(BaseStaticPage):
    u""" Список стран """
    template = 'geo.html'

    def __init__(self):
        self.title = gettext(u"Вокзалы и аэропорты")
        self.url = composeurl('geo')

    def countries(self, national_version):
        all_countries = [Country.objects.get(pk=c) for c in settings.OUR_COUNTRIES]

        # На первое место ставим страну, в которой находимся.
        c = all_countries.pop(all_countries.index(Country.objects.get(code=national_version.upper())))
        all_countries = [c] + all_countries

        for c in all_countries:
            country_stations = c.station_set.filter(hidden=False, region=None).count()

            if country_stations:
                c.other_url = composeurl('info_object', args=['country', c.id])

        return all_countries

    def regions(self, country):
        regions = country.region_set.filter(hidden=False)

        regions = filter(lambda r: not r.disputed_territory, regions)

        for r in regions:
            r.url = composeurl('info_object', args=['region', r.id])

        regions = sorted(regions, key=lambda r: r.L_title(), cmp=strcoll)
        return divide_list(regions)

    def big_cities(self, country):
        cities = get_biggest_cities_by_traffic(country, 5)

        return filter(lambda c: not c[0].disputed_territory, cities)

    def parents(self):
        return []


class RegionPage(BaseStaticPage):
    u""" Область """
    template = 'region.html'

    def __init__(self, region_id):
        self.object = get_object_or_404(Region, pk=region_id)
        if self.object.disputed_territory:
            raise Http404()

        self.title = self.object.L_title()
        self.url = composeurl('info_object', args=['region', self.object.id])

    def parents(self):
        return [GeoPage(), CountryPage(self.object.country_id)]

    def children(self):
        settlements = self.object.settlement_set.filter(hidden=False, has_tablo=True).order_by('majority', 'title')
        for s in settlements:
            s.url = get_city_url(s.id)
        return divide_list(settlements)

    def stations(self):
        return self.object.station_set.filter(hidden=False).order_by('title')

    def stations_parts(self):
        stations = self.stations()

        for s in stations:
            type_choices = s.type_choices_set

            links = []

            if s.t_type_id == TransportType.PLANE_ID and 'tablo' in type_choices:
                links.append((get_tablo_url(s), 'plane', 'plane'))

            if 'train' in type_choices:
                links.append((s.get_schedule_url('train'), 'train', 'train'))

            for type_ in ['suburban', 'aeroex']:
                if type_ in type_choices:
                    links.append((s.get_schedule_url(type_), 'suburban', type_))

            if s.t_type_id == TransportType.BUS_ID:
                links.append((get_tablo_url(s), 'bus', 'bus'))

            if s.t_type_id in TransportType.WATER_TTYPE_IDS:
                links.append((get_tablo_url(s), 'water', s.t_type.code))

            s.links = links

            s.url = composeurl('info_object', args=['station', s.id])

        return divide_list(stations)

    def __eq__(self, obj):
        return self.object == obj.object


class CountryPage(RegionPage):
    u""" Страна """
    template = 'country.html'

    def __init__(self, country_id):
        self.object = get_object_or_404(Country, pk=country_id)
        self.title = self.object.L_title()
        self.url = ''

    def children(self):
        return ()

    def stations(self):
        return self.object.station_set.filter(hidden=False, region=None).order_by('title')

    def parents(self):
        return [GeoPage()]


class SettlementPage(BaseStaticPage):
    u""" Город """
    def __init__(self, settlement_id):
        self.object = get_object_or_404(Settlement, pk=settlement_id)
        if self.object.disputed_territory:
            raise Http404()

        self.title = self.object.L_title()
        self.url = get_city_url(self.object.id)


class StationPage(BaseStaticPage):
    u""" Станция """
    template = 'station.html'

    def __init__(self, station_id):
        self.object = get_object_or_404(Station.objects, pk=station_id)
        if self.object.hidden and self.object.t_type_id == 8:
            raise Http404

        self.tablo_link = self.object.majority_id < 5 and self.object.hidden
        self.sub_type = self.object.t_type.code
        self.title = self.object.L_title()
        self.url = composeurl('info_object', args=['station', self.object.id])

    def parents(self):
        p = [GeoPage()]

        disputed = self.object.region_id and self.object.region.disputed_territory
        disputed = disputed or \
            (self.object.settlement_id and self.object.settlement.disputed_territory)

        if self.object.country_id:
            if not disputed:
                p += [CountryPage(self.object.country_id)]
        if self.object.region_id:
            if self.object.region.disputed_territory:
                p += [self.object.region.L_title()]
            else:
                p += [RegionPage(self.object.region_id)]
        if self.object.settlement_id:
            if self.object.settlement.disputed_territory:
                p += [self.object.settlement.L_title()]
            else:
                p += [SettlementPage(self.object.settlement_id)]
        return p

    def __eq__(self, obj):
        return self.object == obj.object


class CarrierListPage(FlatPage):
    u""" Список стран с фирмами-перевозчиками """

    def __init__(self, page):
        super(CarrierListPage, self).__init__(page)
        self.template = 'carrier_list.html'
        self.title = gettext(u"Перевозчики")

    def children(self):
        list = Company.hidden_manager.values('country').distinct()
        countries = Country.objects.filter(pk__in=[c['country'] for c in list] )
        for c in countries:
            c.url = composeurl('info_object', args=['carrier_country', c.id])

        countries = sorted([c for c in countries], cmp=strcoll, key=lambda c: c.L_title())
        return divide_list(countries)


class CarrierCountryPage(BaseStaticPage):
    u""" Страна с фирмами """
    template = 'carrier_country.html'

    def __init__(self, country_id):
        self.carriers_page = FlatPage(StaticPage.get_page(type='carrier'))
        self.object = get_object_or_404(Country, pk=country_id)
        self.title = self.object.L_title()
        self.url = composeurl('info_object', args=['carrier_country', self.object.id])

    def parents(self):
        return list(self.carriers_page.parents()) + [self.carriers_page]

    def children(self):
        companies = self.object.company_set.filter(hidden=False, strange=False).order_by('title')
        for c in companies:
            c.url = composeurl('info_object', args=['company', c.id])
        return divide_list(companies, 2)

    def tree(self):
        return self.carriers_page.tree()


class CompanyPage(BaseStaticPage):
    u""" Фирма-перевозчик """
    template = 'company.html'

    def __init__(self, company_id):
        self.carriers_page = FlatPage(StaticPage.get_page(type='carrier'))
        self.object = get_object_or_404(Company, pk=company_id, hidden=False)

        self.title = self.object.L_title()
        self.url = composeurl('info_object', args=['company', self.object.id])

    def parents(self):
        parents = list(self.carriers_page.parents()) + [self.carriers_page]

        if self.object.country_id:
            parents += [CarrierCountryPage(self.object.country_id)]

        return parents

    def tree(self):
        return self.carriers_page.tree()


TEMPLATES = {
    'info.html': 'InfoTemplate',
    'static_with_direct.html': 'StaticWithDirectTemplate',
    'static.html': 'StaticTemplate',
    'carrier_list.html': 'CarrierListTemplate',
    'carrier_country.html': 'CarrierListTemplate',
    'company.html': 'CompanyTemplate',
    'station.html': 'StationTemplate',
    'region.html': 'RegionTemplate',
    'country.html': 'CountryTemplate',
}

def get_template(template):
    template_name = TEMPLATES[template]

    return getattr(static_templates, template_name)


# Вьюхи статических страниц

@mysql_try_hard
def info_object(request, object_type, object_id):
    """Информация о станции, городе, компании и тд"""
    # Иначе в шаблонах плохо получается
    try:
        context = {'object_id': int(object_id)}
    except ValueError:
        raise Http404

    page = static_page_factory(object_type, object_id)

    if object_type == 'settlement':
        return HttpResponsePermanentRedirect(get_city_url(object_id))

    elif object_type == 'station':
        if not page.object.has_info:
            if page.object.settlement:
                return HttpResponseRedirect(composeurl('info_object', args=['settlement', page.object.settlement_id]))
            else:
                return static_templates.UnknownStationTemplate.render(request, {'page': page, 'title': gettext(u'Неизвестная станция')})

        context['local_time'] = environment.now_utc(aware=True).astimezone(page.object.pytz)

        if page.object.has_aeroexpress and page.object.t_type_id == 2 and page.object.settlement:
            tariffs = list(AeroexTariff.objects.filter(
                                                  Q(station_from=page.object, station_to__settlement=page.object.settlement) |
                                                  Q(station_to=page.object, station_from__settlement=page.object.settlement, reverse=True)
                                                  ))
            if tariffs:
                context['currency_info'] = fetch_currency_info(request)
                context['tariff'] = tariffs[0].tariff

        station = page.object

        context['awaps_params'] = {
            'page': 'tablo',
            'id': station.id,
            'title': station.title.encode('utf8'),
            'ttype': station.t_type.code,
            'majority': station.majority_id
        }

        if (station.t_type.code == 'train') and (station.country_id not in settings.OUR_COUNTRIES):
            context['show_disclaimer'] = True

        if (station.t_type.code == 'train') and (station.country_id not in settings.OUR_COUNTRIES):
            context['show_disclaimer'] = True

    context['page'] = page
    context['show_print'] = True

    if hasattr(page.object, 'L_info_title'):
        context['title'] = page.object.L_info_title()
    else:
        context['title'] = page.object.L_title()

    teaser_page = {'settlement': 'info_settlement',
                   'station': 'info_station'}.get(object_type, object_type)
    context['teasers'] = TeaserSetMorda(request, teaser_page, page.object)

    if 'page' in context and context['page'].template:
        template = context['page'].template
    else:
        template = 'info.html'

    return get_template(template).render(request, context)


@mysql_try_hard
def info(request, object_id='about'):
    """Статические страницы"""
    page = static_page_factory('static', object_id)

    if (page.template == 'geo.html') or page.is_redirect:
        return HttpResponsePermanentRedirect(langify(page.url))

    if object_id.isdigit() and getattr(page.object, 'slug', None):
        return HttpResponsePermanentRedirect(composeurl('info', args=[page.object.slug]))

    if page.object.type == 'template':
        from tariffs.views import holiday_tickets

        content_template = jinja2.Template(page.object.content)

        page.content = content_template.render(
            MEDIA_URL=settings.MEDIA_URL,
            MARKUP_URL=settings.MARKUP_URL,
            holiday_tickets=holiday_tickets,
        )

    template = page.template or 'info.html'

    context = {'object_id': object_id,
               'page': page,
               'page_code': page.object.slug,
               'title': page.object.title,
               'teasers': TeaserSetMorda(request, 'info'),
               'show_print': True,
               }

    return get_template(template).render(request, context)


def partners(request):
    service =  getattr(request, 'ticket', None) and 'ticket' or 'rasp'

    partners = Partner.objects.filter(is_published=True, services__in=('both', service)).\
                                exclude(t_type=None).\
                                order_by('order').\
                                select_related('t_type')
    train_tab_selected = False
    groups = {}
    for partner in partners:
        key = partner.t_type_id
        if key not in groups:
            title = partner.t_type.L_title_partner_page() or partner.t_type.L_title()

            if key in (4, 5): # речной и морской транспорт объединяем
                key = 4

            if partner.t_type.code == 'train':
                train_tab_selected = True

            groups[key] = {
                'current': partner.t_type.code == 'train',
                'title': title,
                'partners': []
            }
        groups[key]['partners'].append(partner)

    if not train_tab_selected and partners:
        key = partners[0].t_type_id
        groups[key]['current'] = True

    context = {
        'groups': groups,
        'page': FlatPage('partners'),
        'page_code': 'partners',
        'show_print': True,
        'teasers': TeaserSetMorda(request, 'info'),
        'title': gettext(u'Партнёры')
    }

    return PartnersTemplate.render(request, context)


def geo(request):
    u"""Страница стран с главными городами"""
    page = GeoPage()
    context = {'page': page,
               'page_code': 'geo',
               'title': gettext(u'География сервиса'),
               'teasers': TeaserSetMorda(request, 'info'),
               'show_print': True
               }

    return static_templates.GeoTemplate.render(request, context)


def info_mta(request):
    # RASP-12045
    if get_language() != 'ru':
        raise Http404

    sids = [sid for sid, title in settings.MTA_PROMO_STATIONS]
    busstations = Station.hidden_manager.exclude(longitude=None).exclude(latitude=None).in_bulk(sids)
    busstations = [(busstations[sid], title) for sid, title in settings.MTA_PROMO_STATIONS if sid in busstations]
    busstations = [{'id': s.id,
                    'title': title,
                    'title_balloon': s.settlement and (xgettext(u'<title/> (<popular-title/>)', title=s.settlement.L_title, popular_title=s.L_popular_title_extra)) or \
                                     s.L_popular_title_extra,
                    'lng': s.longitude,
                    'lat': s.latitude,
                    'link': composeurl('station', args=[s.id]),
                    'address': s.full_address,
                    'phone': u', '.join(p.phone for p in s.stationphone_set.all()),
                    }
                   for s, title in busstations]

    samples = Station.hidden_manager.filter(region_id=1, majority_id__lte=2, t_type_id=3).order_by('?').all()[:4]
    samples = [{'id': 's%s' % s.id, 'title': s.get_clean_title()} for s in samples]

    context = {
        'busstations': busstations,
        'samples': {
            'from':[
                { 'id': 'c213', 'title': u'Москва' },
                { 'id': 'c10754', 'title': u'Серпухов' }
            ],
            'to': [
                { 'id': 'c10734', 'title': u'Коломна' },
                { 'id': 'c217', 'title': u'Пущино' },
            ],
        },
        'title': gettext(u'Яндекс.Расписания')
    }

    return static_templates.InfoMtaTemplate.render(request, context)
