# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.http import QueryDict
from marshmallow import Schema, fields, pre_load, post_load

from common.utils.namedtuple import namedtuple_with_defaults
from travel.rasp.morda_backend.morda_backend.serialization.fields import CountryTypeField, TransportTypeField


ArticlesContext = namedtuple_with_defaults('ArticlesContext', [
    'country', 'limit', 'settlement_id', 'settlement_slug', 'transport_type',
], defaults={'limit': 10})


class ArticlesQuerySchema(Schema):
    country = CountryTypeField()
    limit = fields.Integer()
    settlement_id = fields.Integer()
    settlement_slug = fields.String()
    transport_type = TransportTypeField(load_from='t_type')

    @post_load
    def make_context(self, data):
        return ArticlesContext(**data)


class ArticlesStaticPageSchema(Schema):
    id = fields.Integer()
    slug = fields.String()
    title = fields.Function(lambda obj: obj.L_title())
    description = fields.Function(lambda obj: obj.L_meta_description())


class ArticlesResponseSchema(Schema):
    articles = fields.Nested(ArticlesStaticPageSchema, many=True)


StaticPageContext = namedtuple_with_defaults('StaticPageContext', [
    'slug', 'country',
], defaults={})


class StaticPageQuerySchema(Schema):
    slug = fields.String(required=True)
    country = CountryTypeField()

    @post_load
    def make_context(self, data):
        return StaticPageContext(**data)

    @pre_load
    def defaults(self, data):
        result = data
        if 'country' not in data:
            if isinstance(data, QueryDict):
                result = data.dict()
            result['country'] = 'ru'
        return result


class StaticPageShortSchema(Schema):
    id = fields.Integer()
    slug = fields.String()
    title = fields.Function(lambda obj: obj.L_title())


class StaticPageSchema(Schema):
    id = fields.Integer()
    slug = fields.String()
    title = fields.Function(lambda obj: obj.L_title())
    content = fields.Function(lambda obj: obj.L_content())
    meta_title = fields.Function(lambda obj: obj.L_meta_title())
    meta_description = fields.Function(lambda obj: obj.L_meta_description())
    parents = fields.List(fields.Nested(StaticPageShortSchema))  # order is important: from root to parent
    children = fields.List(fields.Nested(StaticPageShortSchema))
