# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from common.data_api.search_stats.search_stats import search_stats
from common.models_utils.geo import Point
from common.models.geo import Settlement, Station, StationMajority
from common.models.transport import TransportType
from common.utils.settlement import get_connected_stations
from route_search.models import ZNodeRoute2


def get_point_popular_directions(direction, point, t_type, limit, search_type):
    """
    Возвращает популярные направления для указанной точки, возвращает только канонические поиски
    :param direction: 'from' - направления из точки, или 'to' - направления в точку
    :param point: город или станция, для которого ищутся популярные направления
    :param t_type: тип транспорта, или None для поисков всеми типами
    :param limit: количество возвращаемых направлений
    :param search_type: 'c' - выбирает только города, 's' - станции, 'all' - города и станции
    """
    settlement, station = _get_init_settlement_and_station(point)
    # Для иностранных городов поиск всеми типами заменяем на поиск самолетами
    if not t_type and ((settlement and settlement.is_foreign()) or (station and station.is_foreign())):
        t_type = TransportType.get_plane_type()

    t_type_id = t_type.id if t_type else None

    if station and t_type_id:
        add_init_settlement = _settlement_has_one_station(settlement, t_type_id)
    else:
        add_init_settlement = False

    keys_list, points_dict = _get_popular_stats_points(direction, point, t_type, search_type, limit)

    # Не делаем запрос к ZNR для самых больших городов - долго
    is_slow_city = not t_type_id and point.id in {Settlement.MOSCOW_ID}

    # Формируем список сложных сегментов, каждый из которых может вернуть одно популярное направление
    segments = []
    segments_by_init_points = {}
    for key in keys_list:
        if key not in points_dict:
            continue

        if points_dict[key] == station or points_dict[key] == settlement:
            continue

        if is_slow_city and point.country_id != points_dict[key].country_id:
            continue

        need_znr_filtration = not (
            (t_type_id and t_type_id not in TransportType.RAILWAY_TTYPE_IDS)
            or (isinstance(point, Station) and isinstance(points_dict[key], Station))
            or is_slow_city
        )
        point_segment = ComplexPointsSegment(point, points_dict[key], t_type_id, need_znr_filtration)

        # Если станция одна в городе по типу транспорта, то в итоге выводим title города, а не станции
        if add_init_settlement:
            point_segment.add_inner_point(settlement)

        segments.append(point_segment)
        segments_by_init_points[(point.point_key, key)] = point_segment

    # Для ЖД и поиска всеми типами заполняем для каждого сложного сегмента, список сегментов станция-станция
    stations_by_id = {}
    if (not t_type or t_type.id in TransportType.RAILWAY_TTYPE_IDS) and not is_slow_city:
        _fill_segments_from_znr(direction, point, t_type_id, segments_by_init_points, points_dict)

        stations_ids = set()
        for segment in segments:
            for station_segment in segment.stations_segments:
                stations_ids.add(station_segment[0])
                stations_ids.add(station_segment[1])
        stations_by_id = Station.objects.in_bulk(stations_ids)

    # Формируем популярные направления из сложных сегментов
    direction_slugs = set()
    popular_directions = []
    for segment in segments:
        popular_direction = segment.get_popular_direction(direction, stations_by_id)

        if popular_direction:
            slugs_pair = (popular_direction.inner_point.slug, popular_direction.outer_point.slug)
            if slugs_pair not in direction_slugs:
                popular_directions.append(popular_direction)
                direction_slugs.add(slugs_pair)

        if len(popular_directions) >= limit:
            break

    return popular_directions


def _get_popular_stats_points(direction, point, t_type, search_type, limit):
    """
    Получение популярных направлений из монги
    Берем в 1.5 раза больше, чтобы хватило, некоторые будут выкинуты
    """
    t_type_code = t_type.code if t_type else 'all'

    top_pairs = (
        search_stats.get_top_from(point.point_key, t_type_code, search_type, limit=int(limit * 1.6))
        if direction == 'from'
        else search_stats.get_top_to(point.point_key, t_type_code, search_type, limit=int(limit * 1.6))
    )

    # Получаем точки из базы по ключам
    keys_list = [point_key for (point_key, _) in top_pairs]
    points_dict = Point.in_bulk(keys_list)
    points_dict[point.point_key] = point

    return keys_list, points_dict


def _get_init_settlement_and_station(point):
    """Возвращает по возможности город и станцию для поиска популярных направлений"""
    if isinstance(point, Settlement):
        return point, None
    elif isinstance(point, Station):
        if point.settlement:
            return point.settlement, point
        else:
            return None, point
    return None, None


def _settlement_has_one_station(settlement, t_type_id):
    """Проверяет что для заданного типа транспорта в городе ровно одна станция"""
    if not settlement:
        return False

    stations_count = 0
    for station in get_connected_stations(settlement, StationMajority.IN_TABLO_ID, t_type_id):
        if station.t_type.id == t_type_id:
            stations_count += 1
            if stations_count > 1:
                return False

    return stations_count == 1


def _fill_segments_from_znr(direction, init_point, t_type_id, segments_by_init_points, points_dict):
    """
    Заполняет для сложных сегментов словари stations_segments, содержащие различные сегменты станция-станция
    """
    outer_points = points_dict.values()
    znr_filters = _make_znr_filters(direction, init_point, t_type_id, outer_points)

    def get_point(point_id, point_type):
        if not point_id:
            return None
        point_key = '{}{}'.format(point_type, point_id)
        if point_key in points_dict:
            return points_dict[point_key]
        return None

    def add_stations_segment(inner_point, outer_point, stations_segment):
        if inner_point and outer_point and (inner_point.point_key, outer_point.point_key) in segments_by_init_points:
            segments_by_init_points[
                (inner_point.point_key, outer_point.point_key)
            ].stations_segments.add(stations_segment)

    for iterator in znr_filters:
        for znr_segment in iterator:
            inner_settlement = get_point(znr_segment[0], 'c')
            inner_station = get_point(znr_segment[1], 's')
            inner_station_id = znr_segment[1]
            outer_settlement = get_point(znr_segment[2], 'c')
            outer_station = get_point(znr_segment[3], 's')
            outer_station_id = znr_segment[3]
            t_type_id = znr_segment[4]

            stations_segment = (inner_station_id, outer_station_id, t_type_id)
            add_stations_segment(inner_settlement, outer_settlement, stations_segment)
            add_stations_segment(inner_settlement, outer_station, stations_segment)
            add_stations_segment(inner_station, outer_settlement, stations_segment)


def _make_znr_filters(direction, init_point, t_type_id, outer_points):
    """
    Задает фильтры ZNodeRoute2, нужные для получения всех рейсов между исходной точкой и списком других точек
    :param direction: 'from' - направления из точки, или 'to' - направления в точку
    :param init_point: исходная точка
    :param t_type_id: id типа транспорта
    :param outer_points: список точек популярных направлений
    :return: список фитьтров над ZNodeRoute2
    """
    stations = {point for point in outer_points if isinstance(point, Station)}
    settlements = {point for point in outer_points if isinstance(point, Settlement)}

    if t_type_id:
        t_type_filter = ZNodeRoute2.objects.filter(t_type_id=t_type_id)
    else:
        t_type_filter = ZNodeRoute2.objects.exclude(t_type_id=TransportType.PLANE_ID)

    if isinstance(init_point, Station):
        if direction == 'from':
            station_filters = [t_type_filter.filter(station_from=init_point, settlement_to__in=settlements)]
        else:
            station_filters = [t_type_filter.filter(station_to=init_point, settlement_from__in=settlements)]

    else:
        station_filters = []
        if direction == 'from':
            if settlements:
                station_filters.append(t_type_filter.filter(settlement_from=init_point, settlement_to__in=settlements))
            if stations:
                station_filters.append(t_type_filter.filter(settlement_from=init_point, station_to__in=stations))
        else:
            if settlements:
                station_filters.append(t_type_filter.filter(settlement_to=init_point, settlement_from__in=settlements))
            if stations:
                station_filters.append(t_type_filter.filter(settlement_to=init_point, station_from__in=stations))

    if direction == 'from':
        return [
            station_filter.filter(good_for_finish=True).values_list(
                'settlement_from_id', 'station_from_id', 'settlement_to_id', 'station_to_id', 't_type_id'
            ).distinct()
            for station_filter in station_filters
        ]
    else:
        return [
            station_filter.filter(good_for_start=True).values_list(
                'settlement_to_id', 'station_to_id', 'settlement_from_id', 'station_from_id', 't_type_id'
            ).distinct()
            for station_filter in station_filters
        ]


class ComplexPointsSegment(object):
    """
    Сложный сегмент между двумя точками, предназначеный для формирования одного популярного направления
    Среди прочего осуществляет анализ списка сегментов станция-станция в рамках данного направления точка-точка
    """
    def __init__(self, inner_point, outer_point, t_type_id, need_znr_filtration):

        # Для сегмента есть точки, по которым он сформирован (inner_point, outer_point)
        # А также могут быть ссылки на города и станции для формирования итогового популярного направления
        self.inner_point = inner_point
        self.outer_point = outer_point
        self.t_type_id = t_type_id
        self.inner_station = None
        self.inner_settlement = None
        self.outer_station = None
        self.outer_settlement = None

        self.add_inner_point(inner_point)
        self.add_outer_point(outer_point)
        self.need_znr_filtration = need_znr_filtration
        self.stations_segments = set()

    def __eq__(self, other):
        return (
            self.inner_point == other.inner_point
            and self.outer_point == other.outer_point
            and self.t_type_id == other.t_type_id
        )

    def __hash__(self):
        return hash(''.join([str(self.t_type_id), self.inner_point.point_key, self.outer_point.point_key]))

    def add_inner_point(self, point):
        if isinstance(point, Station):
            self.inner_station = point
        if isinstance(point, Settlement):
            self.inner_settlement = point

    def add_outer_point(self, point):
        if isinstance(point, Station):
            self.outer_station = point
        if isinstance(point, Settlement):
            self.outer_settlement = point

    def get_popular_direction(self, direction, stations_by_id):
        """Возвращает популярное направление для сложного сегмента"""
        self._add_alone_stations_and_types(stations_by_id)

        if self.inner_station and self.inner_station.hidden:
            return None
        if self.outer_station and self.outer_station.hidden:
            return None

        return PopularSegment(
            _make_popular_point(direction, self.inner_station, self.inner_settlement),
            _make_popular_point('from' if direction == 'to' else 'to', self.outer_station, self.outer_settlement),
            self.t_type_id
        )

    def _add_alone_stations_and_types(self, stations_by_id):
        if not self.need_znr_filtration:
            return

        if not self.t_type_id:
            t_type_ids = {segment[2] for segment in self.stations_segments}
            if len(t_type_ids) == 1:  # Если нашлись нитки только одного типа, сужаем до него
                self.t_type_id = t_type_ids.pop()
            elif len(t_type_ids) == 0:  # Если совсем не нашлось ниток, значит все нитки - самолеты, их нет в базе
                self.t_type_id = TransportType.PLANE_ID

        inner_stations = {segment[0] for segment in self.stations_segments}
        outer_stations = {segment[1] for segment in self.stations_segments}

        if self.t_type_id and self.t_type_id in TransportType.RAILWAY_TTYPE_IDS:
            # Если все нитки идут с одной станции, то сужаем до нее
            if len(inner_stations) == 1:
                station_id = inner_stations.pop()
                if station_id in stations_by_id:
                    self.add_inner_point(stations_by_id[station_id])
            if len(outer_stations) == 1:
                station_id = outer_stations.pop()
                if station_id in stations_by_id:
                    self.add_outer_point(stations_by_id[station_id])


def _make_popular_point(direction, station, settlement):
    """
    Формируем точку популярного направления
    """
    if station and not settlement:
        return PopularPoint(station, station, direction)

    if settlement:
        if not station:
            return PopularPoint(settlement, settlement, direction)
        if station:
            return PopularPoint(station, settlement, direction)


class PopularPoint(object):
    """Точка сегмента популярного направления"""
    def __init__(self, key_point, title_point, direction):
        """
        :param key_point: точка, по которой формируются ключи для формирования ссылки
        :param title_point: точка, по которой формируются надписи на странице
        """
        self.id = key_point.id
        self.point_key = key_point.point_key
        self.slug = key_point.slug
        self.title = title_point.title
        self.title_phrase = (
            getattr(title_point, 'L_title_phrase_{}'.format(direction))()
            if isinstance(title_point, Settlement)
            else None
        )


class PopularSegment(object):
    """Сегмент популярного направления"""
    def __init__(self, inner_point, outer_point, t_type_id):
        self.t_type = TransportType.objects.get(id=t_type_id) if t_type_id else None
        self.inner_point = inner_point
        self.outer_point = outer_point
