# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import copy
import functools
import operator
import re
from collections import namedtuple, defaultdict
from datetime import datetime, timedelta

from dateutil import relativedelta

from django.conf import settings

from common.dynamic_settings.default import conf
from common.models.schedule import RThreadType
from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from common.utils.date import calculate_run_days, RunMask


SUBSEGMENTS_MAX_MINUTES_DIFFERENCE = 15

TrainNumber = namedtuple('TrainNumber', ['digits', 'letters'])
train_number_re = re.compile(r'(\d+)(\D+)$')


def _is_through_train(segment):
    return segment.thread and segment.thread.type_id == RThreadType.THROUGH_TRAIN_ID


def _parse_train_number(train_number):
    match = train_number_re.match(train_number)
    if not match:
        return None
    return TrainNumber(*match.groups())


def fill_train_numbers(segments):
    for segment in segments:
        if segment.t_type.id == TransportType.TRAIN_ID:
            segment.train_number = _parse_train_number(segment.number)


def get_possible_numbers(segment):
    thread_number = segment.thread.number
    numbers = {thread_number}

    letters = getattr(segment, 'letters', None)
    train_number = getattr(segment, 'train_number', None)
    if letters and train_number:
        numbers |= {'{}{}'.format(train_number.digits, letter) for letter in letters}

    numbers.update(
        cppk_number
        for cppk_number, express_numbers in conf.TRAIN_PURCHASE_P2_TRAIN_NUMBERS_MAP.items()
        if thread_number in express_numbers
    )

    return numbers


def _choose_main_segment(segments):
    # Выбираем основную нитку, если она есть. Преимущество у быстрых поездов
    return min(segments, key=lambda x: (_is_through_train(x), x.duration))


def _create_meta_train(main_segment, segments):
    meta_train = copy.copy(main_segment)
    meta_train.sub_segments = segments
    all_letters = ''.join(sorted({letter for segment in segments for letter in segment.train_number.letters}))
    meta_train.number = '{}{}'.format(main_segment.train_number.digits, all_letters)
    meta_train.letters = {segment.train_number.letters for segment in segments}
    meta_train.letters.add(all_letters)
    meta_train.companies = {segment.company for segment in segments if segment.company}
    meta_train.min_arrival = min([segment.arrival for segment in segments])
    meta_train.max_arrival = max([segment.arrival for segment in segments])
    return meta_train


def make_meta_trains(segments):
    """
    Группируем сегменты поездов из поиска на день по цифрам номера и времени отправления
    Таким образом, для сегмента формируется sub_segments, содержащий беспересадочные вагоны
    """
    train_numbers = defaultdict(list)
    for segment in segments:
        if segment.t_type.id == TransportType.TRAIN_ID and segment.train_number:
            train_numbers[(segment.train_number.digits, segment.departure)].append(segment)

    for v in train_numbers.values():
        if len(v) >= 2:
            main_segment = _choose_main_segment(v)
            segments.append(_create_meta_train(main_segment, v))
            for segment in v:
                segment.is_sub_segment = True

    return [segment for segment in segments if not getattr(segment, 'is_sub_segment', False)]


def _add_to_train_segments_dict(train_segments_dict, segment):
    if segment.t_type.id == TransportType.TRAIN_ID:
        number = _parse_train_number(segment.number)
        if number:
            from_id = segment.station_from.id
            to_id = segment.station_to.id
            train_segments_dict[(number.digits, from_id, to_id)].append(segment)


def _make_segment_days_number(segment):
    # Формируем для сегментов количество дней хождения за текщие два месяца и за все время
    today = environment.today()
    after_month = today + relativedelta.relativedelta(months=1)
    run_days = segment.run_days

    segment.next_month_days_number = (
        sum(run_days.get(str(today.year), {}).get(str(today.month), [])) +
        sum(run_days.get(str(after_month.year), {}).get(str(after_month.month), []))
    )
    segment.days_number = 0
    for year in run_days.values():
        for month in year.values():
            segment.days_number += sum(month)


def _compare_times(new_time, old_time):
    """Проверяем, что new_time не сильно позднее чем old_time"""
    today = environment.today()
    new_dt = datetime.combine(today, new_time)
    old_dt = datetime.combine(today, old_time)
    return new_dt < old_dt + timedelta(minutes=SUBSEGMENTS_MAX_MINUTES_DIFFERENCE)


def _compare_days_number(main_segment, segment):
    """Сравниваем сегменты по количеству дней хождения"""
    return (
        main_segment.next_month_days_number <= segment.next_month_days_number or
        (
            main_segment.next_month_days_number == segment.next_month_days_number and
            main_segment.days_number <= segment.days_number
        )
    )


def _add_subsegments(main_segment, segments_group):
    """Добавляем подсегменты в главный сегмент"""
    main_segment.sub_segments = [s for s in segments_group if s.is_sub_segment]
    main_segment.sub_segments.sort(key=lambda s: s.departure_time)
    main_segment.companies = {s.company for s in main_segment.sub_segments if s.company}


def _join_twin_segments(segments):
    """
    Склейка календарей сегментов с одинаковым номером, именем нитки, начальной и конечной остановкой
    и временем отправления и прибытия
    https://st.yandex-team.ru/RASPFRONT-9632
    """
    grouped_segments = defaultdict(list)
    for segment in segments:
        key = segment.number, segment.thread.title, segment.departure.time(), segment.arrival.time()
        grouped_segments[key].append(segment)

    for segments_list in grouped_segments.values():
        if len(segments_list) < 2:
            continue

        segments_list.sort(key=lambda segment: (segment.next_month_days_number, segment.days_number), reverse=True)
        main_segment = segments_list[0]

        run_masks = [RunMask(main_segment.thread.year_days)]
        for segment in segments_list[1:]:
            run_masks.append(RunMask(segment.thread.year_days))
            segment.is_twin_segment = True
        main_segment.thread.year_days = str(functools.reduce(operator.or_, run_masks))
        main_segment.start_date = min(segment.start_date for segment in segments_list)

        main_segment.force_recalculate_days_text = True
        main_segment.run_days = calculate_run_days(main_segment, days_ago=settings.SEARCH_DAYS_TO_PAST)
        _make_segment_days_number(main_segment)

    return [segment for segment in segments if not getattr(segment, 'is_twin_segment', False)]


def make_grouped_trains(segments):
    """
    Группируем похожие сегменты поездов из поиска на все дни
    Один делаем главным, остальные кладем к нему в sub_segments
    https://st.yandex-team.ru/RASPFRONT-9549
    """
    train_segments_dict = defaultdict(list)
    # Разбиваем по группам с одинаковыми числами номеров и станциями прибытия и отправления
    for segment in segments:
        segment.run_days = calculate_run_days(segment, days_ago=settings.SEARCH_DAYS_TO_PAST)
        _add_to_train_segments_dict(train_segments_dict, segment)

    for segments_list in train_segments_dict.values():
        # Добавляем в сегменты время отправления и количества дней хождения
        for segment in segments_list:
            segment.departure_time = segment.departure.time()
            _make_segment_days_number(segment)

        segments_list_without_twins = _join_twin_segments(segments_list)
        segments_list_without_twins.sort(key=lambda s: s.departure_time)

        # Разбиваем сегменты на группы
        segments_group = []
        main_segment = segments_list_without_twins[0]
        cur_time = segments_list_without_twins[0].departure_time
        for segment in segments_list_without_twins:
            # Сегменты попадают в одну группу, если времена отправления не сильно отличаются
            if _compare_times(segment.departure_time, cur_time):
                segments_group.append(segment)
                if _compare_days_number(main_segment, segment):
                    main_segment.is_sub_segment = True
                    segment.is_sub_segment = False
                    main_segment = segment
                else:
                    segment.is_sub_segment = True

            else:
                _add_subsegments(main_segment, segments_group)
                segment.is_sub_segment = False
                segments_group = [segment]
                main_segment = segment

            cur_time = segment.departure_time

        _add_subsegments(main_segment, segments_group)

    return [
        segment for segment in segments
        if not getattr(segment, 'is_sub_segment', False) and not getattr(segment, 'is_twin_segment', False)
    ]
