# coding: utf-8
from common.utils.geo import great_circle_distance_km, GreatCircleCalculationError
from common.utils.railway import get_railway_tz_by_point
from travel.rasp.morda_backend.morda_backend.search.parse_context import point as point_data_layer


def geosearch_result_json(result):
    """
    :type result: morda_backend.data_layer.geosearch_wrapper.GeosearchResult
    """
    return result_with_errors_json(result) if result.errors else result_without_errors_json(result)


def ambiguous_variant_json(variant, national_version, language):
    """
    :type variant: morda_backend.data_layer.geosearch_wrapper.AmbiguousVariant
    """
    point = variant.point
    omonim_title = point.L_omonim_title(show_district=True, lang=language, national_version=national_version)

    json = result_without_errors_point_json(point, language)
    json.update({
        'ambiguousTitle': {
            'title': omonim_title['title'],
            'additionalTitle': omonim_title.get('add')
        },
        'isSelected': variant.is_selected
    })
    return json


def ambiguous_variants_json(variants, national_version, language):
    return [ambiguous_variant_json(variant, national_version, language) for variant in variants]


def result_with_errors_point_json(point_name, point_key, point_slug=None):
    return {
        'key': point_key,
        'title': point_name,
        'slug': point_slug,
        'timezone': None,
        'country': None
    }


def result_without_errors_point_json(point, language):
    country = point_data_layer.get_point_country(point)
    region = point_data_layer.get_point_region(point)
    settlement = point_data_layer.get_point_settlement(point)
    return {
        'key': point.point_key,
        'title': point.L_title(lang=language),
        'timezone': point_data_layer.get_point_timezone(point),
        'country': point_country_json(country, language),
        'region': point_region_json(region, language),
        'settlement': point_settlement_json(settlement, language),
        'titleGenitive': point.L_title(lang=language, case='genitive', fallback=False),
        'titleAccusative': point.L_title(lang=language, case='accusative', fallback=False),
        'titleLocative': point.L_title(lang=language, case='locative', fallback=False),
        'preposition': point.L_title(lang=language, case='preposition_v_vo_na', fallback=False),
        'shortTitle': point.L_short_title(lang=language, fallback=False),
        'popularTitle': point.L_popular_title(lang=language, fallback=False),
        'slug': getattr(point, 'slug', None)
    }


def result_with_errors_json(geosearch_result):
    """
    :type geosearch_result: morda_backend.data_layer.geosearch_wrapper.GeosearchResult
    """
    input_context = geosearch_result.input_context
    original_context = geosearch_result.original_context

    if geosearch_result.point_from:
        from_json = result_without_errors_point_json(geosearch_result.point_from, input_context.language)
    else:
        from_json = result_with_errors_point_json(original_context.from_title, original_context.from_key,
                                                  original_context.from_slug)

    if geosearch_result.original_point_from:
        original_from_json = result_without_errors_point_json(geosearch_result.original_point_from,
                                                              input_context.language)
    else:
        original_from_json = from_json

    if geosearch_result.point_to:
        to_json = result_without_errors_point_json(geosearch_result.point_to, input_context.language)
    else:
        to_json = result_with_errors_point_json(original_context.to_title, original_context.to_key,
                                                original_context.to_slug)

    if geosearch_result.original_point_to:
        original_to_json = result_without_errors_point_json(geosearch_result.original_point_to,
                                                            input_context.language)
    else:
        original_to_json = to_json

    return {
        'transportType': input_context_transport_type_json(original_context),
        'from': from_json,
        'originalFrom': original_from_json,
        'to': to_json,
        'originalTo': original_to_json,
        'errors': geosearch_result.errors
    }


def result_without_errors_json(geosearch_result):
    """
    :type geosearch_result: morda_backend.data_layer.geosearch_wrapper.GeosearchResult
    """
    input_context = geosearch_result.input_context

    try:
        distance = great_circle_distance_km(geosearch_result.point_from, geosearch_result.point_to)
    except GreatCircleCalculationError:
        distance = None

    return {
        'transportType': input_context_transport_type_json(input_context),
        'from': result_without_errors_point_json(geosearch_result.point_from, input_context.language),
        'originalFrom': result_without_errors_point_json(geosearch_result.original_point_from, input_context.language),
        'to': result_without_errors_point_json(geosearch_result.point_to, input_context.language),
        'originalTo': result_without_errors_point_json(geosearch_result.original_point_to, input_context.language),
        'distance': distance,
        'sameSuburbanZone': geosearch_result.same_suburban_zone(),
        'errors': []
    }


def input_context_transport_type_json(input_context):
    """
    :type input_context: morda_backend.data_layer.geosearch_wrapper.InputSearchContext
    """
    return input_context.t_type.code if input_context.t_type else 'all'


def point_country_json(country, language):
    if not country:
        return {}

    return {
        'code': country.code,
        'title': country.L_title(lang=language),
        'railwayTimezone': get_railway_tz_by_point(country).zone
    }


def point_region_json(region, language):
    if not region:
        return {}

    return {
        'title': region.L_title(lang=language),
    }


def point_settlement_json(settlement, language):
    if not settlement:
        return {}

    return {
        'title': settlement.L_title(lang=language),
        'slug': getattr(settlement, 'slug', None),
        'key' : settlement.point_key
    }
