# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from copy import copy
from functools import reduce

from django.core.exceptions import ObjectDoesNotExist

from common.models.geo import Settlement, Station, Country
from common.models.transport import TransportType
from geosearch.views import pointtopoint
from geosearch.views.point import (
    SuburbanPointSearch, PointSearch, StopWordError, NoPointsError, TooShortError, InvalidPointKey, InvalidSlug
)
from geosearch.views.pointtopoint import SamePointError
from route_search.models import NearestSuburbanDirection
from travel.rasp.morda_backend.morda_backend.search.parse_context import geosearch_serialization


class InputSearchContext(object):
    def __init__(self, from_key, from_title, to_key, to_title, t_type_code, client_settlement_id,
                 national_version, language, nearest=False, from_slug=None, to_slug=None):
        """
        :param from_key: пункт отправления - ключ, пришедший из suggest'а: 'c213'
        :type from_key: str or unicode
        :param from_title: пункт отправления - текст, введенный пользователем: 'Москва', 'москва', 'moskva', 'moscow'
        :type from_title: str or unicode
        :param to_key: пункт прибытия - ключ
        :type to_key: str or unicode
        :param to_title: пункт прибытия - текст
        :type to_title: str or unicode
        :param t_type_code: код тип транспорта - 'bus', 'train' и т.п.
        :type t_type_code: str or unicode or None
        :param nearest: запрос от ближайжих электричек
        :type nearest: bool
        :param from_slug: human-friendly идентификатор пункта отправления, ex. 'ryazan', 'kaluga'
        :type from_slug: str or unicode
        :param to_slug: human-friendly идентификатор пункта назначения, ex. 'ryazan', 'kaluga'
        :type to_slug: str or unicode
        """
        self.from_slug = from_slug or None
        self.from_key = from_key or None
        self.from_title = from_title or None
        self.to_slug = to_slug or None
        self.to_key = to_key or None
        self.to_title = to_title or None
        self.national_version = national_version
        self.language = language
        self.nearest = nearest

        try:
            self.t_type = TransportType.objects.get(code=t_type_code) if t_type_code and t_type_code != 'all' else None
        except ObjectDoesNotExist as ex:
            raise TransportTypeNotFoundError(ex.message)

        try:
            self.client_settlement = Settlement.objects.get(id=client_settlement_id) if client_settlement_id else None
        except ObjectDoesNotExist as ex:
            raise ClientSettlementNotFoundError(ex.message)


class TransportTypeNotFoundError(Exception):
    pass


class ClientSettlementNotFoundError(Exception):
    pass


class GeosearchState(object):
    def __init__(self, input_context, point_list_from=None, point_list_to=None, errors=None):
        """
        :type input_context: InputSearchContext
        :type point_list_from: PointList
        :type point_list_to: PointList
        :param errors: [{'field': 'from', 'type': 'some_error'}, ...]
        """
        self.input_context = input_context
        self.point_list_from = point_list_from
        self.point_list_to = point_list_to
        self.errors = errors


class GeosearchResult(object):
    def __init__(self, state, original_context, original_point_from=None, original_point_to=None):
        """
        :type state: GeosearchState
        """
        self.input_context = state.input_context
        self.original_context = original_context
        self.point_from = state.point_list_from.point if state.point_list_from else None
        self.point_to = state.point_list_to.point if state.point_list_to else None
        self.original_point_from = original_point_from if original_point_from else self.point_from
        self.original_point_to = original_point_to if original_point_to else self.point_to
        self.errors = state.errors or []

    def no_ambiguous_errors(self):
        return not any(e['type'] == 'ambiguous' for e in self.errors)

    def same_suburban_zone(self):
        if self.errors:
            return False

        if isinstance(self.point_from, Country) or isinstance(self.point_to, Country):
            return False

        for d1 in self.point_from.get_externaldirections():
            for d2 in self.point_to.get_externaldirections():
                if d1.suburban_zone_id and d1.suburban_zone_id == d2.suburban_zone_id:
                    return True

        return False


class AmbiguousVariant(object):
    def __init__(self, point, is_selected):
        self.point = point
        self.is_selected = is_selected


def apply_processors(input_state, processors):
    """
    Применяет список процессоров к определенному промежуточному состоянию
    В случае, если какой-то процессор генерирует ошибки, выполнение последующих процессоров не выполняется.
    :type input_state: GeosearchState
    :param processors: список функций-процессоров
    :rtype: GeosearchState
    """

    def apply_processor(st, processor):
        return st if st.errors else processor(st)

    return reduce(apply_processor, processors, input_state)


def process_nearest(state):
    input_context = state.input_context
    if not input_context.nearest:
        return state

    if not (input_context.t_type and input_context.t_type.id == TransportType.SUBURBAN_ID):
        return state

    if (input_context.from_key or input_context.from_title or
            input_context.to_key or input_context.to_title):
        return state

    if input_context.client_settlement:
        try:
            nearest_direction = NearestSuburbanDirection.objects.get(settlement=input_context.client_settlement)
        except NearestSuburbanDirection.DoesNotExist:
            pass
        else:
            new_input_context = copy(input_context)
            new_input_context.from_key = nearest_direction.station.point_key
            new_input_context.to_key = nearest_direction.transport_center.point_key

            return GeosearchState(new_input_context, state.point_list_from, state.point_list_to)

    return GeosearchState(input_context, state.point_list_from, state.point_list_to, errors=[{
        'nearest': 'default_direction_not_found',
        'type': 'nearest_default_direction_not_found'
    }])


def init_point_lists(state):
    """
    Получает и валидирует списки PointList для пунктов отправления, прибытия.
    :type state: GeosearchState
    :rtype: GeosearchState
    """
    input_context = state.input_context

    point_list_from, from_error_type = safe_get_point_list(input_context.from_key, input_context.from_title,
                                                           input_context.t_type, input_context.from_slug)
    point_list_to, to_error_type = safe_get_point_list(input_context.to_key, input_context.to_title,
                                                       input_context.t_type, input_context.to_slug)

    # Не нашли ничего подходящего для пункта отправления или пункта прибытия - показываем ошибку
    if point_list_from is None or point_list_to is None:
        errors = []
        if from_error_type:
            errors.append({'fields': ['from'], 'type': from_error_type})
        if to_error_type:
            errors.append({'fields': ['to'], 'type': to_error_type})
        return GeosearchState(state.input_context, point_list_from, point_list_to, errors=errors)

    return GeosearchState(state.input_context, point_list_from, point_list_to)


def remove_countries_from_point_lists(state):
    """
    Вычищает страны из PointList'ов.
    :type state: GeosearchState
    :rtype: GeosearchState
    """
    point_list_from = remove_countries(state.point_list_from)
    point_list_to = remove_countries(state.point_list_to)
    errors = []

    if not point_list_from:
        errors.append({'fields': ['from'], 'type': 'point_not_found'})

    if not point_list_to:
        errors.append({'fields': ['to'], 'type': 'point_not_found'})

    if errors:
        return GeosearchState(state.input_context, state.point_list_from, state.point_list_to, errors=errors)

    return GeosearchState(state.input_context, point_list_from, point_list_to)


def process_point_lists(state, disable_reduce_from=False, disable_reduce_to=False):
    """
    Обрабатывает, фильтрует списки подходящих пунктов отправления, прибытия.
    При этом учитывает текущий город клиента и отношение пунктов прибытия, отправления друг к другу.
    :param state: GeosearchState
    :param disable_reduce_from: отключает сужение для списка "откуда"
    :type disable_reduce_from: bool
    :param disable_reduce_to: отключает сужение для списка "куда"
    :type disable_reduce_to: bool
    :rtype: GeosearchState
    """
    input_context = state.input_context
    try:
        point_list_from, point_list_to = pointtopoint.process_points_lists(
            state.point_list_from,
            state.point_list_to,
            client_city=input_context.client_settlement,
            suburban=(input_context.t_type.id == TransportType.SUBURBAN_ID if input_context.t_type else False),
            disable_reduce_from=disable_reduce_from,
            disable_reduce_to=disable_reduce_to
        )
        return GeosearchState(input_context, point_list_from, point_list_to)
    # В процессе фильтрации списков пунктов случилось так, что списки отфильтровались так,
    # что пункт отправления = пункту прибытия ==> показываем ошибку.
    except SamePointError:
        return GeosearchState(input_context, state.point_list_from, state.point_list_to,
                              errors=[{'fields': ['from', 'to'], 'type': 'same_points'}])


def check_ambiguous(state):
    """
    Проверяет, удалось ли однозначно определить пункт отправления и пункт прибтия.
    Если не удалось, возвращает список ошибок, иначе - пустой спсок.
    Пример получившегося списка ошибок:
    [
        {
            'fields': ['from'],
            'type': 'ambiguous',
            'variants': [
                {
                    'key': 'с77',
                    'title': 'Благовещенск',
                    'additionalTitle': 'Амурская область, Россия',
                    'timezone': 'Asia/Yakutsk'
                    'isSelected': False
                },
                {
                    'key': 'с11112',
                    'title': 'Благовещенск',
                    'additionalTitle': 'Республика Башкортостан, Россия',
                    'timezone': 'Asia/Yekaterinburg'
                    'isSelected': True
                }
            ]
        },
        {
            'fields': ['to'],
            'type': ambiguous,
            'variants': [...]
        }]
    :type state: GeosearchState
    :rtype: GeosearchState
    """
    input_context = state.input_context
    national_version = input_context.national_version
    errors = []

    def get_error(field_name, point_list, point_title):
        variants = get_ambiguous_variants(point_list, point_title, input_context.client_settlement, national_version)
        variants_json = geosearch_serialization.ambiguous_variants_json(variants, national_version,
                                                                        input_context.language)
        return {
            'fields': [field_name],
            'type': 'ambiguous',
            'variants': variants_json
        }

    if state.point_list_from.has_variants():
        errors.append(get_error('from', state.point_list_from, input_context.from_title))

    if state.point_list_to.has_variants():
        errors.append(get_error('to', state.point_list_to, input_context.to_title))

    if not errors:
        return state

    return GeosearchState(input_context, state.point_list_from, state.point_list_to, errors=errors)


def check_same_points(state):
    """
    Проверяет, совпадают ли найденные пункты отправления, прибытия. Если совпадают, возвращает список ошибок.
    :type state: GeosearchState
    :rtype: GeosearchState
    """
    if state.point_list_from.point == state.point_list_to.point:
        return GeosearchState(state.input_context, state.point_list_from, state.point_list_to,
                              errors=[{'fields': ['from', 'to'], 'type': 'same_points'}])
    return state


def safe_get_point_list(point_key, point_title, t_type, point_slug=None):
    """
    Получает PointList и код ошибки.
    :type point_key: str or unicode
    :type point_title: str or unicode
    :type t_type: TransportType
    :type point_slug: str or unicode
    :return: (PointList, None) или (None, error_type), где error_type - строка.
    """
    searcher = SuburbanPointSearch if t_type and t_type.id == TransportType.SUBURBAN_ID else PointSearch

    try:
        t_type_code = t_type.code if t_type else None
        return searcher.find_point(
            point_title, t_type=t_type_code, point_key=point_key, slug=point_slug, can_return_hidden=True
        ), None
    except StopWordError:
        return None, 'too_general'
    except (NoPointsError, InvalidPointKey, InvalidSlug):
        return None, 'point_not_found'
    except TooShortError:
        return None, 'too_short'
    except Station.DoesNotExist:
        return None, 'station_not_found'
    except Settlement.DoesNotExist:
        return None, 'settlement_not_found'


def remove_countries(point_list):
    """
    Удаляет страны из PointList'а.
    :type point_list: PointList
    """
    if isinstance(point_list.point, Country):
        point_list.point = None
        point_list.exact_variant = False

    if point_list.variants:
        point_list.variants = [variant for variant in point_list.variants if not isinstance(variant, Country)]

    if point_list.point or point_list.variants:
        return point_list
    return None


def get_ambiguous_variants(point_list, point_title, client_settlement, national_version):
    """
    'Процессит' (выполняет какую-то магию внутри geosearch) PointList и получает список неоднозначных вариантов.
    :type point_list: PointList
    :param point_title: текст, введенный пользователем
    :type client_settlement: Settlement
    :return: список вариантов, где вариант - AmbiguousVariant
    """
    point_list.process()
    if not point_list.dont_rearrange:
        point_list.rearrange_variants(client_settlement, point_title, national_version)

    has_variants = point_list.has_variants()

    return [AmbiguousVariant(variant_point, (not has_variants and variant_point == point_list.point))
            for variant_point in point_list.variants]


def tune_airport_point(point_list):
    """
    Подменяет наименование станции, если она является аэропортом
    https://st.yandex-team.ru/RASPFRONT-7570
    :type point_list: PointList
    """
    if isinstance(point_list.point, Station) and point_list.point.t_type_id == TransportType.PLANE_ID:
        new_airport = copy(point_list.point)
        if point_list.point.settlement and point_list.point.title_ru == point_list.point.settlement.title_ru:
            new_airport.title_ru = "Аэропорт ({})".format(point_list.point.title_ru)
            new_airport.title_ru_locative = "Аэропорту ({})".format(point_list.point.title_ru)
        else:
            new_airport.title_ru = "Аэропорт {}".format(point_list.point.title_ru)
            new_airport.title_ru_locative = "аэропорту {}".format(point_list.point.title_ru)
        new_airport.title_ru_accusative = new_airport.title_ru
        point_list.point = new_airport


def tune_airports(state):
    """
    Подменяет наименование станций-аэропортов начальной и конечной точки поиска
    https://st.yandex-team.ru/RASPFRONT-7570
    :type state: GeosearchState
    :rtype: GeosearchState
    """
    if state.input_context.t_type and state.input_context.t_type.id == TransportType.SUBURBAN_ID:
        tune_airport_point(state.point_list_from)
        tune_airport_point(state.point_list_to)
    return state
