# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response

from geosearch.views.pointlist import PointList

from travel.rasp.morda_backend.morda_backend.search.parse_context import geosearch_wrapper
from travel.rasp.morda_backend.morda_backend.search.parse_context.geosearch_wrapper import (
    InputSearchContext, process_nearest, init_point_lists, remove_countries_from_point_lists,
    process_point_lists, check_ambiguous, check_same_points, GeosearchState, tune_airports, GeosearchResult
)
from travel.rasp.morda_backend.morda_backend.search.parse_context import geosearch_serialization
from travel.rasp.morda_backend.morda_backend.search.parse_context.serialization import InputSearchContextSchema


log = logging.getLogger(__name__)


@api_view(['GET'])
def parse_context(request):
    input_context, errors = InputSearchContextSchema().load(request.GET)
    if errors:
        return Response({'errors': [{'field': f, 'type': t} for f, ts in errors.items() for t in ts]},
                        status=status.HTTP_400_BAD_REQUEST)

    collecting_stage = [process_nearest, init_point_lists, remove_countries_from_point_lists]
    processing_stage = [process_point_lists]
    validating_stage = [check_ambiguous, check_same_points]

    collecting_state = geosearch_wrapper.apply_processors(GeosearchState(input_context), collecting_stage)
    state = geosearch_wrapper.apply_processors(collecting_state, processing_stage)
    state = geosearch_wrapper.apply_processors(state, validating_stage)

    # При получении точек до сужения мы рассчитываем на тот эффект, что в списках в collecting_state
    # основные точки правильно для нас меняются при обработке до того момента, когда начинается сужение
    # (при сужении создается новый список и уже становится основным для дальнейшей обработки).
    # Такое решение приняли, т.к. пока не собираемся трогать geosearch
    # и не готовы идти на повторный запуск обработки без сужения.
    orig_point_from = collecting_state.point_list_from.point if collecting_state.point_list_from else None
    orig_point_to = collecting_state.point_list_to.point if collecting_state.point_list_to else None

    # after original points
    state = geosearch_wrapper.apply_processors(state, [tune_airports])

    result = GeosearchResult(state=state,
                             original_context=input_context,
                             original_point_from=orig_point_from,
                             original_point_to=orig_point_to)

    if result.errors and result.no_ambiguous_errors() and input_context.t_type:
        # Пробуем определить поисковый контекст для поиска по всем видам транспорта.
        # Если такой контекст будет валидный, то отдаем распознанные пункты отправления, прибытия
        # и ошибку "между пунктами нет маршрутов заданным типом траснпорта"
        all_t_type_context = InputSearchContext(input_context.from_key, input_context.from_title,
                                                input_context.to_key, input_context.to_title,
                                                None,
                                                input_context.client_settlement and input_context.client_settlement.id,
                                                input_context.national_version, input_context.language,
                                                input_context.nearest,
                                                from_slug=input_context.from_slug, to_slug=input_context.to_slug)

        all_t_type_validating_stage = [check_same_points]

        all_t_type_collecting_state = geosearch_wrapper.apply_processors(GeosearchState(all_t_type_context),
                                                                         collecting_stage)
        all_t_type_state = geosearch_wrapper.apply_processors(all_t_type_collecting_state, processing_stage)
        all_t_type_state = geosearch_wrapper.apply_processors(all_t_type_state, all_t_type_validating_stage)

        all_t_type_orig_point_from = (all_t_type_collecting_state.point_list_from.point
                                      if all_t_type_collecting_state.point_list_from
                                      else None)
        all_t_type_orig_point_to = (all_t_type_collecting_state.point_list_to.point
                                    if collecting_state.point_list_to
                                    else None)

        # after original points
        all_t_type_state = geosearch_wrapper.apply_processors(all_t_type_state, [tune_airports])

        all_t_type_result = GeosearchResult(state=all_t_type_state,
                                            original_context=all_t_type_context,
                                            original_point_from=all_t_type_orig_point_from,
                                            original_point_to=all_t_type_orig_point_to)

        if not all_t_type_result.errors:
            result = GeosearchResult(
                state=GeosearchState(
                    input_context,
                    PointList(all_t_type_result.point_from),
                    PointList(all_t_type_result.point_to),
                    errors=[{'type': 'no_routes'}]
                ),
                original_context=input_context,
                original_point_from=all_t_type_orig_point_from,
                original_point_to=all_t_type_orig_point_to
            )

            return Response(geosearch_serialization.geosearch_result_json(result))

    return Response(geosearch_serialization.geosearch_result_json(result))
