# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from marshmallow import Schema, fields

from route_search.transfers.transfer_segment import BarisTransferSegment
from travel.rasp.morda_backend.morda_backend.search.parse_context.point import get_point_timezone
from travel.rasp.morda_backend.morda_backend.serialization.fields import AwareDateTime


class PMSettlementSchema(Schema):
    id = fields.Integer()
    title = fields.Function(lambda obj: obj.L_title())


class PMStationSchema(Schema):
    id = fields.Integer()
    title = fields.Function(lambda obj: obj.L_title())
    timezone = fields.Function(lambda obj: get_point_timezone(obj))
    settlement = fields.Nested(PMSettlementSchema)


class RMTransportSchema(Schema):
    id = fields.Integer()
    code = fields.String()


class BasePMThreadSchema(Schema):
    uid = fields.String()
    number = fields.String()


class BarisPMThreadSchema(BasePMThreadSchema):
    title = fields.String()


class RaspDbPMThreadSchema(BasePMThreadSchema):
    title = fields.Function(lambda obj: obj.L_title())


class BasePMSegmentSchema(Schema):
    departure = AwareDateTime()
    station_from = fields.Nested(PMStationSchema, dump_to='departure_station')
    arrival = AwareDateTime()
    station_to = fields.Nested(PMStationSchema, dump_to='arrival_station')


class BarisPMSegmentSchema(BasePMSegmentSchema):
    transport = fields.Method('build_transport')
    thread = fields.Nested(BarisPMThreadSchema)

    def build_transport(self, segment):
        transport = segment.transport
        return {'id': transport.id, 'code': transport.code}


class RaspDbPMSegmentSchema(BasePMSegmentSchema):
    transport = fields.Method('build_transport')
    thread = fields.Nested(RaspDbPMThreadSchema)

    def build_transport(self, segment):
        t_type = segment.t_type
        return {'id': t_type.id, 'code': t_type.code}


class PMVariantResponseSchema(Schema):
    segments = fields.Method('_make_segments')

    def _make_segments(self, response):
        segments = []
        for segment in response.segments:
            if isinstance(segment, BarisTransferSegment):
                data, _ = BarisPMSegmentSchema(context=self.context).dump(segment)
            else:
                data, _ = RaspDbPMSegmentSchema(context=self.context).dump(segment)
                full_path = getattr(segment, 'full_path', None)
                if full_path is not None:
                    data['thread']['full_path'] = full_path
            segments.append(data)
        return segments


class PMVariantsResponseSchema(Schema):
    pm_variants = fields.Nested(PMVariantResponseSchema, many=True, dump_to='variants')
