# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from travel.rasp.morda_backend.morda_backend.search.search.data_layer.base_search import SearchResult, BaseSearch
from travel.rasp.morda_backend.morda_backend.search.search.data_layer.rasp_db_search import (
    RaspDbNearestSearch, RaspDbAllDaysSearch, RaspDbOneDaySearch
)
from travel.rasp.morda_backend.morda_backend.search.search.data_layer.baris_search import (
    BarisNearestSearch, BarisAllDaysSearch, BarisOneDaySearch
)


logger = logging.getLogger(__name__)


class AllTypesBaseSearch(BaseSearch):
    """
    Базовый класс для поиска всеми типами транспорта
    """
    def __init__(self, original_context, rasp_db_search, baris_search):
        super(AllTypesBaseSearch, self).__init__(original_context)
        self.rasp_db_search = rasp_db_search
        self.baris_search = baris_search

    def find_segments(self, context):
        """
        Поиск сегментов в базе расписаний и в БАРиС
        """
        self.rasp_db_search.current_plan = self.current_plan
        self.rasp_db_search.next_plan = self.next_plan

        rasp_db_result = self.rasp_db_search.find_segments(context)
        baris_result = self.baris_search.find_segments(context)

        if not baris_result.segments:
            return rasp_db_result

        if not rasp_db_result.segments:
            return baris_result

        return self.merge_results(context, rasp_db_result, baris_result)

    def merge_results(self, context, rasp_db_result, baris_result):
        """
        Склейка списков рейсов из базы расписаний и БАРиС. Должна быть переопределена в наследниках
        return: объект класса SearchResult
        """
        raise NotImplementedError()

    def get_canonical(self, context, rasp_db_result, baris_result):
        if not baris_result.canonical:
            return rasp_db_result.canonical

        if not rasp_db_result.canonical:
            return baris_result.canonical

        return {
            'point_from': context.point_from,
            'point_to': context.point_to,
            'transport_type': None
        }


class AllTypesNearestSearch(AllTypesBaseSearch):
    """
    Поиск ближайших рейсов всеми типами транспорта, чуть более чем за сутки
    """
    def __init__(self, original_context):
        rasp_db_search = RaspDbNearestSearch(original_context)
        baris_search = BarisNearestSearch(original_context)
        super(AllTypesNearestSearch, self).__init__(original_context, rasp_db_search, baris_search)

    def merge_results(self, context, rasp_db_result, baris_result):
        segments = rasp_db_result.segments + baris_result.segments
        segments.sort(key=lambda s: s.departure)

        transport_types = set(s.t_type.code for s in rasp_db_result.segments)
        if baris_result.segments:
            transport_types.add('plane')
        latest_datetime = self.get_nearest_search_latest_datetime(segments)

        return SearchResult(segments, transport_types, latest_datetime, canonical=None)


class AllTypesOneDaySearch(AllTypesBaseSearch):
    """
    Поиск на один день всеми типами транспорта
    """
    def __init__(self, original_context):
        rasp_db_search = RaspDbOneDaySearch(original_context)
        baris_search = BarisOneDaySearch(original_context)
        super(AllTypesOneDaySearch, self).__init__(original_context, rasp_db_search, baris_search)

    def merge_results(self, context, rasp_db_result, baris_result):
        segments = rasp_db_result.segments + baris_result.segments
        segments.sort(key=lambda s: s.departure)

        latest_datetime = rasp_db_result.latest_datetime
        transport_types = rasp_db_result.transport_types | baris_result.transport_types
        canonical = self.get_canonical(context, rasp_db_result, baris_result)

        return SearchResult(segments, transport_types, latest_datetime, canonical)


class AllTypesAllDaysSearch(AllTypesBaseSearch):
    """
    Поиск на все дни всеми типами транспорта
    """
    def __init__(self, original_context):
        rasp_db_search = RaspDbAllDaysSearch(original_context)
        baris_search = BarisAllDaysSearch(original_context)
        super(AllTypesAllDaysSearch, self).__init__(original_context, rasp_db_search, baris_search)

    def merge_results(self, context, rasp_db_result, baris_result):
        segments = rasp_db_result.segments + baris_result.segments
        segments.sort(key=lambda s: s.departure.time())

        transport_types = rasp_db_result.transport_types | baris_result.transport_types
        canonical = self.get_canonical(context, rasp_db_result, baris_result)

        return SearchResult(segments, transport_types, None, canonical)
