# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from dateutil import parser

from common.data_api.baris.base_avia_segment import BaseAviaSegment


class SearchAviaSegment(BaseAviaSegment):
    """
    Базовый класс для сегмента авиа поиска
    """
    def __init__(self, baris_flight, baris_data):
        super(SearchAviaSegment, self).__init__(baris_flight, baris_data)

        # Значения по-умолчанию для совместимости
        self.is_interval = False
        self.is_through_train = False
        self.suburban_facilities = None
        self.arrival_event = None
        self.departure_event = None
        self.arrival_event_key = None
        self.departure_event_key = None
        self.has_train_tariffs = None
        self.stops = ''

    def make_time_values(self):
        self.duration = self.arrival - self.departure
        self.start_date = self.departure.date()


class OneDayAviaSegment(SearchAviaSegment):
    def __init__(self, baris_flight, baris_data, is_nearest):
        super(OneDayAviaSegment, self).__init__(baris_flight, baris_data)

        self.departure = parser.parse(baris_flight['departureDatetime'])
        self.arrival = parser.parse(baris_flight['arrivalDatetime'])
        self.make_time_values()
        self.tariffs_keys = [self.number] if is_nearest else self._get_tariffs_keys(self.number)

        codeshares = baris_flight.get('codeshares', None)
        if not is_nearest and codeshares:
            self.codeshares = []
            for codeshare in codeshares:
                codeshare_data = {
                    'number': codeshare['title'],
                    'tariffs_keys': self._get_tariffs_keys(codeshare['title'])
                }
                if codeshare['airlineID'] in baris_data.companies_by_ids:
                    codeshare_data['company'] = baris_data.companies_by_ids[codeshare['airlineID']]

                self.codeshares.append(codeshare_data)

    def _get_tariffs_keys(self, number):
        """
        Ключ для тикет-демона
        """
        number_for_tariff = number.replace(' ', '-')
        departure_code = self.departure.strftime("%m%d")
        return ['daemon {} {}'.format(number_for_tariff, departure_code)]


class AllDaysAviaSegment(SearchAviaSegment):
    def __init__(self, baris_flight, baris_data):
        super(AllDaysAviaSegment, self).__init__(baris_flight, baris_data)

        self.days_by_tz = {
            self.station_from.time_zone: {'days_text': baris_flight['daysText']}
        }
        self.run_days = baris_flight['runDays']
        self.departure = baris_flight['departure']
        self.arrival = baris_flight['arrival']

        self.make_time_values()

        self.tariffs_keys = [self.number]  # Ключ для тикет-демона
