# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.conf import settings
from django.http import QueryDict
from django.utils.encoding import force_text

from common.dynamic_settings.default import conf
from common.models.geo import Country, Region
from common.models.schedule import RThreadType
from travel.rasp.library.python.common23.date import environment
from common.apps.suburban_events.api import get_states_for_segments, get_segments_keys, get_cancelled_path_for_segments
from common.data_api.platforms.helpers import SegmentPlatformsBatch
from common.data_api.platforms.instance import platforms as platforms_client
from common.models.transport import TransportType


log = logging.getLogger(__name__)


def calculate_days_by_tz(segment, timezones, next_plan):
    result = {}
    for timezone in timezones:
        if segment.thread.type_id == RThreadType.INTERVAL_ID:
            # Для интервальных сегментов не нужно сдвигать дни хождения,
            # иначе сдвиг будет по окончанию интервала хождения, что не правильно.
            shift = 0
            thread_start_date = environment.now_aware().astimezone(segment.thread.pytz).date()
        else:
            segment.rtstation_from.thread = segment.thread
            shift = segment.rtstation_from.calc_days_shift(
                event='departure',
                start_date=segment.calculated_start_date,
                event_tz=timezone
            )
            thread_start_date = segment.calculated_start_date
        days_data = segment.thread.L_days_text_dict(
            shift=shift,
            thread_start_date=thread_start_date,
            next_plan=next_plan,
            show_days=True,
            force_recalculate_days_text=getattr(segment, 'force_recalculate_days_text', False)
        )
        result[str(timezone)] = days_data
    return result


def add_days_by_tz(segments, timezones, next_plan):
    for segment in segments:
        segment.days_by_tz = calculate_days_by_tz(segment, timezones, next_plan)
        for sub_segment in getattr(segment, 'sub_segments', []):
            sub_segment.days_by_tz = calculate_days_by_tz(sub_segment, timezones, next_plan)


def add_railway_sales_limit(segments, national_version='ru'):
    for segment in segments:
        if segment.t_type.id == TransportType.SUBURBAN_ID:
            segment.sales_limit_in_days = conf.SUBURBAN_ORDER_DEFAULT_DEPTH_OF_SALES

        elif segment.t_type.id == TransportType.TRAIN_ID:
            segment.sales_limit_in_days = conf.TRAIN_ORDER_DEFAULT_DEPTH_OF_SALES

            if national_version == 'ua':
                if (
                    segment.station_from.country_id == Country.UKRAINE_ID or
                    segment.station_to.country_id == Country.UKRAINE_ID
                ):
                    segment.sales_limit_in_days = settings.UKRMINTRANS_TRAIN_DEFAULT_DEPTH_OF_SALES


def get_suburban_schedule_plan(thread, current_plan, next_plan):
    """
    :type thread: common.models.schedule.RThread
    :type current_plan: common.models.schedule.TrainSchedulePlan
    :type next_plan: common.models.schedule.TrainSchedulePlan
    :rtype: common.models.schedule.TrainSchedulePlan
    """

    if thread.t_type.id != TransportType.SUBURBAN_ID:
        return None

    if thread.schedule_plan:
        return thread.schedule_plan

    def is_thread_in_plan(first_run_date, last_run_date, plan):
        return first_run_date <= plan.end_date and last_run_date >= plan.start_date

    run_days = thread.get_run_date_list()
    first_run_date = run_days[0]
    last_run_date = run_days[-1]

    in_current_plan = is_thread_in_plan(first_run_date, last_run_date, current_plan)
    in_next_plan = is_thread_in_plan(first_run_date, last_run_date, next_plan)

    if in_current_plan and not in_next_plan:
        return current_plan

    if in_next_plan and not in_current_plan:
        return next_plan


def is_kaliningrad_train(segment):
    if segment.t_type.id != TransportType.TRAIN_ID:
        return False

    if segment.station_from.get_country_id() != Country.RUSSIA_ID:
        return False

    if segment.station_to.get_country_id() != Country.RUSSIA_ID:
        return False

    station_from_in_kaliningrad = segment.station_from.region_id == Region.KALININGRAD_REGION_ID
    station_to_in_kaliningrad = segment.station_to.region_id == Region.KALININGRAD_REGION_ID

    if station_from_in_kaliningrad and not station_to_in_kaliningrad:
        return True

    if not station_from_in_kaliningrad and station_to_in_kaliningrad:
        return True

    return False


def mark_kaliningrad_trains(segments):
    """RASPFRONT-2690 UFS запрещает продавать билеты из России в Калининградскую область через интернет"""
    for segment in segments:
        segment.do_not_sell = is_kaliningrad_train(segment)


def get_avia_company_url(company_yandex_avia_code, tld, utm_source='rasp', utm_medium='rasp_airline'):
    host = settings.NATIONAL_AVIA_DOMAINS.get(tld, settings.NATIONAL_AVIA_DOMAINS['ru'])
    params = QueryDict('', mutable=True)
    params['utm_source'] = utm_source
    params['utm_medium'] = utm_medium
    return 'https://{host}/airline/{company_yandex_avia_code}/?{params}'.format(
        host=host,
        company_yandex_avia_code=company_yandex_avia_code,
        params=params.urlencode()
    )


def update_platforms_by_dynamic(segments):
    if not segments:
        return

    platform_segments = [
        seg for seg in segments
        if seg.t_type.id in [TransportType.SUBURBAN_ID, TransportType.TRAIN_ID] and seg.departure
    ]
    if not platform_segments:
        return

    dynamic_platforms = SegmentPlatformsBatch()
    dynamic_platforms.try_load(platforms_client, platform_segments)

    for seg in platform_segments:
        seg.rtstation_from.platform = dynamic_platforms.get_departure(seg, seg.rtstation_from.platform)


def fill_suburban_events(segments):
    try:
        segments_states = get_states_for_segments(segments, cancels_as_possible_delay=False)
    except Exception:
        log.exception('Незвестная ошибка при получении данных об опозданиях по сегментам')
        segments_states = {}

    for segment, segment_events in segments_states.items():
        segment.arrival_event = _fix_event_state(segment_events.arrival)  # SegmentEventState
        segment.departure_event = _fix_event_state(segment_events.departure)


def _fix_event_state(event_state):
    if not event_state:
        return None
    if not event_state.type:
        return None

    return event_state


def fill_suburban_event_keys(segments):
    try:
        segments_keys = get_segments_keys(segments)
    except Exception:
        log.exception('Незвестная ошибка при получении ключей опозданий для сегментов')
        segments_keys = {}

    for segment, value in segments_keys.items():
        segment.departure_event_key = force_text(value['from'])
        segment.arrival_event_key = force_text(value['to'])


def fill_thread_cancels(segments):
    try:
        segments_with_cancels = [
            segment for segment in segments
            if (
                (segment.departure_event and segment.departure_event.type == 'cancelled') or
                (segment.arrival_event and segment.arrival_event.type == 'cancelled')
            )
        ]
        thread_cancels = get_cancelled_path_for_segments(segments_with_cancels)
    except Exception:
        log.exception('Незвестная ошибка при получении данных об отменах на нитке')
        thread_cancels = {}

    for segment, thread_state in thread_cancels.items():
        segment.thread.fully_cancelled = thread_state.is_fully_cancelled
        segment.thread.cancelled_segments = [
            {
                'station_from': cancelled_segment.rtstation_from.station,
                'station_to': cancelled_segment.rtstation_to.station
            }
            for cancelled_segment in thread_state.cancelled_segments
        ]
