# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import timedelta, datetime, time

from django.utils.translation import get_language

from travel.rasp.library.python.common23.date import environment
from common.data_api.baris.helpers import get_plane_stations_ids, BarisData
from common.data_api.baris.instance import baris

from travel.rasp.morda_backend.morda_backend.search.search.data_layer.base_search import SearchResult
from travel.rasp.morda_backend.morda_backend.search.search.data_layer.avia_segments import (
    AllDaysAviaSegment, OneDayAviaSegment
)
from travel.rasp.morda_backend.morda_backend.search.search.data_layer.base_search import BaseSearch


logger = logging.getLogger(__name__)


class BarisBaseSearch(BaseSearch):
    """
    Базовый класс для поиска авиарейсов
    """
    def __init__(self, original_context):
        super(BarisBaseSearch, self).__init__(original_context)

    def find_segments(self, context):
        """
        Поиск сегментов
        """
        station_from_ids = get_plane_stations_ids(context.point_from)
        station_to_ids = get_plane_stations_ids(context.point_to)

        segments, latest_datetime, canonical = self.get_segments_from_baris(context, station_from_ids, station_to_ids)
        transport_types = {'plane'} if segments else None

        return SearchResult(segments, transport_types, latest_datetime, canonical)

    def get_segments_from_baris(self, context, station_from_ids, station_to_ids):
        """
        Получение списка сегментов из БАРиС, должно быть переопределено в наследниках
        """
        raise NotImplementedError()

    def get_canonical(self, context, segments):
        if not segments:
            return None

        return {
            'point_from': context.point_from,
            'point_to': context.point_to,
            'transport_type': 'plane'
        }


class BarisNearestSearch(BarisBaseSearch):
    """
    Поиск ближайших авиарейсов, чуть более чем за сутки
    """
    def __init__(self, original_context):
        super(BarisNearestSearch, self).__init__(original_context)

    def get_segments_from_baris(self, context, station_from_ids, station_to_ids):
        """
        Логика скопирована из find_next в route_search
        Рейсы берутся не более чем за сутки с малым
        """
        now = environment.now_aware()
        day_start = now.replace(hour=0, minute=0, second=0)
        four_at_next_night = day_start + timedelta(days=1, hours=4)
        max_limit = max(now + timedelta(days=1), four_at_next_night)

        baris_data = BarisData(baris.get_p2p_search(
            station_from_ids, station_to_ids,
            after=datetime.combine(day_start, time(0)), before=datetime.combine(max_limit, time(0)),
            national_version=context.national_version
        ))

        segments = [OneDayAviaSegment(flight, baris_data, is_nearest=True) for flight in baris_data.flights]
        latest_datetime = self.get_nearest_search_latest_datetime(segments)
        return segments, latest_datetime, None


class BarisOneDaySearch(BarisBaseSearch):
    """
    Поиск авиарейсов на один день
    """
    def __init__(self, original_context):
        super(BarisOneDaySearch, self).__init__(original_context)

    def get_segments_from_baris(self, context, station_from_ids, station_to_ids):
        after = datetime.combine(self.when_date, time(0))
        before = datetime.combine(self.when_date + timedelta(days=1, hours=4), time(0))
        baris_data = BarisData(baris.get_p2p_search(
            station_from_ids, station_to_ids,
            after=after, before=before,
            national_version=context.national_version
        ))

        segments = [OneDayAviaSegment(flight, baris_data, is_nearest=False) for flight in baris_data.flights]
        latest_datetime = self.get_usual_search_latest_datetime(context)
        canonical = self.get_canonical(context, segments)
        return segments, latest_datetime, canonical


class BarisAllDaysSearch(BarisBaseSearch):
    """
    Поиск авиарейсов на все дни
    """
    def __init__(self, original_context):
        super(BarisAllDaysSearch, self).__init__(original_context)

    def get_segments_from_baris(self, context, station_from_ids, station_to_ids):
        baris_data = BarisData(baris.get_p2p_all_days_search(
            get_language(), station_from_ids, station_to_ids,
            context.point_from.time_zone, context.point_to.time_zone, context.national_version
        ))

        segments = [AllDaysAviaSegment(flight, baris_data) for flight in baris_data.flights]
        segments.sort(key=lambda s: s.departure.time())
        canonical = self.get_canonical(context, segments)
        return segments, None, canonical
