# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import datetime, timedelta
from operator import attrgetter

from django.conf import settings

from common.apps.archival_data.api import get_archival_data
from common.models.schedule import TrainSchedulePlan
from common.utils.date import MSK_TZ
from geosearch.views.pointtopoint import iter_extended_points
from route_search import shortcuts
from route_search.models import IntervalRThreadSegment
from travel.rasp.library.python.common23.date import environment
from travel.rasp.morda_backend.morda_backend.search.search.data_layer.banner import get_banner_info
from travel.rasp.morda_backend.morda_backend.search.search.data_layer.teasers import get_teasers


logger = logging.getLogger(__name__)


class SearchResult(object):
    """
    Данные результата поиска, возвращаются из функции find_segments
    """
    def __init__(self, segments=None, transport_types=None, latest_datetime=None, canonical=None):
        self.segments = segments if segments else []
        self.transport_types = transport_types if transport_types else set()
        self.latest_datetime = latest_datetime
        self.canonical = canonical
        self.search_context = None


class BaseSearch(object):
    """
    Базовый класс для поиска сегментов
    """
    def __init__(self, original_context):
        self.context = original_context
        self.when_date = self._get_search_date(original_context)
        self.current_plan = None
        self.next_plan = None

    def _get_search_date(self, context):
        """
        Получение даты из контекста поиска
        """
        if not context.when:
            return None
        if context.when not in {'today', 'tomorrow'}:
            return datetime.strptime(context.when, "%Y-%m-%d").date()

        dt_now = context.point_from.local_time(environment.now_aware())
        today = dt_now.date()

        if context.when == 'today':
            return today

        if context.when == 'tomorrow':
            return today + timedelta(days=1)

    def find_segments(self, context):
        """
        Поиск сегментов, должен переопределяться в наследниках
        """
        raise NotImplementedError()

    def get_nearest_search_latest_datetime(self, segments):
        """
        Получение последнего времени сегментов для поисков Nearest
        """
        rthread_segments = [s for s in segments if not isinstance(s, IntervalRThreadSegment)]
        return (
            None if not rthread_segments
            else max(rthread_segments, key=attrgetter('departure')).departure
        )

    def get_usual_search_latest_datetime(self, context):
        """
        Получение последнего времени сегментов для поиска на день и на все дни
        """
        if self.when_date:
            _, latest_datetime = shortcuts.get_loc_search_range_aware(context.point_from, self.when_date, True)
            return latest_datetime
        return None

    def _extended_search(self):
        """
        Поиск по двум точкам, с последовательным обобщением до городов в случае неудачи
        """
        result = self.find_segments(self.context)

        if result.segments or not self.context.allow_change_context:
            return result, self.context

        # Обобщение до городов
        points_iterator = iter_extended_points(
            self.context.point_from,
            self.context.point_to,
            self.context.transport_type == 'suburban'
        )
        for extended_point_from, extended_point_to in points_iterator:
            search_context = self.context._replace(
                point_from=extended_point_from,
                point_to=extended_point_to
            )
            result = self.find_segments(search_context)
            if result.segments:
                return result, search_context

        return result, self.context

    def _make_schedule_plans(self):
        """
        Получение планов расписаний электричек
        """
        now = self.when_date or environment.now_aware()
        self.current_plan, self.next_plan = TrainSchedulePlan.get_current_and_next(
            now.astimezone(MSK_TZ).date() if isinstance(now, datetime) else now
        )

    def _get_archival_data(self, segments, search_context):
        if segments and not settings.ARCHIVAL_DATA_SHOW_ALWAYS:
            return None

        return get_archival_data(
            search_context.point_from,
            search_context.point_to,
            {self.context.transport_type} if self.context.transport_type else None,
            not self.when_date
        )

    def search(self):
        """
        Главная функция, выполняющая поиск. Возвращает ответ, готовый к сериализации
        """
        self._make_schedule_plans()
        result, search_context = self._extended_search()
        teasers = get_teasers(self.context, result.segments)
        archival_data = self._get_archival_data(result.segments, search_context)

        result = {
            'segments': result.segments,
            'plans': {
                'current': self.current_plan,
                'next': self.next_plan
            },
            'context': {
                'is_changed': self.context != search_context,
                'original': self.context,
                'search': search_context,
                'transport_types': result.transport_types,
                'latest_datetime': result.latest_datetime
            },
            'teasers': teasers,
            'canonical': result.canonical,
            'archival_data': archival_data
        }

        banner_info = get_banner_info(self.context)
        if banner_info:
            result['banner_info'] = banner_info

        return result
