# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.conf import settings

from common.dynamic_settings.default import conf
from common.dynamic_settings.core import DynamicSetting
from common.models.geo import Settlement
from common.models.transport import TransportType, TransportSubtype
from common.models.schedule import RThreadType, RThread
from common.models_utils import fetch_related
from common.models_utils.i18n import RouteLTitle
from travel.rasp.library.python.common23.date import environment
from common.utils.date import calculate_run_days

from route_search.base import get_threads_from_znoderoute
from route_search.helpers import remove_duplicates
from route_search import shortcuts
from route_search.facilities import fill_suburban_facilities

from travel.rasp.morda_backend.morda_backend.search.segment import (
    fill_ufs_old_order, mark_train_with_subtype_as_express, fill_thread_first_last_country_code, fill_express_codes_for_segments
)
from travel.rasp.morda_backend.morda_backend.search.merge_trains import (
    fill_train_numbers, make_meta_trains, make_grouped_trains
)
from travel.rasp.morda_backend.morda_backend.search.search.data_layer.backend import (
    add_days_by_tz, add_railway_sales_limit, fill_suburban_event_keys, fill_suburban_events, fill_thread_cancels,
    get_suburban_schedule_plan, mark_kaliningrad_trains, update_platforms_by_dynamic
)
from travel.rasp.morda_backend.morda_backend.search.search.data_layer.base_search import BaseSearch, SearchResult
from travel.rasp.morda_backend.morda_backend.data_layer.segments_helpers import fetch_and_filter_train_purchase_numbers


logger = logging.getLogger(__name__)

conf.register_settings(
    RASP_SEARCH_THREAD_LIMIT=DynamicSetting(
        5000,
        cache_time=10 * 60,
        description='Ограничение на количество ниток в результате поиска')
)


class RaspDbBaseSearch(BaseSearch):
    """
    Базовый класс для поиска рейсов в базе расписаний (т.е. кроме авиа)
    """
    def __init__(self, original_context):
        super(RaspDbBaseSearch, self).__init__(original_context)

    def get_transport_types_for_search(self, t_type):
        """
        Получение списка типов транспорта и дополнительных подтипов
        """
        transport_types = t_type and [TransportType.objects.get(code=t_type)]

        if (transport_types and TransportType.get_suburban_type() in transport_types
                and TransportType.get_train_type() not in transport_types):
            add_train_subtypes = TransportSubtype.get_train_search_codes()
        else:
            add_train_subtypes = None

        return transport_types, add_train_subtypes

    def find_segments(self, context):
        """
        Формирование списка сегментов
        """
        result = self.get_segments(context)
        if result.segments:
            result.segments = self.update_segments(result.segments, result.transport_types)
            result.segments = self.update_segments_specially(result.segments, context)
        return result

    def get_segments(self, context):
        """
        Получение сегментов из базы. Должно быть переопределено в наследниках
        """
        raise NotImplementedError()

    def update_segments_specially(self, segments, context):
        """
        Дополнительное обновление сегментов. Может быть переопределено в наследниках
        """
        return segments

    def update_segments(self, segments, transport_types):
        """
        Разные фильтры и обновления списка сегментов
        """

        # Убираем нитки самолетов, они должны браться из БАРиС
        if 'plane' in transport_types:
            segments = [s for s in segments if s.t_type.id != TransportType.PLANE_ID]

        fetch_related(
            [segment.thread for segment in segments if segment and getattr(segment, 'thread', None)],
            't_subtype', 'schedule_plan', 't_model',
            model=RThread
        )

        segments = fetch_and_filter_train_purchase_numbers(segments)

        if self.current_plan and self.next_plan:
            for segment in segments:
                thread = segment.thread
                if segment.t_type.id == TransportType.SUBURBAN_ID and not thread.schedule_plan:
                    thread.schedule_plan = get_suburban_schedule_plan(thread, self.current_plan, self.next_plan)

        fill_ufs_old_order(segments)
        mark_kaliningrad_trains(segments)
        mark_train_with_subtype_as_express(segments)
        fill_thread_first_last_country_code(
            segments, lambda s: s.thread and s.t_type.id in TransportType.RAILWAY_TTYPE_IDS
        )

        RouteLTitle.fetch([s.thread.L_title for s in segments])
        fill_express_codes_for_segments(segments)

        return segments


class RaspDbNearestSearch(RaspDbBaseSearch):
    """
    Ищет подходящие сегменты на ближайшее время.
    """
    def __init__(self, original_context):
        super(RaspDbNearestSearch, self).__init__(original_context)

    def get_segments(self, context):
        """
        Получение списка сегментов из базы
        """
        transport_types, add_train_subtypes = self.get_transport_types_for_search(context.transport_type)

        segments = list(shortcuts.find_next(
            context.point_from,
            context.point_to,
            environment.now_aware(),
            transport_types,
            add_train_subtypes=add_train_subtypes
        ))

        if not segments:
            return SearchResult()

        transport_types = set(s.t_type.code for s in segments)
        latest_datetime = self.get_nearest_search_latest_datetime(segments)

        return SearchResult(segments, transport_types, latest_datetime, canonical=None)


class RaspDbUsualSearch(RaspDbBaseSearch):
    """
    Обычный поиск - ищет все сегменты на указанную дату или на все дни
    """
    def __init__(self, original_context):
        super(RaspDbUsualSearch, self).__init__(original_context)

    def get_segments(self, context):
        """
        Получение списка сегментов из базы
        """
        transport_types, add_train_subtypes = self.get_transport_types_for_search(context.transport_type)

        prepared_threads = get_threads_from_znoderoute(
            context.point_from, context.point_to,
            transport_types, add_train_subtypes=add_train_subtypes,
            limit=conf.RASP_SEARCH_THREAD_LIMIT
        )

        segments, _, transport_types = shortcuts.search_routes(
            context.point_from,
            context.point_to,
            departure_date=self.when_date,
            transport_types=transport_types,
            add_train_subtypes=add_train_subtypes,
            check_date=None,
            include_interval=True,
            add_z_tablos=False,
            prepared_threads=prepared_threads
        )

        latest_datetime = self.get_usual_search_latest_datetime(context)
        if not segments:
            return SearchResult(latest_datetime=latest_datetime)

        transport_types = set(s.t_type.code for s in segments)
        canonical = self._get_canonical(context, prepared_threads)

        return SearchResult(segments, transport_types, latest_datetime, canonical)

    def _get_canonical(self, context, prepared_threads):
        """
        Информация для формирования канонической ссылки на поиск
        """
        threads = [thread for thread in prepared_threads
                   if thread.type_id != RThreadType.CANCEL_ID and getattr(thread, 'station_from', None)]
        if not threads:
            return None

        t_types = {thread.t_type.code for thread in threads}

        def get_same_title_point(stations, point):
            if len(stations) == 1:
                station = stations.pop()
                # По аналогии с обычным сужением
                # https://a.yandex-team.ru/arc/trunk/arcadia/travel/rasp/library/python/geosearch/views/pointtopoint.py?rev=r6342410#L386
                if station.title.lower() == point.title.lower():
                    return station

        if t_types.issubset({'train', 'suburban'}):
            threads = list(remove_duplicates(threads))
            stations_from = {thread.station_from for thread in threads}
            stations_to = {thread.station_to for thread in threads}

            if (t_types == {TransportType.get_train_type().code} and
                    isinstance(context.point_from, Settlement) and
                    isinstance(context.point_to, Settlement)):
                station_from_same_title = get_same_title_point(stations_from, context.point_from)
                station_to_same_title = get_same_title_point(stations_to, context.point_to)

                if station_from_same_title and station_to_same_title:
                    point_from = station_from_same_title
                    point_to = station_to_same_title
                else:
                    point_from = context.point_from
                    point_to = context.point_to

                return {'point_from': point_from,
                        'point_to': point_to,
                        'transport_type': t_types.pop()}
            else:
                return {'point_from': stations_from.pop() if len(stations_from) == 1 else context.point_from,
                        'point_to': stations_to.pop() if len(stations_to) == 1 else context.point_to,
                        'transport_type': t_types.pop() if len(t_types) == 1 else context.transport_type}

        return {'point_from': context.point_from,
                'point_to': context.point_to,
                'transport_type': t_types.pop() if len(t_types) == 1 else context.transport_type}


class RaspDbOneDaySearch(RaspDbUsualSearch):
    """
    Обычный поиск - ищет все сегменты на указанную дату
    """
    def __init__(self, original_context):
        super(RaspDbOneDaySearch, self).__init__(original_context)

    def update_segments_specially(self, segments, context):
        """
        Дополнительное обновление сегментов на фиксированную дату
        """
        fill_suburban_facilities(segments)
        fill_suburban_events(segments)
        fill_suburban_event_keys(segments)
        fill_thread_cancels(segments)
        fill_train_numbers(segments)
        segments = make_meta_trains(segments)
        update_platforms_by_dynamic(segments)

        return segments


class RaspDbAllDaysSearch(RaspDbUsualSearch):
    """
    Обычный поиск - ищет все сегменты на все дни
    """
    def __init__(self, original_context):
        super(RaspDbAllDaysSearch, self).__init__(original_context)

    def update_segments_specially(self, segments, context):
        """
        Дополнительное обновление сегментов на все дни
        """
        if self.next_plan:
            next_plan_segments = [segment for segment in segments if segment.thread.schedule_plan == self.next_plan]
            if not next_plan_segments:
                self.next_plan = None

        add_railway_sales_limit(segments, context.national_version)

        if context.group_trains:
            segments = make_grouped_trains(segments)
        else:
            for segment in segments:
                segment.run_days = calculate_run_days(segment, days_ago=settings.SEARCH_DAYS_TO_PAST)

        add_days_by_tz(segments, context.timezones, self.next_plan)

        return segments
