# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response
import requests

from travel.rasp.morda_backend.morda_backend.search.search.data_layer.all_types_search import (
    AllTypesNearestSearch, AllTypesOneDaySearch, AllTypesAllDaysSearch
)
from travel.rasp.morda_backend.morda_backend.search.search.data_layer.baris_search import (
    BarisNearestSearch, BarisOneDaySearch, BarisAllDaysSearch
)
from travel.rasp.morda_backend.morda_backend.search.search.data_layer.rasp_db_search import (
    RaspDbNearestSearch, RaspDbOneDaySearch, RaspDbAllDaysSearch
)
from travel.rasp.morda_backend.morda_backend.search.search.serialization.request_serialization import ContextQuerySchema
from travel.rasp.morda_backend.morda_backend.search.search.serialization.response_serialization import ResponseSchema
from travel.rasp.morda_backend.morda_backend.tariffs.bus.service import use_settlement_keys

log = logging.getLogger(__name__)


@api_view(['GET'])
def search(request):
    """
    Поиск маршрутов.
    /ru/search/search/?pointFrom=c213&pointTo=c2
    /ru/search/search/?pointFrom=c213&pointTo=c2&when=2017-06-06
    """
    context, errors = ContextQuerySchema(context={'request': request}).load(request.GET)
    if errors:
        return Response({'result': {}, 'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    try:
        searcher = _choose_searcher(context)
        result = searcher.search()
    except requests.HTTPError as ex:
        code = ex.response.status_code
        return Response({'result': {}, 'errors': [ex.message]}, status=code)

    dump_context = {
        'is_all_days_search': not context.when,
        'tld': context.national_version,
        't_type': context.transport_type,
        'bus_settlement_keys': use_settlement_keys(context)
    }
    data, errors = ResponseSchema(context=dump_context).dump(result)

    return Response({'result': data, 'errors': errors}, status=status.HTTP_200_OK)


def _choose_searcher(context):
    transport_type = context.transport_type
    nearest = context.nearest
    when = context.when

    if not transport_type:
        if nearest:
            return AllTypesNearestSearch(context)
        elif when:
            return AllTypesOneDaySearch(context)
        return AllTypesAllDaysSearch(context)

    elif transport_type == 'plane':
        if nearest:
            return BarisNearestSearch(context)
        elif when:
            return BarisOneDaySearch(context)
        return BarisAllDaysSearch(context)

    if nearest:
        return RaspDbNearestSearch(context)
    elif when:
        return RaspDbOneDaySearch(context)
    return RaspDbAllDaysSearch(context)
