# coding: utf-8
from __future__ import unicode_literals, absolute_import

from functools import reduce
from itertools import groupby

from django.db.models import Q

from common.apps.train.models import UFSNewOrderBlackList
from common.models.geo import StationCode
from common.models.schedule import RTStation
from common.models_abstract.schedule import ExpressType
from common.models.transport import TransportType, TransportSubtype


def fill_thread_first_last_country_code(segments, predicate=lambda s: s.thread is not None):
    """
    :param predicate: Заполняем страны, только сегментам, удовлетворяющим условию
    """
    threads = [s.thread for s in segments if predicate(s)]
    rtstations = (
        RTStation.objects
        .filter(thread__in=threads)
        .filter(Q(tz_arrival=None) | Q(tz_departure=None))
        .order_by('thread', 'id')
        .select_related('station__country')
        .values('thread_id', 'station__country__code')
    )
    country_codes = {}
    for thread_id, stops in groupby(rtstations, lambda stop: stop['thread_id']):
        stops = list(stops)
        country_codes[thread_id] = (stops[0]['station__country__code'], stops[-1]['station__country__code'])

    for thread in threads:
        if thread.id in country_codes:
            thread.first_country_code, thread.last_country_code = country_codes[thread.id]

    return segments


def fill_ufs_old_order(segments):
    black_list_numbers = reduce(
        lambda set1, set2: set1 | set2,
        (bl.numbers for bl in UFSNewOrderBlackList.objects.all()),
        set()
    )

    for s in segments:
        if s.thread and s.thread.t_type_id == TransportType.TRAIN_ID and s.thread.number in black_list_numbers:
            s.old_ufs_order = True

    return segments


def mark_train_with_subtype_as_express(segments):
    """RASPFRONT-8781 Поезда с подтипом транспорта для поиска электричками (lastdal, dnexpress..) считаем экспрессами"""
    for segment in segments:
        if segment.thread and segment.thread.t_subtype and segment.thread.t_subtype.code in TransportSubtype.get_train_search_codes():
            segment.thread.express_type = ExpressType.EXPRESS


def fill_express_codes_for_segments(segments):
    """
    Заполняет поле codes['express'] (коды в кодовой системе express) для станций сегментов и для подсегментов
    """
    station_ids = set()
    for segment in segments:
        station_ids.add(segment.station_from.id)
        station_ids.add(segment.station_to.id)

        if hasattr(segment, 'sub_segments'):
            for sub_segment in segment.sub_segments:
                station_ids.add(sub_segment.station_from.id)
                station_ids.add(sub_segment.station_to.id)

    if not station_ids:
        return

    codes_by_station_ids = {
        stid: code
        for stid, code in (
            StationCode.objects.filter(
                station__id__in=list(station_ids), system__code='express'
            ).values_list('station__id', 'code')
        )
    }

    for segment in segments:
        _fill_segment_codes(segment, codes_by_station_ids)
        if hasattr(segment, 'sub_segments'):
            for sub_segment in segment.sub_segments:
                _fill_segment_codes(sub_segment, codes_by_station_ids)


def _fill_segment_codes(segment, codes_by_station_ids):
    if segment.station_from.id in codes_by_station_ids:
        segment.station_from.codes = {'express': codes_by_station_ids[segment.station_from.id]}
    if segment.station_to.id in codes_by_station_ids:
        segment.station_to.codes = {'express': codes_by_station_ids[segment.station_to.id]}
