# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from marshmallow import Schema, fields

from common.models.schedule import RThreadType
from common.models.transport import TransportType
from common.serialization.common_schemas import MultiValueDictSchemaMixin
from common.utils.railway import get_railway_tz_by_point
from route_search.transfers.transfer_segment import BarisTransferSegment

from travel.rasp.morda_backend.morda_backend.search.parse_context.point import get_point_timezone
from travel.rasp.morda_backend.morda_backend.serialization.fields import AwareDateTime, TransportTypeField
from travel.rasp.morda_backend.morda_backend.serialization.schema_bases import PointsQuerySchema
from travel.rasp.morda_backend.morda_backend.serialization.segment import (
    RaspDBThreadSchema, SearchCompanySchema
)
from travel.rasp.morda_backend.morda_backend.serialization.segment_transport import build_transport
from travel.rasp.morda_backend.morda_backend.search.search.serialization.avia_response_serialization import (
    AviaThreadSchema, AviaTransportSchema
)


class TransfersVariantsQuerySchema(PointsQuerySchema, MultiValueDictSchemaMixin):
    transport_types = fields.List(TransportTypeField(), load_from='transportType')
    when = fields.Date()


class TransferSettlementSchema(Schema):
    id = fields.Integer()
    title = fields.Function(lambda obj: obj.L_title())
    title_genitive = fields.Function(lambda obj: obj.L_title(case='genitive', fallback=False),
                                     dump_to='titleGenitive', allow_none=False)
    title_accusative = fields.Function(lambda obj: obj.L_title(case='accusative', fallback=False),
                                       dump_to='titleAccusative')
    title_locative = fields.Function(lambda obj: obj.L_title(case='locative', fallback=False), dump_to='titleLocative')
    preposition_v_vo_na = fields.Function(lambda obj: obj.L_title(case='preposition_v_vo_na', fallback=False),
                                          dump_to='preposition')


class TransferStationSchema(Schema):
    id = fields.Integer()
    title = fields.Function(lambda obj: obj.L_title())
    title_genitive = fields.Function(lambda obj: obj.L_title(case='genitive', fallback=False), dump_to='titleGenitive')
    title_accusative = fields.Function(lambda obj: obj.L_title(case='accusative', fallback=False),
                                       dump_to='titleAccusative')
    title_locative = fields.Function(lambda obj: obj.L_title(case='locative', fallback=False), dump_to='titleLocative')
    preposition_v_vo_na = fields.Function(lambda obj: obj.L_title(case='preposition_v_vo_na', fallback=False),
                                          dump_to='preposition')
    popular_title = fields.Function(lambda obj: obj.L_popular_title(fallback=False), dump_to='popularTitle')
    timezone = fields.Function(lambda obj: get_point_timezone(obj))
    country_id = fields.Integer(dump_to='countryId')
    settlement = fields.Nested(TransferSettlementSchema)


class BaseTransferSegmentSchema(Schema):
    arrival = AwareDateTime()
    departure = AwareDateTime()
    price = fields.Float()
    convenience = fields.Integer()
    station_from = fields.Method('build_station_from', dump_to='stationFrom')
    station_to = fields.Method('build_station_to', dump_to='stationTo')
    company = fields.Nested(SearchCompanySchema)
    train_numbers = fields.List(fields.String, dump_to='trainNumbers')

    def build_station_from(self, segment):
        return self._build_station(segment.station_from, segment.t_type)

    def build_station_to(self, segment):
        return self._build_station(segment.station_to, segment.t_type)

    def _build_station(self, station, segment_t_type):
        """
        segment_t_type нужен для Аэроэкспрессов: электричка прибывает на станцию с типом транспорта "самолет".
        Поэтому нельзя использовать station.t_type.
        """
        station_data = TransferStationSchema().dump(station).data

        if segment_t_type.id in TransportType.RAILWAY_TTYPE_IDS:
            railway_timezone = get_railway_tz_by_point(station)
            if railway_timezone:
                station_data['railwayTimezone'] = railway_timezone.zone

        return station_data


class RaspDBTransferSegmentSchema(BaseTransferSegmentSchema):
    thread = fields.Nested(RaspDBThreadSchema)
    transport = fields.Function(build_transport)
    is_interval = fields.Function(lambda obj: obj.departure is None, dump_to='isInterval')
    is_through_train = fields.Function(
        lambda obj: obj.thread and obj.thread.type_id == RThreadType.THROUGH_TRAIN_ID,
        dump_to='isThroughTrain'
    )


class BarisTransferSegmentSchema(BaseTransferSegmentSchema):
    thread = fields.Nested(AviaThreadSchema)
    transport = fields.Nested(AviaTransportSchema)

    # Должны присутствовать в ответе, несмотря на то, что не имеют смысла для авиа
    is_interval = fields.Boolean(dump_to='isInterval')
    is_through_train = fields.Boolean(dump_to='isThroughTrain')


class TransfersVariantResponseSchema(Schema):
    segments = fields.Method('_make_segments')

    def _make_segments(self, response):
        segments = []
        for segment in response.segments:
            if isinstance(segment, BarisTransferSegment):
                data, _ = BarisTransferSegmentSchema(context=self.context).dump(segment)
            else:
                data, _ = RaspDBTransferSegmentSchema(context=self.context).dump(segment)
            segments.append(data)
        return segments


class TransfersVariantsResponseSchema(Schema):
    transfers_variants = fields.Nested(TransfersVariantResponseSchema, many=True, dump_to='transferVariants')
