# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from marshmallow import Schema, fields, post_dump, missing

from common.models.schedule import RThreadType
from common.models.transport import TransportType
from common.serialization.fields import DictNestedField
from common.utils.railway import get_railway_tz_by_point
from route_search.models import BaseThreadSegment

from travel.rasp.morda_backend.morda_backend.search.search.data_layer.backend import get_avia_company_url
from travel.rasp.morda_backend.morda_backend.serialization.fields import AwareDateTime, FlagField
from travel.rasp.morda_backend.morda_backend.serialization.segment_station import SegmentStationSchema
from travel.rasp.morda_backend.morda_backend.serialization.segment_suburban_facilities import SuburbanFacilitySchema
from travel.rasp.morda_backend.morda_backend.serialization.segment_transport import DeluxeTrainSchema, build_transport
from travel.rasp.morda_backend.morda_backend.tariffs.bus.service import make_ybus_tariff_keys, KEY_TEMPLATES_ALL_DAYS
from travel.rasp.morda_backend.morda_backend.tariffs.daemon.service import make_daemon_key, make_min_price_daemon_key
from travel.rasp.morda_backend.morda_backend.tariffs.static.service import (
    make_static_segment_keys, make_min_static_segment_keys
)
from travel.rasp.morda_backend.morda_backend.tariffs.suburban.service import make_suburban_segment_keys
from travel.rasp.morda_backend.morda_backend.tariffs.train.base.utils import make_segment_train_keys


INVALID_VALUE = 'invalid_value'


class BaseSearchSegmentDaysSchema(Schema):
    days_text = fields.String(dump_to='text')


class SearchSegmentDaysSchema(BaseSearchSegmentDaysSchema):
    days_text_short = fields.String(dump_to='shortText')
    schedule_plan_appendix = fields.String(dump_to='schedulePlanAppendix')
    except_days_text = fields.String(dump_to='exceptText')
    has_more_days = fields.Boolean(dump_to='hasMoreDays')
    cancel = fields.Boolean(dump_to='canceled')


class SearchCompanySchema(Schema):
    id = fields.Integer()
    title = fields.Function(lambda obj: obj.L_title())
    short_title = fields.Function(lambda obj: obj.L_short_title(), dump_to='shortTitle')
    url = fields.String()
    hidden = fields.Boolean()
    yandex_avia_url = fields.Method('make_yandex_avia_url', dump_to='yandexAviaUrl')

    def make_yandex_avia_url(self, company):
        if company.yandex_avia_code:
            return get_avia_company_url(company.yandex_avia_code, self.context.get('tld', None))


class TrainSchedulePlanSchema(Schema):
    code = fields.String()
    title = fields.String()
    start_date = fields.Date(dump_to='startDate')
    end_date = fields.Date(dump_to='endDate')


class BaseThreadSchema(Schema):
    uid = fields.String()
    number = fields.String()

    is_express = fields.Boolean(dump_to='isExpress', default=False)
    is_aeroexpress = fields.Boolean(dump_to='isAeroExpress', default=False)
    begin_time = fields.Time(dump_to='beginTime')
    end_time = fields.Time(dump_to='endTime')
    density = fields.String()
    comment = fields.String()


class RaspDBThreadSchema(BaseThreadSchema):
    title = fields.Function(lambda obj: obj.L_title())
    deluxe_train = fields.Nested(DeluxeTrainSchema, dump_to='deluxeTrain')
    is_basic = fields.Function(lambda thread: thread.type_id == RThreadType.BASIC_ID, dump_to='isBasic')

    # На фронте не верно трактуют
    schedule_plan = fields.Nested(TrainSchedulePlanSchema, only='code', dump_to='schedulePlanCode')

    first_country_code = fields.String(dump_to='firstCountryCode')
    last_country_code = fields.String(dump_to='lastCountryCode')

    displace_yabus = fields.Function(
        lambda t: (t.supplier and t.supplier.displace_yabus) if t.t_type_id == TransportType.BUS_ID else None,
        dump_to='displaceYabus'
    )

    canonical_uid = fields.String(dump_to='canonicalUid')

    fully_cancelled = fields.Bool(dump_to='cancelled')
    cancelled_segments = fields.Method('_make_cancelled_segments', dump_to='cancelledSegments')

    def _make_cancelled_segments(self, thread):
        def get_cancelled_title(station):
            return station.L_popular_title(case='genitive', fallback=True) or station.L_title()

        cancelled_segments = []
        for cancelled_segment in thread.cancelled_segments:
            cancelled_segments.append({
                'fromTitleGenitive': get_cancelled_title(cancelled_segment['station_from']),
                'toTitleGenitive': get_cancelled_title(cancelled_segment['station_to'])
            })
        return cancelled_segments

    @post_dump
    def post_dump(self, data):
        if data['deluxeTrain'] is None:
            data.pop('deluxeTrain')
        return data


class EventStateSchema(Schema):
    type = fields.String()
    minutes_from = fields.Integer(dump_to='minutesFromNew')
    minutes_to = fields.Integer(dump_to='minutesToNew')
    minutes_from_old = fields.Function(lambda obj: obj.minutes_from or 0, dump_to='minutesFrom')
    minutes_to_old = fields.Function(lambda obj: obj.minutes_to or obj.minutes_from or 0, dump_to='minutesTo')


class BaseSegmentSchema(Schema):
    arrival = AwareDateTime()
    departure = AwareDateTime()
    arrival_local_dt = fields.LocalDateTime(attribute='arrival', dump_to='arrivalLocalDt')
    departure_local_dt = fields.LocalDateTime(attribute='departure', dump_to='departureLocalDt')
    duration = fields.TimeDelta()
    number = fields.String()

    run_days = fields.Dict(dump_to='runDays')  # {'2016': {'1': [1, 1, 0, 1,]}}
    start_date = fields.Date(dump_to='startDate')

    company = fields.Nested(SearchCompanySchema)
    suburban_facilities = fields.List(fields.Nested(SuburbanFacilitySchema), dump_to='suburbanFacilities')

    arrival_event = fields.Nested(EventStateSchema, dump_to='arrivalEvent')
    departure_event = fields.Nested(EventStateSchema, dump_to='departureEvent')
    arrival_event_key = fields.String(dump_to='arrivalEventKey')
    departure_event_key = fields.String(dump_to='departureEventKey')

    train_purchase_numbers = fields.List(fields.String(), many=True, dump_to='trainPurchaseNumbers')


class RaspDBSegmentSchema(BaseSegmentSchema):
    title = fields.Function(lambda obj: obj.L_title())
    station_from = fields.Method('build_station_from', dump_to='stationFrom')
    station_to = fields.Method('build_station_to', dump_to='stationTo')

    thread = fields.Nested(RaspDBThreadSchema)

    transport = fields.Method('build_transport')

    is_interval = fields.Function(lambda obj: obj.departure is None, dump_to='isInterval')
    is_through_train = fields.Function(
        lambda obj: obj.thread and obj.thread.type_id == RThreadType.THROUGH_TRAIN_ID, dump_to='isThroughTrain'
    )
    is_fuzzy_from = FlagField(dump_to='isFuzzyFrom')
    is_fuzzy_to = FlagField(dump_to='isFuzzyTo')
    stops = fields.Function(lambda obj: obj.L_stops() if isinstance(obj, BaseThreadSegment) else missing)
    days_by_tz = DictNestedField(SearchSegmentDaysSchema, dump_to='daysByTimezone')

    sales_limit_in_days = fields.Integer(dump_to='salesLimitInDays')
    do_not_sell = FlagField(dump_to='doNotSell')
    old_ufs_order = fields.Boolean(dump_to='oldUfsOrder')

    arrival_event = fields.Nested(EventStateSchema, dump_to='arrivalEvent')
    departure_event = fields.Nested(EventStateSchema, dump_to='departureEvent')

    arrival_event_key = fields.String(dump_to='arrivalEventKey')
    departure_event_key = fields.String(dump_to='departureEventKey')

    interval_thread_departure_from_date = fields.Date(dump_to='intervalThreadDepartureFromDate')

    url = fields.String(dump_to='url')

    def build_transport(self, segment):
        return build_transport(segment, self.context.get('t_type'))

    def build_station_from(self, segment):
        return self._build_station(segment.station_from, getattr(segment, 'rtstation_from', None), segment.t_type)

    def build_station_to(self, segment):
        return self._build_station(segment.station_to, getattr(segment, 'rtstation_to', None), segment.t_type)

    def _build_station(self, station, rtstation, segment_t_type):
        station_data = SegmentStationSchema(context=self.context).dump(station).data
        if rtstation:
            station_data['platform'] = rtstation.L_platform()

        if segment_t_type.id in TransportType.RAILWAY_TTYPE_IDS:
            railway_timezone = get_railway_tz_by_point(station)
            if railway_timezone:
                station_data['railwayTimezone'] = railway_timezone.zone
        return station_data


class RaspDBSearchSegmentSchema(RaspDBSegmentSchema):
    tariffs_keys = fields.Method('build_tariffs_keys', dump_to='tariffsKeys')
    has_train_tariffs = fields.Function(
        lambda s: (
            len(getattr(s, 'train_purchase_numbers', [])) > 0 if s.t_type.id == TransportType.SUBURBAN_ID else None
        ),
        dump_to='hasTrainTariffs'
    )

    def get_attribute(self, key, data, default):
        if key == 'tariffs_keys':
            return self.build_tariffs_keys(data)

        return super(RaspDBSearchSegmentSchema, self).get_attribute(key, data, default)

    def build_tariffs_keys(self, segment):
        if self.context.get('is_all_days_search'):
            return sum([
                make_ybus_tariff_keys(segment, KEY_TEMPLATES_ALL_DAYS, bus_settlement_keys=self.context.get('bus_settlement_keys')),
                make_suburban_segment_keys(segment),
                make_min_static_segment_keys(segment),
                [make_min_price_daemon_key(segment)]
            ], [])

        return sum([
            make_ybus_tariff_keys(segment, bus_settlement_keys=self.context.get('bus_settlement_keys')),
            [make_daemon_key(segment)],
            make_suburban_segment_keys(segment),
            make_static_segment_keys(segment),
            make_segment_train_keys(segment)
        ], [])


class RaspDBSearchMainSegmentSchema(RaspDBSearchSegmentSchema):
    sub_segments = fields.Nested(RaspDBSearchSegmentSchema, many=True, dump_to='subSegments')
    min_arrival = AwareDateTime(dump_to='minArrival')
    max_arrival = AwareDateTime(dump_to='maxArrival')
    companies = fields.List(fields.Nested(SearchCompanySchema))
