# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.utils.translation import get_language

from common.data_api.baris.instance import baris
from common.models.geo import StationMajority, Station
from common.models.transport import TransportType
from common.models_utils import fetch_related
from common.utils.settlement import get_connected_stations, get_related_stations

from travel.rasp.morda_backend.morda_backend.settlement.serialization import station as station_serialization


log = logging.getLogger(__name__)


def get_stations_by_settlement(settlement, t_type_id):
    """
    Возвращает списки станций города
    :param settlement: город
    :param t_type_id: id типа транспорта или None, если нужно вернуть станции для всех типов
    """
    language = get_language()

    # Главная в городе или попадает в табло
    connected_stations = get_connected_stations(settlement, StationMajority.IN_TABLO_ID, t_type_id)
    related_stations = get_related_stations(settlement, StationMajority.IN_TABLO_ID, t_type_id)
    all_stations = connected_stations + related_stations

    plane_stations_ids = [station.id for station in all_stations if station.t_type_id == TransportType.PLANE_ID]
    stats = _get_stations_delayed_flights(plane_stations_ids)

    fetch_related(all_stations, 'settlement', model=Station)

    return {
        'connected': [
            station_serialization.station_json(st, _get_station_stat(stats, st), language) for st in connected_stations
        ],
        'related': [
            station_serialization.station_json(st, _get_station_stat(stats, st), language) for st in related_stations
        ],
    }


def _get_stations_delayed_flights(plane_stations_ids):
    if plane_stations_ids:
        try:
            return baris.get_delayed_flights(plane_stations_ids)
        except Exception as ex:
            log.exception(ex.message)
    return {}


def _get_station_stat(stats, station):
    """
    Получение статистики по задежанным и отмененным рейсам
    """
    if station.id in stats and station.tablo_state == 'real':
        stat = stats[station.id]
        return {'cancelled': stat['cancelled'], 'late': stat['delayed']}
    return {'cancelled': 0, 'late': 0}
