# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import itertools
import operator

from common.data_api.search_stats.search_stats import search_stats
from common.models.geo import Country, Settlement
from common.models.transport import TransportType
from route_search.models import ZNodeRoute2
from travel.rasp.morda_backend.morda_backend.settlement.data_layer.settlement import get_visible_region_capital


DEFAULT_REGION_CITIES = {
    11004: 35,  # Адыгея - Краснодар
    977: 35,  # Крым - Краснодар
    10231: 975,  # Горный Алтай - Бийск
    11012: 1092,  # Ингушетия - Назрань
    11015: 38,  # Калмыкия - Волгоград
    11020: 11064,  # Карачаево-Черкесия - Невинномыск
    10233: 1095,  # Тыва - Абакан
    11443: 11437,  # Якутия - Нерюнгри
    11193: 973,  # ХМАО - Сургут
    11232: 11230  # ЯНАО - Новый Уренгой
}
KALININGRAD_REGION_ID = 10857  # Калининградская обл. - запрещена продажа электронных билетов а материковую Россию
KRYM_REGION_ID = 977  # Крым - не продаем билеты ни оттуда, ни туда
MAX_DIRECTIONS = 4


def check_city_has_trains(city):
    return ZNodeRoute2.objects.filter(t_type_id=TransportType.TRAIN_ID, settlement_from=city).exists()


def check_direction(departure_city, arrival_city):
    if arrival_city.region_id == KRYM_REGION_ID:
        return False

    if (
        departure_city.region_id == KALININGRAD_REGION_ID and
        arrival_city.country_id == Country.RUSSIA_ID and arrival_city.region_id != KALININGRAD_REGION_ID
    ) or (
        arrival_city.region_id == KALININGRAD_REGION_ID and
        departure_city.country_id == Country.RUSSIA_ID and departure_city.region_id != KALININGRAD_REGION_ID
    ):
        return False

    return ZNodeRoute2.objects.filter(t_type_id=TransportType.TRAIN_ID,
                                      settlement_from_id=departure_city.id,
                                      settlement_to_id=arrival_city.id).exists()


def iter_arrival_cities(departure_city):
    arrival_city_keys = map(operator.itemgetter(0),
                            search_stats.get_top_to(departure_city.point_key, 'train', search_type='c', limit=20))
    arrival_cities = Settlement.in_bulk(arrival_city_keys)
    for arrival_city_key in arrival_city_keys:
        arrival_city = arrival_cities[arrival_city_key]
        if check_direction(departure_city, arrival_city):
            yield arrival_city


def get_departure_city(client_city):
    if client_city.country_id == Country.RUSSIA_ID:
        if client_city.region_id != KRYM_REGION_ID and check_city_has_trains(client_city):
            return client_city

        default_city = (Settlement.objects.get(id=DEFAULT_REGION_CITIES[client_city.region_id])
                        if client_city.region_id in DEFAULT_REGION_CITIES else
                        get_visible_region_capital(client_city))
        if check_city_has_trains(default_city):
            return default_city

    return Settlement.objects.get(id=Settlement.MOSCOW_ID)


def get_train_popular_directions(client_city):
    departure_city = get_departure_city(client_city)
    return {
        'departure_city': departure_city,
        'arrival_cities': list(itertools.islice(iter_arrival_cities(departure_city), MAX_DIRECTIONS))
    }
