# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.shortcuts import get_object_or_404
from django.utils.translation import get_language
from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response

from common.models.geo import ExternalDirection, Settlement, Point
from common.views import teasers as teasers_view

from travel.rasp.morda_backend.morda_backend.settlement.data_layer import (
    settlement as settlement_data_layer,
    search_sample_points as search_sample_points_data_layer
)
from travel.rasp.morda_backend.morda_backend.settlement.data_layer.popular_directions import (
    get_settlement_transport_popular_directions, get_settlement_popular_directions
)
from travel.rasp.morda_backend.morda_backend.settlement.data_layer.stations_by_settlement import (
    get_stations_by_settlement
)
from travel.rasp.morda_backend.morda_backend.settlement.data_layer.train_popular_directions import (
    get_train_popular_directions
)
from travel.rasp.morda_backend.morda_backend.settlement.data_layer import request as request_helper
from travel.rasp.morda_backend.morda_backend.data_layer.decorators import json_response
from travel.rasp.morda_backend.morda_backend.serialization import teasers as teaser_serialization
from travel.rasp.morda_backend.morda_backend.settlement.serialization import (
    settlement as settlement_serialization,
    search_sample_point as search_sample_point_serialization,
)
from travel.rasp.morda_backend.morda_backend.settlement.serialization.popular_directions import (
    AllTransportsPopularDirectionsSchema, TransportPopularDirectionsRequestSchema, TransportPopularDirectionsSchema
)
from travel.rasp.morda_backend.morda_backend.settlement.serialization.train_popular_directions import (
    TrainPopularDirectionsSchema
)


@api_view(['GET'])
def stations_by_settlement_id(request, settlement_id):
    try:
        settlement = Settlement.objects.get(id=settlement_id)
    except Settlement.DoesNotExist:
        return Response({'errors': 'Settlement not found'}, status=status.HTTP_404_NOT_FOUND)

    return _get_stations_by_settlement_view(request, settlement)


@api_view(['GET'])
def stations_by_settlement_slug(request, settlement_slug):
    try:
        settlement = Settlement.objects.get(slug=settlement_slug)
    except Settlement.DoesNotExist:
        return Response({'errors': 'Settlement not found'}, status=status.HTTP_404_NOT_FOUND)

    return _get_stations_by_settlement_view(request, settlement)


def _get_stations_by_settlement_view(request, settlement):
    context, errors = settlement_serialization.SettlementQuerySchema().load(request.GET)
    if errors:
        return Response({'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    t_type_id = context.transport_type.id if context.transport_type else None
    return Response(get_stations_by_settlement(settlement, t_type_id))


@json_response
def directions(request, settlement_id):
    settlement = get_object_or_404(Settlement, id=settlement_id)
    language = get_language()

    if settlement.suburban_zone:
        directions = list(settlement.suburban_zone.externaldirection_set.all())

        connected_directions_ids = set(ExternalDirection.objects.filter(
            externaldirectionmarker__station__settlement=settlement
        ).values_list('id', flat=True))

        return [
            {
                'code': direction.code,
                'title': direction.L_title(lang=language),
                'connected': direction.id in connected_directions_ids
            } for direction in directions
        ]

    return []


@json_response
def get_settlement(request):
    settlement_id = None
    geo_ids = request_helper.get_int_list_param_value(request, 'geo_ids')
    root_domain = request.GET.get('root_domain')
    national_version = request_helper.get_national_version(request)

    settlement = None

    if 'id' in request.GET:
        settlement_id = request_helper.get_int_param_value(request, 'id')
        if settlement_id is None:
            return None
        settlement = settlement_data_layer.get_visible_settlement_by_id(settlement_id)
        if not settlement:
            return None
    elif 'slug' in request.GET:
        settlement = settlement_data_layer.get_visible_settlement_by_slug(request.GET.get('slug'))
        if not settlement:
            return None

    if not settlement and geo_ids:
        settlement = settlement_data_layer.get_visible_nearest_settlement_by_geo_ids(geo_ids, national_version)

    if root_domain:
        if settlement:
            # Переопределение города по нац-домену (не-ru)
            if settlement_id is None and root_domain != 'ru':
                country_domain = settlement.country and settlement.country.domain_zone

                if not country_domain or country_domain != root_domain:
                    settlement_for_root_domain = Settlement.get_default_for_root_domain(root_domain)

                    if settlement_for_root_domain:
                        settlement = settlement_for_root_domain
        else:
            settlement = Settlement.get_default_for_root_domain(root_domain)

    if not settlement:
        settlement = settlement_data_layer.get_moscow()

    if settlement:
        json = settlement_serialization.settlement_json(settlement, national_version, get_language())
        return json

    return None


@json_response
def teasers(request, settlement_id):
    settlement = get_object_or_404(Settlement, id=settlement_id)
    national_version = request_helper.get_national_version(request)
    language = get_language()

    teaser_dict = teasers_view.fetch_rasp_teasers([u'index', u'info_settlement'], settlement, False, national_version,
                                                  language)

    for teaser_type in teaser_dict.keys():
        teasers = teaser_dict[teaser_type]
        teaser_dict[teaser_type] = [teaser_serialization.teaser_json(teaser) for teaser in teasers]

    return teaser_dict


@json_response
def search_sample_points(request, settlement_id):
    """
    Получает примеры поисковых контекстов ПукнтА - ПунктБ.
    Пример ответа:
    {
        'point_from': {'key': 'c213', 'title': 'Москва'},
        'point_to': {'key': 'c2', 'title': 'Санкт-Петербург'},
    }
    """
    national_version = request_helper.get_national_version(request)
    language = get_language()

    point_from_key, point_to_key = search_sample_points_data_layer.get_search_sample_points_keys(int(settlement_id),
                                                                                                 national_version)

    point_from = Point.get_by_key(point_from_key)
    point_to = Point.get_by_key(point_to_key)

    return {
        'point_from': search_sample_point_serialization.search_sample_point_json(point_from, language),
        'point_to': search_sample_point_serialization.search_sample_point_json(point_to, language)
    }


@api_view(['GET'])
def train_popular_directions(request, settlement_id):
    """
    Получает популярные направления поездов.
    """
    try:
        client_city = Settlement.objects.get(id=settlement_id)
    except Settlement.DoesNotExist:
        return Response({'errors': 'Settlement not found'}, status=status.HTTP_404_NOT_FOUND)

    train_directions = get_train_popular_directions(client_city)
    return Response(TrainPopularDirectionsSchema().dump(train_directions).data)


@api_view(['GET'])
def popular_directions(request, settlement_id):
    """
    Получает популярные направления на основе статистики поисковых запросов.
    """
    try:
        settlement = Settlement.objects.get(id=settlement_id)
    except Settlement.DoesNotExist:
        return Response({'errors': 'Settlement not found'}, status=status.HTTP_404_NOT_FOUND)

    result = get_settlement_popular_directions(settlement)
    return Response(AllTransportsPopularDirectionsSchema().dump(result).data)


@api_view(['GET'])
def transport_popular_directions(request, slug):
    """
    Получает популярные направления для конкретного вида транспорта на основе статистики поисковых запросов.
    """
    context, errors = TransportPopularDirectionsRequestSchema().load(request.GET)
    try:
        settlement = Settlement.objects.get(slug=slug)
    except Settlement.DoesNotExist:
        return Response({'errors': 'Settlement not found'}, status=status.HTTP_404_NOT_FOUND)

    if context['t_type'] not in {'train', 'suburban', 'plane', 'bus'}:
        return Response({'errors': 'Invalid transport type'}, status=status.HTTP_404_NOT_FOUND)

    result = get_settlement_transport_popular_directions(settlement, context['t_type'], context['limit'])

    return Response(TransportPopularDirectionsSchema().dump(result).data)
