# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.utils.translation import get_language

from common.models.geo import StationType, WayToAirport
from common.views.teasers import TeaserSetMobile, TeaserSetRasp


class BaseStationForPage(object):
    """
    Базовый класс для страниц станции разных типов
    """
    def __init__(self, station_page_type, country='ru'):
        self.station_page_type = station_page_type
        self.station = station = station_page_type.station
        self.page_type = station_page_type
        self.language = get_language()
        self.country = country

        self.station_properties = {
            'id': station.id,
            'title': station.L_popular_title(),
            'title_genitive': station.L_popular_title(case='genitive', fallback=False),
            'full_title': self._get_full_title(),
            'full_title_genitive': self._get_full_title('genitive'),
            'full_title_dative': self._get_full_title('dative'),
            'has_popular_title': self._get_has_popular_title(),
            'address': station.L_address(),
            'subway': 'м. ' + station.near_metro if station.near_metro else None,
            'longitude': station.longitude,
            'latitude': station.latitude,
            'way_to_airport': self.get_way_to_airport(station),
            'station_type': self.station.station_type.L_title(),
            'phones': [phone.phone for phone in station.stationphone_set.all()]
        }
        if station.settlement:
            self.station_properties['settlement'] = {
                'id': station.settlement.id,
                'slug': station.settlement.slug,
                'title': station.settlement.L_title(),
                'title_genitive': station.settlement.L_title(case='genitive', fallback=False) or None
            }

        self.page_context = None
        self.threads = []
        self.schedule_routes = []
        self.teasers = {}
        self.companies_by_ids = {}
        self.companies = []

    def _get_has_popular_title(self):
        if self.station.L_popular_title(fallback=False):
            return True
        return False

    def _get_full_title(self, case='nominative'):
        """
        Формирование полного названия станции вместе с типом станции в разных падежах
        """
        station_type = self.station.station_type
        popular_title = self.station.L_popular_title(fallback=False)
        if popular_title:  # например, Московский вокзал
            if case == 'nominative':
                return popular_title

            if case == 'genitive':
                full_title = self.station.L_popular_title(fallback=False, case='genitive')
                if full_title:
                    return full_title

            station_type = StationType.objects.get(id=StationType.STATION_ID)
            return self._format_title(station_type, popular_title, case)

        # иначе, например, станция Шарташ
        return self._format_title(station_type, self.station.L_title(), case)

    def _format_title(self, station_type, station_title, case):
        station_type_name = station_type.L_name(case=case).lower()

        if station_type_name not in station_title.lower():
            if case == 'nominative' or station_type.L_name('nominative').lower() not in station_title.lower():
                return '{0} {1}'.format(station_type_name, station_title)

        return station_title

    def get_way_to_airport(self, station):
        ways = list(WayToAirport.objects.filter(station_id=station.id))
        if len(ways) == 0:
            return None

        way = ways[0]
        from_station = way.from_station
        to_station = way.to_station

        return {
            'way_type': way.way_type,
            'link_title': way.L_title(),
            'from_point_key': from_station.point_key,
            'from_point_slug': from_station.slug,
            'to_point_key': to_station.point_key,
            'to_point_slug': to_station.slug,
        }

    def make_companies(self):
        self.companies = [StationCompany(company) for company in self.companies_by_ids.values()]

    def make_teasers(self, is_mobile):
        """
        Получение тизеров для станции
        """
        teaser_set_cls = TeaserSetMobile if is_mobile else TeaserSetRasp
        selected = teaser_set_cls(
            request=None, page_code='tablo',
            data=(self.station, self.schedule_routes),
            language=self.language
        ).selected
        self.teasers = {
            teaser_type: teaser
            for teaser_type, teaser in selected.items()
        }

    def threads_smart_sort(self):
        """
        Сортировка ниток по времени или по дата+время
        """
        if self.page_context.is_all_days:
            self.threads.sort(key=lambda r: r.event_date_and_time['time'])
        else:
            self.threads.sort(key=lambda r: r.event_date_and_time['datetime'])


class StationCompany(object):
    def __init__(self, company):
        self.id = company.id
        self.title = company.L_title()
        self.short_title = company.L_short_title()
        self.url = company.url
        self.hidden = company.hidden
        self.icon = company.svg_logo
