# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function


class BaseStationThread(object):
    """
    Базовый класс ниток станции для разных типов транспорта
    """
    def __init__(self, st_for_page):
        self.st_for_page = st_for_page
        self.event = st_for_page.page_context.event
        self.is_all_days = st_for_page.page_context.is_all_days

    def make_event_date_and_time(self, dt_or_time):
        """
        Информация о времени прибытия или отправления
        """
        self.event_date_and_time = {'time': dt_or_time.strftime('%H:%M')}
        if not self.is_all_days:
            self.event_date_and_time['datetime'] = dt_or_time


class StationRaspDBThread(BaseStationThread):
    """
    Базовый класс ниток станции, которые берутся из MySQL, то есть все, кроме аэропортов
    """
    def __init__(self, schedule_route, st_for_page, next_plan=None):
        super(StationRaspDBThread, self).__init__(st_for_page)

        self.thread = schedule_route.thread
        self.schedule_route = schedule_route

        self._make_thread_link()
        if hasattr(schedule_route, 'event_dt'):
            self.make_event_date_and_time(schedule_route.event_dt)

        self.t_type = schedule_route.t_type.code
        self.title = schedule_route.L_title()
        self.number = schedule_route.number
        self.comment = self.thread.comment

        if schedule_route.company:
            self.company_id = schedule_route.company.id
            self.st_for_page.companies_by_ids[self.company_id] = schedule_route.company

        self._make_t_subtype()
        self._make_days_text(next_plan)

    def _make_thread_link(self):
        """
        Данные для формирования ссылки на нитку
        """
        self.canonical_uid = self.thread.canonical_uid
        if self.event == 'arrival':
            self.departure = self.schedule_route.start_date
        else:
            self.departure_from = self.schedule_route.event_dt

    def _make_t_subtype(self):
        """
        Информация про transport_subtype
        """
        if self.thread.t_subtype:
            self.transport_subtype = {
                'code': self.thread.t_subtype.code,
                'title': self.thread.t_subtype.L_title_without_html_symbols()
            }

    def get_shift(self):
        return self.schedule_route.rtstation.calc_days_shift(
            event=self.event,
            start_date=self.schedule_route.naive_start_dt
        )

    def get_thread_start_date(self):
        return self.schedule_route.naive_start_dt.date()

    def _make_days_text(self, next_plan):
        """
        Формирование строки с текстом дней хождения
        """
        if self.is_all_days:
            shift = self.get_shift()
            thread_start_date = self.get_thread_start_date()
            self.days_text = self.thread.L_days_text(
                shift,
                except_separator=', ',
                html=False,
                template_only=False,
                show_days=True,
                show_all_days=False,
                thread_start_date=thread_start_date,
                next_plan=next_plan
            )
            days_data = self.thread.L_days_text_dict(
                shift,
                thread_start_date=thread_start_date,
                show_days=True,
                show_all_days=False,
                next_plan=next_plan
            )
            if 'days_text' in days_data:
                self.run_days_text = days_data['days_text']
            if 'except_days_text' in days_data:
                self.except_days_text = days_data['except_days_text']
