# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import defaultdict
from functools import partial
from locale import strcoll

from django.db.models import Q

from common.models.geo import Settlement, Station, StationMajority
from travel.rasp.library.python.common23.date import environment
from travel.rasp.library.python.common23.logging import log_run_time
from common.utils.text import NBSP
from common.utils.title_generator import DASH
from common.xgettext.common import get_datetemplate_translation
from route_search.models import ZNodeRoute2
from stationschedule.type.base import BaseIntervalSchedule, IntervalScheduleRoute
from stationschedule.views import get_schedule_class

from travel.rasp.morda_backend.morda_backend.station.data_layer.base_station import BaseStationForPage


log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log, log_level=logging.DEBUG)


class BusStationForPage(BaseStationForPage):
    """
    Страница станции автобусов или кораблей
    """
    def __init__(self, station_page_type, country):
        super(BusStationForPage, self).__init__(station_page_type, country)
        self.t_type_code = station_page_type.t_type_code
        self.schedule_blocks = []
        self.routes = []

    def load_threads(self, page_context):
        """
        Формирование списка ниток станции
        """
        self.page_context = page_context

        schedule_cls = get_schedule_class(self.station, t_type_code=self.t_type_code)
        schedule = schedule_cls(self.station, t_type_code=self.t_type_code, all_days_next_days=60, limit=100000)
        interval_schedule = BaseIntervalSchedule(self.station, all_days_next_days=60)

        local_date = page_context.date if hasattr(page_context, 'date') else None
        schedule.build(schedule_date=local_date)
        interval_schedule.build(schedule_date=local_date)

        self.routes.extend(schedule.schedule_routes)
        self.routes.extend(interval_schedule.schedule_routes)

    def threads_smart_sort(self):
        blocks = defaultdict(list)
        for route in self.routes:
            key = (route.number, route.title)
            blocks[key].append(route)

        for routes in blocks.values():
            if self.page_context.is_all_days:
                schedule_groups = defaultdict(list)

                for route in routes:
                    schedule_groups[str(type(route)) + route.L_days_text()].append(route)

                schedule = []
                for grouped_routes in schedule_groups.values():
                    schedule_item = self.create_scedule(grouped_routes)
                    schedule_item.make_days_text()
                    schedule.append(schedule_item)

                schedule.sort(key=self.schedule_sort_key_func)
            else:
                schedule = [self.create_scedule(routes)]

            block = BusStationScheduleBlock(schedule, self)
            self.schedule_blocks.append(block)

        self.schedule_blocks.sort(key=self.schedule_block_sort_key_func, cmp=strcoll)

    def create_scedule(self, routes):
        if isinstance(routes[0], IntervalScheduleRoute):
            return BusStationIntervalSchedule(routes)
        else:
            return BusStationSchedule(routes)

    def schedule_block_sort_key_func(self, schedule_block):
        return (('0' * 10) + (schedule_block.number or 'z' * 10))[-10:] + schedule_block.title

    def schedule_sort_key_func(self, schedule):
        everyday_text = get_datetemplate_translation(self.language, 'AllDateTemplate')
        days_text = schedule.days_text

        if days_text == everyday_text:
            return 1
        elif days_text.startswith(everyday_text):
            return 2
        else:
            return 3

    def make_stops(self):
        threads = [r.thread for r in self.routes]
        query = Q(station_from_id=self.station.id, thread_id__in=[t.id for t in threads])
        z_node_routes = ZNodeRoute2.objects.filter(query).values_list('thread_id', 'station_to')

        paths = defaultdict(list)
        station_ids = set()
        for znr in z_node_routes:
            thread_id, station_id = znr
            paths[thread_id].append(station_id)
            station_ids.add(station_id)

        stations = {s.id: s for s in Station.objects.filter(Q(id__in=station_ids,
                                                                   hidden=False,
                                                                   majority__id__lte=StationMajority.STATION_ID))}

        settlement_ids = {s.settlement_id for s in stations.values()}
        settlements = {s.id: s for s in Settlement.objects.filter(Q(id__in=settlement_ids))}
        stops = dict()
        for thread in threads:
            path = paths[thread.id]
            for station_id in path:
                if station_id not in stations:
                    continue

                if station_id not in stops:
                    stops[station_id] = BusStationScheduleStop(stations[station_id], settlements)

                stops[station_id].threads.add(thread.canonical_uid)

        self.stops = stops.values()


class BusStationScheduleBlock(object):
    """
    Расписание маршрута автобусной станции
    """
    def __init__(self, schedule, st_for_page):
        schedule_route = schedule[0].schedule_route
        thread = schedule_route.thread

        self.schedule = schedule
        self.number = thread.number
        self.comment = thread.comment
        self.t_type = schedule_route.t_type.code

        if schedule_route.company:
            self.company_id = schedule_route.company.id
            st_for_page.companies_by_ids[self.company_id] = schedule_route.company

        self.title = self._build_schedule_block_title(thread, schedule_route)

    def _build_schedule_block_title(self, thread, schedule_route):
        title_separator = '{}{} '.format(NBSP, DASH)

        title = (
            thread.title
            .replace(' - ', title_separator)
            .replace(' {} '.format(DASH), title_separator)
            .replace('{}- '.format(NBSP), title_separator)
        )

        if schedule_route.rtstation.tz_arrival is None:
            stations_titles = map(unicode.strip, title.split(title_separator))
            if len(stations_titles) == 2:
                _, station_to_title = stations_titles
                return station_to_title

        return title


class BusStationSchedule(object):
    """
    Расписание маршрута автобусной станции
    """
    def __init__(self, schedule_routes):
        self.schedule_route = schedule_routes[0]
        self.thread = self.schedule_route.thread
        self.id = self.thread.id
        self.threads = [BusStationThread(r) for r in sorted(schedule_routes, key=self.sort_key_func)]

    def sort_key_func(self, route):
        if isinstance(route, IntervalScheduleRoute):
            return route.begin_time
        else:
            return route.event_dt.time()

    def get_shift(self):
        return self.schedule_route.rtstation.calc_days_shift(
            event='departure',
            start_date=self.schedule_route.naive_start_dt
        )

    def get_thread_start_date(self):
        return self.schedule_route.naive_start_dt.date()

    def make_days_text(self):
        """
        Формирование строки с текстом дней хождения
        """
        shift = self.get_shift()
        thread_start_date = self.get_thread_start_date()
        self.days_text = self.thread.L_days_text(
            shift,
            except_separator=', ',
            html=False,
            template_only=False,
            show_days=True,
            show_all_days=False,
            thread_start_date=thread_start_date
        )
        days_data = self.thread.L_days_text_dict(
            shift,
            thread_start_date=thread_start_date,
            show_days=True,
            show_all_days=False
        )
        if 'days_text' in days_data:
            self.run_days_text = days_data['days_text']
        if 'except_days_text' in days_data:
            self.except_days_text = days_data['except_days_text']


class BusStationIntervalSchedule(BusStationSchedule):
    def __init__(self, schedule_routes):
        super(BusStationIntervalSchedule, self).__init__(schedule_routes)

    def get_thread_start_date(self):
        return environment.now().date()

    def get_shift(self):
        return self.schedule_route.mask_shift


class BusStationThread(object):
    """
    Нитка автобуса для страницы станции
    """
    def __init__(self, schedule_route):
        self.canonical_uid = schedule_route.thread.canonical_uid

        if isinstance(schedule_route, IntervalScheduleRoute):
            self.interval = {
                'begin_time': schedule_route.thread.begin_time.strftime('%H:%M'),
                'end_time': schedule_route.thread.end_time.strftime('%H:%M'),
                'density': schedule_route.thread.density
            }
        else:
            self.departure_from = schedule_route.event_dt


class BusStationScheduleStop(object):
    def __init__(self, station, settlements):
        self.id = station.id
        self.title = station.L_popular_title()
        self.majority = station.majority_id
        self.threads = set()

        if station.settlement_id in settlements:
            self.settlement = settlements[station.settlement_id].get_popular_title()
