# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import timedelta, datetime, time

from common.settings import DAYS_TO_PAST

from travel.rasp.morda_backend.morda_backend.station.data_layer.page_type import FindStationError


class StationPageContext(object):
    """
    Контекст страницы станции (время, событие, направление электричек и т.д.)
    """
    def __init__(self, station, date, event, now_dt):
        self.event = event
        self.station = station

        self.is_all_days = False  # Признак выдачи на все дни (all-days или schedule-plan)
        self.dt_now = station.local_time(now_dt)  # Текущее время на станции
        self.today = self.dt_now.date()  # Текущая дата на станции

        if date == 'today':
            self.special = 'today'
            self.date = self.today

        elif date == 'tomorrow':
            self.special = 'tomorrow'
            self.date = self.today + timedelta(days=1)

        elif date == 'all-days':
            self.special = 'all-days'
            self.is_all_days = True

        else:  # В date указана конкретная дата
            self.date = datetime.strptime(date, "%Y-%m-%d").date()
            if not (self.today - timedelta(DAYS_TO_PAST) <= self.date < self.today + timedelta(365 - DAYS_TO_PAST)):
                raise FindStationError('В БД нет информации об отправлениях на дату: {}'.format(self.date.isoformat()))

    @property
    def when(self):
        # Для сериализации when берутся только поля из WhenSchema
        return self

    def set_time(self, time_after, time_before):
        """
        Добавление в контекст страницы станции ограничений по времени
        """
        if time_after is not None:
            self.time_after = time_after
            self.time_before = time_before

        if not self.is_all_days:
            # Если не заданы ограничения по времени, то берем за весь день
            self.dt_after = self._make_dt(self.date, time_after if time_after else time(0))
            self.dt_before = self._make_dt(
                self.date if time_before else self.date + timedelta(1),
                time_before if time_before else time(0)
            )

    def _make_dt(self, date, time):
        return self.station.pytz.localize(datetime.combine(date, time))
