# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from dateutil import parser

from django.conf import settings
from urllib import quote

from common.data_api.baris.instance import baris
from common.data_api.baris.helpers import BarisData

from travel.rasp.morda_backend.morda_backend.station.data_layer.base_station import BaseStationForPage
from travel.rasp.morda_backend.morda_backend.station.data_layer.base_station_thread import BaseStationThread


log = logging.getLogger(__name__)


class BasePlaneStationThread(BaseStationThread):
    """
    Базовый класс для рейсов табло аэропорта
    """
    def __init__(self, flight, st_for_page):
        super(BasePlaneStationThread, self).__init__(st_for_page)
        self.is_valid = True
        self.flight = flight
        self.t_type = 'plane'
        self.terminal = flight['terminal']

        self._make_numbers()
        self._make_route_stations()

    def _make_numbers(self):
        """
        Получение номера и компании для основного рейса и кодшеров
        """
        self.number = self.flight['title']
        self.company_id = self.flight['airlineID']

        if 'codeshares' in self.flight and self.flight['codeshares']:
            self.codeshares = []
            for codeshare in self.flight['codeshares']:
                self.codeshares.append({
                    'number': codeshare['title'],
                    'company_id': codeshare['airlineID']
                })

    def _make_route_stations(self):
        """
        Формирование списка аропортов рейса
        """
        is_arrival = self.st_for_page.page_context.event == 'arrival'
        before_current_station = True
        self.route_stations = []

        for station_id in self.flight['route']:
            if station_id == self.st_for_page.station.id:
                before_current_station = False
            else:
                if (is_arrival and before_current_station) or (not is_arrival and not before_current_station):
                    station_info = self.get_station_info(station_id)
                    if station_info:
                        self.route_stations.append(station_info)
                    else:
                        self.is_valid = False

    def get_station_info(self, station_id):
        if station_id not in self.st_for_page.stations_by_ids:
            return None

        station = self.st_for_page.stations_by_ids[station_id]
        result = {
            'title': station.L_popular_title(),
            'iata_code': station.iata
        }
        if station.settlement:
            result['settlement'] = station.settlement.L_title()
        return result

    def get_base_avia_link(self):
        """
        Получение базовой ссылки на карточку рейса, для потомков наполняется дополнительными параметрами
        """
        host = settings.NATIONAL_AVIA_DOMAINS.get(self.st_for_page.country, settings.NATIONAL_AVIA_DOMAINS['ru'])
        number = quote(self.number.replace(' ', '-').encode('utf-8'))
        code = self.st_for_page.iata_code or self.st_for_page.sirena_code or self.st_for_page.icao_code

        return None if not code else 'https://{host}/flights/{number}/?lang={lang}'.format(
            host=host, number=number, lang=self.st_for_page.language
        )


class OneDayPlaneStationThread(BasePlaneStationThread):
    """
    Рейс для табло аэропорта на один день
    """
    def __init__(self, flight, st_for_page, dt_after=None, dt_before=None):
        super(OneDayPlaneStationThread, self).__init__(flight, st_for_page)

        if not flight['datetime']:
            self.is_valid = False
            return

        dt = parser.parse(flight['datetime'])
        if not st_for_page.trusted and ((dt_before and dt > dt_before) or (dt_after and dt < dt_after)):
            self.is_valid = False
            return

        self.make_event_date_and_time(dt)

        # TODO На все дни тоже нужно будет добавлять в ссылку станцию, но это когда БАРиС корректно будет отрабатывать
        # https://st.yandex-team.ru/TRAVELFRONT-3862
        direction = 'from' if self.st_for_page.page_context.event == 'departure' else 'to'
        code = self.st_for_page.iata_code or self.st_for_page.sirena_code or self.st_for_page.icao_code
        self.avia_link = '{base_link}&when={when}&{direction}={code}'.format(
            base_link=self.get_base_avia_link(), when=dt.date(), direction=direction, code=code
        )

        self._make_status()
        self.is_supplement = flight['source'] == 'flight-board'

    def _make_status(self):
        if not self.st_for_page.trusted or 'status' not in self.flight:
            return

        flight_status = self.flight['status']
        self.status = {
            'status': flight_status['{}Status'.format(self.event)],
            'actual_terminal': flight_status['{}Terminal'.format(self.event)],
            'gate': flight_status['{}Gate'.format(self.event)],
            'check_in_desks': flight_status['checkInDesks'],
            'baggage_carousels': flight_status['baggageCarousels']
        }
        dt = flight_status[self.event]
        if dt:
            actual_dt = self.st_for_page.station.pytz.localize(parser.parse(dt))
            self.status['actual_dt'] = actual_dt.isoformat()
        else:
            self.status['actual_dt'] = ''

        if flight_status['diverted'] and 'divertedAirportID' in flight_status:
            diverted_station_id = flight_status['divertedAirportID']
            if diverted_station_id:
                station_info = self.get_station_info(diverted_station_id)
                if station_info:
                    self.status['diverted'] = station_info


class AllDaysPlaneStationThread(BasePlaneStationThread):
    """
    Рейс для табло аэропорта на все дни
    """
    def __init__(self, flight, st_for_page):
        super(AllDaysPlaneStationThread, self).__init__(flight, st_for_page)

        time = parser.parse(flight['time']).time()
        self.make_event_date_and_time(time)
        self.avia_link = self.get_base_avia_link()
        self.days_text = flight['daysText']


class PlaneStationForPage(BaseStationForPage):
    """
    https://st.yandex-team.ru/RASPFRONT-7718
    Страница табло аэропорта
    """
    def __init__(self, station_page_type, country):
        super(PlaneStationForPage, self).__init__(station_page_type, country)

    def load_threads(self, page_context, time_after, time_before):
        """
        Формирование списка рейсов
        """
        page_context.set_time(time_after, time_before)
        self.page_context = page_context
        self._make_airport_codes()

        if page_context.is_all_days:
            self._make_all_days_threads()
        else:
            self._make_one_day_threads()

    def _make_airport_codes(self):
        self.iata_code = self.station.iata
        self.station_properties['iata_code'] = self.iata_code
        self.sirena_code = self.station.get_code('sirena')
        self.icao_code = self.station.icao

    def _make_one_day_threads(self):
        """
        Получение расписания на один день за временной интервал
        """
        dt_after = self.page_context.dt_after
        dt_before = self.page_context.dt_before
        baris_data = baris.get_station_tablo(
            self.station.id,
            direction=self.page_context.event,
            after=dt_after,
            before=dt_before
        )
        tablo = BarisData(baris_data)
        self.stations_by_ids = tablo.stations_by_ids
        self.companies_by_ids = tablo.companies_by_ids
        self.trusted = self.station.tablo_state == 'real'
        self.station_properties['trusted'] = self.trusted

        for flight in tablo.flights:
            thread = OneDayPlaneStationThread(flight, self, dt_after, dt_before)
            if thread.is_valid:
                self.threads.append(thread)

    def _make_all_days_threads(self):
        """
        Получение расписания на все дни
        """
        baris_data = baris.get_station_all_days_tablo(
            self.station.id, self.station.time_zone, self.language, direction=self.page_context.event
        )
        tablo = BarisData(baris_data)
        self.stations_by_ids = tablo.stations_by_ids
        self.companies_by_ids = tablo.companies_by_ids

        for flight in tablo.flights:
            thread = AllDaysPlaneStationThread(flight, self)
            if thread.is_valid:
                self.threads.append(thread)
