# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import datetime, time

from common.models.transport import TransportType
from stationschedule.views import get_schedule_class
from stationschedule.utils import TIMEDELTA_ONE_DAY, TIME_FOUR_HOURS

from travel.rasp.morda_backend.morda_backend.station.data_layer.base_station import BaseStationForPage
from travel.rasp.morda_backend.morda_backend.station.data_layer.suburban import SuburbanStationThread
from travel.rasp.morda_backend.morda_backend.station.data_layer.train import TrainStationThread
from travel.rasp.morda_backend.morda_backend.data_layer.segments_helpers import fetch_and_filter_train_purchase_numbers


class TabloStationForPage(BaseStationForPage):
    def __init__(self, station_page_type, country):
        super(TabloStationForPage, self).__init__(station_page_type, country)

    def _get_datetime_limits(self, schedule_date):
        # расписание на день выводим до 4:00 след дня, как и для TrainSchedule
        start_datetime = datetime.combine(schedule_date, time.min)
        end_datetime = datetime.combine(schedule_date + TIMEDELTA_ONE_DAY, TIME_FOUR_HOURS)

        return start_datetime, end_datetime

    def load_threads(self, page_context):
        self.page_context = page_context

        # Формирование класса формирования расписаний (из station_schedule)
        schedule_cls = get_schedule_class(self.station, schedule_type='tablo')
        schedule = schedule_cls(self.station, event=page_context.event, t_type_code=['train', 'suburban'])

        # Построение расписания
        if not page_context.is_all_days and hasattr(page_context, 'date'):
            start_datetime_limit, end_datetime_limit = self._get_datetime_limits(page_context.date)
            schedule.build(start_datetime_limit=start_datetime_limit, end_datetime_limit=end_datetime_limit)
        else:
            schedule.build()

        _, next_plan = schedule.current_next_plans(page_context.today)

        self.schedule_routes = schedule.schedule_routes
        self.schedule_routes = fetch_and_filter_train_purchase_numbers(self.schedule_routes)

        self.threads = []
        for schedule_route in self.schedule_routes:
            if schedule_route.event == page_context.event:
                t_type = schedule_route.t_type

                if t_type.id == TransportType.TRAIN_ID:
                    thread = TrainStationThread(schedule_route, self)
                elif t_type.id == TransportType.SUBURBAN_ID:
                    thread = SuburbanStationThread(schedule_route, self, next_plan)
                else:
                    raise ValueError('Transport type "{}" is unsupported for tablo station thread'.format(t_type))

                self.threads.append(thread)
