# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import datetime

from marshmallow import Schema, fields, post_load, validates_schema, ValidationError

from common.utils.namedtuple import namedtuple_with_defaults


class StationByIdQuerySchema(Schema):
    """
    Базовая схема для запросов к ручкам станции
    """
    station_id = fields.Int()

    @validates_schema(skip_on_field_errors=True)
    def validate_params(self, data):
        if 'station_id' not in data:
            raise ValidationError('station_id should be in the request params', ['station_id'])


StationQuickContext = namedtuple_with_defaults('StationQuickContext', [
    'station_id', 'subtype'
])


class StationQuickContextQuerySchema(StationByIdQuerySchema):
    """
    Схема запроса к быстрой ручке станции
    """
    subtype = fields.String()

    @post_load
    def make_context(self, data):
        return StationQuickContext(**data)


StationContext = namedtuple_with_defaults('StationContext', [
    'station_id', 'subtype', 'is_mobile', 'event', 'date',
    'direction', 'time_after', 'time_before', 'country'
], defaults={'is_mobile': False, 'event': 'departure', 'date': 'today', 'country': 'ru'})


class StationContextQuerySchema(StationByIdQuerySchema):
    """
    Схема запроса к ручке станции
    """
    subtype = fields.String()
    is_mobile = fields.Bool()
    event = fields.String()
    date = fields.String()

    direction = fields.String()

    time_after = fields.Time()
    time_before = fields.Time()
    country = fields.String()

    @validates_schema(skip_on_field_errors=True)
    def validate_params(self, data):
        if 'station_id' not in data:
            raise ValidationError('station_id should be in the request params', ['station_id'])

        if 'date' in data:
            if not data['date'] in {'today', 'tomorrow', 'all-days'}:
                try:
                    datetime.strptime(data['date'], "%Y-%m-%d")
                except:
                    raise ValidationError('wrong date format', ['date'])

        if 'time_after' in data and 'time_before' not in data:
            raise ValidationError('parameter time_after without parameter time_before', ['time_after'])
        if 'time_before' in data and 'time_after' not in data:
            raise ValidationError('parameter time_before without parameter time_after', ['time_before'])

    @post_load
    def make_context(self, data):
        return StationContext(**data)


class StationStopsContextQuerySchema(StationContextQuerySchema):
    return_for_types = fields.String()

    @post_load
    def make_context(self, data):
        return_for_types = {t for t in data.pop('return_for_types', '').split(',') if t}
        context = StationContext(**data)
        context.return_for_types = return_for_types

        return context


PopularDirectionsContext = namedtuple_with_defaults('StationPopularDirectionsContext', [
    'station_id', 'subtype', 'limit'
], defaults={'limit': 2})


class PopularDirectionsQuerySchema(StationByIdQuerySchema):
    """
    Схема запроса к ручке популярных направлений
    """
    subtype = fields.String()
    limit = fields.Int()

    @post_load
    def make_context(self, data):
        return PopularDirectionsContext(**data)
