# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response
import requests

from travel.rasp.library.python.common23.date import environment

from travel.rasp.morda_backend.morda_backend.station.data_layer.city_stations import get_station_city_stations
from travel.rasp.morda_backend.morda_backend.station.data_layer.page_type import FindStationError
from travel.rasp.morda_backend.morda_backend.station.data_layer.popular_directions import get_station_popular_directions
from travel.rasp.morda_backend.morda_backend.station.data_layer.station import (
    get_station_stops, get_station_quick, get_station)
from travel.rasp.morda_backend.morda_backend.station.request_serialization import (
    StationContextQuerySchema, StationQuickContextQuerySchema, StationStopsContextQuerySchema,
    StationByIdQuerySchema, PopularDirectionsQuerySchema
)
from travel.rasp.morda_backend.morda_backend.station.response_serialization import (
    AirportSchema, ScheduleSchema, ScheduleStopsSchema, StationSchema, StationPageTypeSchema, PopularDirectionsSchema
)


log = logging.getLogger(__name__)


@api_view(['GET'])
def station_quick(request):
    """
    Быстрая ручка станции, возвращает только информацию о типах транспорта станции
    /ru/station/quick/?station_id=2000005&subtype=suburban
    """
    context, errors = StationQuickContextQuerySchema().load(request.GET)
    if errors:
        return Response({'result': {}, 'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    try:
        station_data = get_station_quick(context)
    except FindStationError as ex:
        return Response({'result': {}, 'errors': [ex.message]}, status=status.HTTP_404_NOT_FOUND)

    data, errors = StationPageTypeSchema().dump(station_data)
    return Response({'result': data, 'errors': errors}, status=status.HTTP_200_OK)


@api_view(['GET'])
def station(request):
    """
    Основная информация о станции
    /ru/station/?station_id=2000005&date=today&subtype=suburban&event=departure
    """
    context, errors = StationContextQuerySchema().load(request.GET)
    if errors:
        return Response({'result': {}, 'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    try:
        station_data = get_station(context, environment.now_aware())
    except FindStationError as ex:
        return Response({'result': {}, 'errors': [ex.message]}, status=status.HTTP_404_NOT_FOUND)
    except requests.HTTPError as ex:
        code = ex.response.status_code
        return Response({'result': {}, 'errors': [ex.message]}, status=code)

    if station_data.station_page_type.page_type == 'plane':
        data, errors = AirportSchema().dump(station_data)
    elif station_data.station_page_type.page_type in {'bus', 'water'}:
        data, errors = ScheduleSchema().dump(station_data)
    else:
        data, errors = StationSchema().dump(station_data)
    return Response({'result': data, 'errors': errors}, status=status.HTTP_200_OK)


@api_view(['GET'])
def station_stops(request):
    """
    Список остановок до которых можно добраться с текущей станции
    /ru/station/stops/?station_id=2000005&date=today&return_for_types=bus,water
    """
    context, errors = StationStopsContextQuerySchema().load(request.GET)
    if errors:
        return Response({'result': {}, 'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    try:
        station_data = get_station_stops(context, environment.now_aware())
    except FindStationError as ex:
        return Response({'result': {}, 'errors': [ex.message]}, status=status.HTTP_404_NOT_FOUND)
    except requests.HTTPError as ex:
        code = ex.response.status_code
        return Response({'result': {}, 'errors': [ex.message]}, status=code)

    data, errors = ScheduleStopsSchema().dump(station_data)

    return Response({'result': data, 'errors': errors}, status=status.HTTP_200_OK)


@api_view(['GET'])
def station_city_stations(request):
    """
    Информация о других станциях города станции
    /ru/station/city_stations/?station_id=2000005
    """
    context, errors = StationByIdQuerySchema().load(request.GET)
    if errors:
        return Response({'result': {}, 'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    try:
        city_stations_data = get_station_city_stations(context)
    except FindStationError as ex:
        return Response({'result': {}, 'errors': [ex.message]}, status=status.HTTP_404_NOT_FOUND)

    return Response({'result': city_stations_data, 'errors': []}, status=status.HTTP_200_OK)


@api_view(['GET'])
def station_popular_directions(request):
    """
    Информация о популярных направлениях поиска для станции
    /ru/station/popular_directions/?station_id=2000005&subtype=suburban&limit=3
    """
    context, errors = PopularDirectionsQuerySchema().load(request.GET)
    if errors:
        return Response({'result': {}, 'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    try:
        popular_directions_data = get_station_popular_directions(context)
    except FindStationError as ex:
        return Response({'result': {}, 'errors': [ex.message]}, status=status.HTTP_404_NOT_FOUND)

    data, errors = PopularDirectionsSchema().dump(popular_directions_data)
    return Response({'result': data, 'errors': errors}, status=status.HTTP_200_OK)
