# coding: utf8

from copy import copy

from django.utils.functional import cached_property
from marshmallow import Schema, fields, post_load, pre_dump

from common.models.transport import TransportType
from common.serialization.common_schemas import MultiValueDictSchemaMixin
from common.serialization.schema import get_defaults_from_schema
from travel.rasp.library.python.common23.date import environment
from common.utils.namedtuple import namedtuple_with_defaults
from travel.rasp.morda_backend.morda_backend.serialization.schema_bases import PointsQuerySchema
from travel.rasp.morda_backend.morda_backend.serialization.segment_station import SegmentStationSchema
from travel.rasp.morda_backend.morda_backend.tariffs.bus.service import make_ybus_tariff_keys
from travel.rasp.morda_backend.morda_backend.tariffs.serialization import SegmentTariffsSchema


class BusSettlementKeysSchemaMixin(object):
    bus_settlement_keys = fields.Boolean(required=False, load_from='busSettlementKeys', missing=False)


class BusQuerySchema(PointsQuerySchema, MultiValueDictSchemaMixin, BusSettlementKeysSchemaMixin):
    dates = fields.List(fields.Date(), load_from='date')
    national_version = fields.String(required=True, load_from='nationalVersion')

    @post_load
    def make_query(self, data):
        if 'dates' not in data:
            now_msk = environment.now_aware()
            data['dates'] = [data['point_from'].localize(now_msk).date()]
        return BusQuery(**data)


class BusQuery(namedtuple_with_defaults('BusQuery',
                                        BusQuerySchema().fields.keys(),
                                        get_defaults_from_schema(BusQuerySchema))):
    pass


class SegmentSchema(Schema):
    id = fields.String(dump_to='yBusId')
    departure = fields.DateTime()
    arrival = fields.DateTime()
    duration = fields.TimeDelta()

    transport = fields.Method(
        allow_none=True,
        load_from='transport_model',
        serialize='serialize_transport'
    )
    company = fields.Function(
        serialize=lambda v: {'title': v.company_name, 'shortTile': v.company_name},
        load_from='company_name'
    )

    tariffs = fields.Nested(SegmentTariffsSchema)
    can_pay_offline = fields.Boolean(dump_to='canPayOffline')

    keys = fields.Method(serialize='make_ybus_tariff_keys')
    number = fields.String(allow_none=True)
    title = fields.String(allow_none=True)

    station_from = fields.Nested(SegmentStationSchema, dump_to='stationFrom')
    station_to = fields.Nested(SegmentStationSchema, dump_to='stationTo')

    def make_ybus_tariff_keys(self, segment):
        return make_ybus_tariff_keys(segment, bus_settlement_keys=self.context.get('bus_settlement_keys'))

    @pre_dump
    def add_tariffs(self, segment):
        segment.tariffs = {
            'classes': {
                'bus': {
                    'price': segment.price,
                    'seats': segment.seats,
                    'order_url': segment.order_url
                }
            }
        }
        return segment

    def serialize_transport(self, value):
        transport = copy(self.bus_ttype)
        transport['model'] = {
            'title': value.transport_model
        }
        return transport

    @cached_property
    def bus_ttype(self):
        t_type = TransportType.objects.get(id=TransportType.BUS_ID)
        return {
            'id': t_type.id,
            'code': t_type.code,
            'title': t_type.L_title(),
        }


class ResponseSchema(Schema):
    segments = fields.Nested(SegmentSchema, many=True)
    querying = fields.Boolean()
