# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from itertools import chain

from common.data_api.yandex_bus.api import get_yandex_buses_results
from common.models.geo import Settlement
from common.models.transport import TransportType
from common.utils.date import FuzzyDateTime

KEY_TEMPLATES = [
    'ybus {from_point_key} {departure:%Y-%m-%dT%H:%M} {to_point_key} {number}',
    'ybus {from_point_key} {departure:%Y-%m-%dT%H:%M} {to_point_key}'
]

KEY_TEMPLATES_ALL_DAYS = [
    'ybus {from_point_key} {departure:%H:%M} {to_point_key} {number}',
    'ybus {from_point_key} {departure:%H:%M} {to_point_key}'
]


def collect_yandex_buses_results(point_from, point_to, dates, bus_settlement_keys):
    result = []
    querying = False
    for date in dates:
        segments, q = get_yandex_buses_results(point_from, point_to, date, bus_settlement_keys)
        querying = querying or q
        if segments:
            result.append(segments)
    return list(chain.from_iterable(result)), querying


def ybus_tariff_keys_combination(departure, number, station_from, station_to, key_templates, bus_settlement_keys):
    return list(chain.from_iterable([
        _make_ybus_tariff_keys(departure, number, point_from, point_to, key_templates)
        for point_from, point_to in _gen_combinations(station_from, station_to, bus_settlement_keys)
        if departure and not isinstance(departure, FuzzyDateTime)
    ]))


def make_ybus_tariff_keys(segment, key_templates=None, bus_settlement_keys=False):
    if not key_templates:
        key_templates = KEY_TEMPLATES
    if segment.t_type.id != TransportType.BUS_ID:
        return []

    return ybus_tariff_keys_combination(segment.departure, segment.number, segment.station_from, segment.station_to,
                                        key_templates, bus_settlement_keys)


def make_ybus_thread_tariff_keys(thread, departure, station_from, station_to, key_templates=None, bus_settlement_keys=False):
    if not key_templates:
        key_templates = KEY_TEMPLATES
    if thread.t_type.id != TransportType.BUS_ID:
        return []

    return ybus_tariff_keys_combination(departure, thread.number, station_from, station_to, key_templates, bus_settlement_keys)


def _make_ybus_tariff_keys(departure, number, point_from, point_to, key_templates):
    return [
        template.format(
            from_point_key=point_from.point_key,
            to_point_key=point_to.point_key,
            departure=departure,
            number=number
        )
        for template in key_templates
    ]


def _gen_combinations(station_from, station_to, bus_settlement_keys):
    yield station_from, station_to

    if not bus_settlement_keys:
        return

    if hasattr(station_to, 'settlement') and station_to.settlement:
        yield station_from, station_to.settlement


def use_settlement_keys(context):
    return context.bus_settlement_keys or isinstance(context.point_to, Settlement)
