# coding: utf-8

import logging
from collections import defaultdict

from django.conf import settings
from django.utils.http import urlencode

from common.data_api.min_prices.api import min_price_storage
from common.data_api.ticket_daemon import query as ticket_daemon_query
from common.data_api.ticket_daemon.async_api import get_results_async
from common.data_api.ticket_daemon.jsend import Fail
from common.models.transport import TransportType


KEY_PREFIX = 'daemon'
log = logging.getLogger(__name__)


def create_daemon_queries(daemon_query):
    for transport_type in daemon_query.transport_types:
        for query_date in daemon_query.dates:
            yield ticket_daemon_query.Query(
                user_settlement=daemon_query.client_settlement,
                point_from=daemon_query.point_from,
                point_to=daemon_query.point_to,
                date_forward=query_date,
                date_backward=None,
                passengers={'adults': 1},
                klass='economy',
                national_version=daemon_query.national_version,
                t_code=transport_type.code,
                yandexuid=daemon_query.yandexuid
            )


def make_route_key(segment):
    result = segment.number

    if not result and segment.t_type.id == TransportType.BUS_ID:
        if segment.thread:
            result = segment.thread.hidden_number or ''

        # для сопоставления рейсов можно (но аккуратно, только для некоторых партнеров)
        # использовать время отправления в качестве ключа
        if not result and getattr(segment, 'supplier_code', None) == 'swdfactory':  # XXX
            result = segment.departure.strftime('time%H%M')

    return result.replace(' ', '-')


def make_daemon_key(segment):
    result = (KEY_PREFIX, make_route_key(segment), segment.departure.strftime("%m%d") if segment.departure else '')
    if segment.t_type.id == TransportType.PLANE_ID:
        try:
            result += (segment.data, )
        except AttributeError:
            pass
    return ' '.join(result)


def make_min_price_daemon_key(segment):
    """
    За неимением лучшего мы считаем что номера рейса достаточно
    для поездов и самолетов.
    """
    return segment.number


def get_tariffs_classes(variant, default_class):
    raw_seats = getattr(variant, 'raw_seats', None)
    raw_tariffs = getattr(variant, 'raw_tariffs', None)
    raw_is_several_prices = getattr(variant, 'raw_is_several_prices', {})

    return {
        default_class: {
            'price': variant.tariff,
            'seats': getattr(variant, 'seats', 1)
        }
    } if raw_seats is None or raw_tariffs is None else {
        tariff_class: {
            'price': tariff,
            'seats': raw_seats.get(tariff_class, 1),
            'several_prices': raw_is_several_prices.get(tariff_class, False)
        }
        for tariff_class, tariff in raw_tariffs.iteritems()
    }


def get_plane_tariffs_classes(variant):
    return {
        'economy': {
            'from_company': variant.from_company,
            'query_time': variant.query_time,
            'price': variant.tariff,
            'seats': 1,
            'order_url': variant.order_link,
            'deep_url': variant.deep_link
        }
    }


def add_order_urls(classes, point_from, point_to, segment):
    params = {
        'point_from': point_from.point_key,
        'point_to': point_to.point_key,
        'station_from': segment.station_from.id,
        'station_to': segment.station_to.id,
        'departure': segment.msk_departure.replace(tzinfo=None),
        'arrival': segment.msk_arrival.replace(tzinfo=None),
        'title': segment.title,
        'date': segment.departure.date(),
        'number': segment.number or segment.thread and segment.thread.hidden_number or '',
        't_type': segment.t_type.code,
    }

    if getattr(segment, 'electronic_ticket', False):
        params['et'] = 't'

    if getattr(segment, 'thread', None):
        params['thread'] = segment.thread.uid

    for tariff_class, tariff in classes.items():
        tariff['order_url'] = '/buy/?' + urlencode(dict(params, **{
            'cls': tariff_class,
            'tariff': tariff['price'].value
        }))


def iter_variants_segments(variants):
    for variant_pair in variants.items():
        partner = variant_pair[0]
        for variant in variant_pair[1]:
            segment = variant.forward.segments[0]
            if not (segment.departure and segment.arrival):
                continue

            yield partner, variant, segment


def parse_variants_segments(query, variants_segments):
    for partner, variant, segment in variants_segments:
        t_type = segment.t_type
        if t_type.id == TransportType.PLANE_ID:
            classes = get_plane_tariffs_classes(variant)
        else:
            classes = get_tariffs_classes(variant, t_type.code)
            point_from, point_to = query.point_from, query.point_to
            add_order_urls(classes, point_from, point_to, segment)

        segment.key = make_daemon_key(segment)
        segment.tariffs = {
            'partner': partner,
            'electronic_ticket': getattr(segment, 'electronic_ticket', False),
            'classes': classes
        }
        yield segment


def collect_daemon_result(daemon_queries, send_query=False, timeout=settings.TICKET_ASYNC_CLIENT_TIMEOUT):
    querying = False
    query_variants_segments = []
    for result in get_results_async(daemon_queries, send_query, timeout):
        querying = querying or result.querying
        if result.variants:
            variants_segments = iter_variants_segments(result.variants)
            query_variants_segments.append((result.query, variants_segments))

    return {'querying': querying}, query_variants_segments


def collect_daemon_segments(daemon_queries, send_query=False):
    result, query_variants_segments = collect_daemon_result(daemon_queries, send_query)

    segments_by_key = defaultdict(list)
    for query, variants_segments in query_variants_segments:
        for parsed_segment in parse_variants_segments(query, variants_segments):
            segments_by_key[parsed_segment.key].append(parsed_segment)

    result['segments'] = [
        reduce(merge_segments, segments) for segments in segments_by_key.values()
    ]
    return result


def collect_segments(query, send_query=False):
    daemon_queries = create_daemon_queries(query)
    return collect_daemon_segments(daemon_queries, send_query)


def get_better_class(current, other, t_type):
    if current is None:
        return other

    if other is None:
        return current

    if current['price'].value > other['price'].value:
        return other

    if current['price'].value < other['price'].value:
        return current

    # для не самолетных тарифов выбираем только по цене
    if t_type.id != TransportType.PLANE_ID:
        return current

    # от авиакомпаний тарифы лучше
    if current['from_company'] != other['from_company']:
        if current['from_company']:
            return current
        else:
            return other

    # кто быстрее ответил тот и победил
    if current['query_time'] <= other['query_time']:
        return current
    else:
        return other


def merge_segments(current, new):
    if current.tariffs == new.tariffs:
        return new

    current_classes = current.tariffs['classes']
    new_classes = new.tariffs['classes']
    for name, class_ in current_classes.items():
        new_classes[name] = get_better_class(class_, new_classes.get(name), current.t_type)
    return new


def get_dynamic_min_tariffs(query):
    t_codes = [t.code for t in query.transport_types] if query.transport_types else None
    min_prices = min_price_storage.get_min_prices(query.point_from, query.point_to, t_codes=t_codes)

    result = defaultdict(dict)
    for item in min_prices:
        result[item['route_uid']][item['class']] = item['price']

    return [
        {
            'key': key,
            'classes': {
                class_name: {
                    'price': {
                        'currency': 'RUB',
                        'value': price,
                    }
                } for class_name, price in classes.items()
            }
        } for key, classes in result.items()
    ]


def init_plane_tariff_queries(query):
    query = query._replace(transport_types=[TransportType.get_plane_type()])
    daemon_queries = create_daemon_queries(query)
    qids = []
    for q in daemon_queries:
        if q.is_valid():
            try:
                qid = q.query_all()
                qids.append(qid)
            except Fail as f:
                log.warning('Fail ignored from ticket daemon results: %s', f.message)
    return {'qids': qids}


def create_poll_tariff_query(query):
    return ticket_daemon_query.QueryQidPoll(**query)
